<?php

namespace App\Controller\Admin\Datatable;

use DateTime;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\DBAL\Types\Types;
use Doctrine\Persistence\ManagerRegistry;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Component\PropertyAccess\PropertyAccessor;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

class DatatableController extends AbstractController
{

    public function __construct(private readonly ManagerRegistry $managerRegistry)
    {
    }

    /**
     * Edit field.
     *
     *
     * @throws Exception
     *
     * @return Response
     */
    #[Route(path: '/datatables/edit/field', methods: ['POST'], name: 'app_datatables_edit')]
    public function editAction(Request $request)
    {

        if ($request->isXmlHttpRequest()) {
            // x-editable sends some default parameters
            $pk = $request->request->get('pk');       // entity primary key
            $field = $request->request->get('name');  // e.g. comments.createdBy.username
            $value = $request->request->get('value'); // the new value

            // additional params
            $entityClassName = $request->request->get('entityClassName'); // e.g. AppBundle\Entity\Post
            $token = $request->request->get('token');
            $originalTypeOfField = $request->request->get('originalTypeOfField');
            $path = $request->request->get('path'); // for toMany - the current element

            // check token
            if (! $this->isCsrfTokenValid('sg-datatables-editable', $token)) {
                throw new AccessDeniedException('DatatableController::editAction(): The CSRF token is invalid.');
            }

            //Ugly hack to use devices table instead of view which cannot be updated, use RealDevice entity
            if($entityClassName === 'App\Entity\VDevice'){
                $entityClassName = 'App\Entity\Device';
            }

            // get an object by its primary key
            $entity = $this->getEntityByPk($entityClassName, $pk);

            /** @var PropertyAccessor $accessor */
            /** @noinspection PhpUndefinedMethodInspection */
            $accessor = PropertyAccess::createPropertyAccessorBuilder()
                ->enableMagicCall()
                ->getPropertyAccessor()
            ;

            // normalize the new value
            $value = $this->normalizeValue($originalTypeOfField, $value);

            // set new value
            null !== $path ? $accessor->setValue($entity, $path, $value) : $accessor->setValue($entity, $field, $value);

            // save all
            try{

                $em = $this->managerRegistry->getManager();
                $em->persist($entity);
                $em->flush();

            }catch (UniqueConstraintViolationException){

                $entityName = 'Entity';
                
                if($entityClassName === \App\Entity\Label::class) {
                    $entityName = 'Label';
                }elseif($entityClassName === \App\Entity\SystemLabel::class){
                    $entityName = 'System Label';
                }elseif($entityClassName === \App\Entity\MonitoringGroup::class){
                    $entityName = 'Monitoring Group';
                }

                return new Response($entityName . ' already exists.', 400);

            }catch (\Exception){

                return new Response('Bad request', 400);

            }

            return new Response('Success', 200);
        }

        return new Response('Bad request', 400);
    }

    //-------------------------------------------------
    // Helper
    //-------------------------------------------------

    /**
     * Finds an object by its primary key / identifier.
     *
     * @param string $entityClassName
     */
    private function getEntityByPk($entityClassName, $pk): object
    {
        $em = $this->managerRegistry->getManager();

        $entity = $em->getRepository($entityClassName)->find($pk);
        if (! $entity) {
            throw $this->createNotFoundException('DatatableController::getEntityByPk(): The entity does not exist.');
        }

        return $entity;
    }

    /**
     * @throws Exception
     *
     * @return bool|DateTime|float|int|string|null
     */
    private function normalizeValue(string $originalTypeOfField, $value)
    {
        switch ($originalTypeOfField) {
            case Types::DATETIME_MUTABLE:
                $value = new DateTime($value);

                break;
            case Types::BOOLEAN:
                $value = $this->strToBool($value);

                break;
            case Types::TEXT:
            case Types::STRING:
                break;
            case Types::SMALLINT:
            case Types::INTEGER:
                $value = (int) $value;

                break;
            case Types::BIGINT:
                $value = (string) $value;

                break;
            case Types::FLOAT:
            case Types::DECIMAL:
                $value = (float) $value;

                break;
            default:
                throw new Exception("DatatableController::prepareValue(): The field type {$originalTypeOfField} is not editable.");
        }

        return $value;
    }

    /**
     * @throws Exception
     */
    private function strToBool(string $str): ?bool
    {
        $str = strtolower($str);

        if ('null' === $str) {
            return null;
        }

        if ('true' === $str || '1' === $str) {
            return true;
        }

        if ('false' === $str || '0' === $str) {
            return false;
        }

        throw new Exception('DatatableController::strToBool(): Cannot convert string to boolean, expected string "true" or "false".');
    }
}
