<?php

declare(strict_types=1);

namespace App\Service;

use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Process\Process;

class Command
{
    public const SERVICE_STATUS_ACTIVE = 'active';
    public const SERVICE_STATUS_INACTIVE = 'inactive';

    private const COMMAND_SERVICE_IS_ACTIVE = 'is-active';
    private const COMMAND_SERVICE_RESTART = 'restart';
    private const COMMAND_SERVICE_START = 'start';
    private const COMMAND_SERVICE_STOP = 'stop';

    private const COMMAND_SUDO = '/usr/bin/sudo';
    private const COMMAND_PHP = '/usr/bin/php';
    private const COMMAND_SYMFONY_CONSOLE = 'bin/console';

    private const COMMAND_NOTIFY_EMAIL = 'app:email-notification';
    private const COMMAND_NOTIFY_SLACK = 'app:slack-notification';

    private ParameterBagInterface $params;

    private string $dwarfgRootDir;

    private string $serviceName;

    private string $sysctlCommandPath;

    private string $uploadFwDirPath;

    private string $cmdUser;

    private string $licmanCmd;

    private string $customizedAgentCmd;

    private string $grabLogsCmd;

    private string $setServerIpCmd;

    private string $notifierConsoleCmd;

    private string $publicDir;

    /**
     * @param ConfigurationService $configurationService
     * @param ParameterBagInterface $params
     * @param string $publicDir
     * @throws \Exception
     */
    public function __construct(ConfigurationService $configurationService, ParameterBagInterface $params, string $publicDir)
    {

        $this->dwarfgRootDir = $configurationService->getProductRootDir() . DIRECTORY_SEPARATOR;
        $this->serviceName = $configurationService->getServiceName();
        $this->sysctlCommandPath = $configurationService->getSysctlCommandPath();

        $this->params = $params;
        $this->publicDir = $publicDir;

        $this->cmdUser = $this->params->get('app.cmdUser');
        $this->licmanCmd = $this->dwarfgRootDir . $this->params->get('app.command.licman');
        $this->customizedAgentCmd = $this->dwarfgRootDir . $this->params->get('app.command.customizedAgent');
        $this->grabLogsCmd = $this->dwarfgRootDir . $this->params->get('app.command.grabLogs');
        $this->setServerIpCmd = $this->dwarfgRootDir . $this->params->get('app.command.setServerIpAddress');
        $this->notifierConsoleCmd = $this->dwarfgRootDir . $this->params->get('app.notifierPath') . DIRECTORY_SEPARATOR . self::COMMAND_SYMFONY_CONSOLE;

        $this->uploadFwDirPath = $this->params->get('app.uploadLicensePath') . DIRECTORY_SEPARATOR;

    }

    /**
     * @return string
     */
    public function getServiceStatus(): string
    {

        $process = new Process($this->getSysctlCommandArray(
            self::COMMAND_SERVICE_IS_ACTIVE, $this->serviceName . '.service'
        ));
        $process->run();
        $status = self::SERVICE_STATUS_INACTIVE;

        if ($process->isSuccessful()) {
            $status = $process->getOutput();
        }

        return $status;

    }

    /**
     * @return bool
     */
    public function restartService(): bool
    {

        return $this->runSysctlProcess([self::COMMAND_SERVICE_RESTART, $this->serviceName . '.service']);

    }

    /**
     * @return bool
     */
    public function startService(): bool
    {

        return $this->runSysctlProcess([self::COMMAND_SERVICE_START, $this->serviceName . '.service']);

    }

    public function getSnmpGatewayServiceStatus(): string
    {

        $process = new Process($this->getSysctlCommandArray(
            self::COMMAND_SERVICE_IS_ACTIVE, $this->serviceName . '_snmp_gw' . '.service'
        ));
        $process->run();
        $status = self::SERVICE_STATUS_INACTIVE;

        if ($process->isSuccessful()) {
            $status = $process->getOutput();
        }

        return $status;

    }

    public function startSnmpGatewayService(): bool
    {

        return $this->runSysctlProcess([self::COMMAND_SERVICE_START, $this->serviceName . '_snmp_gw' . '.service']);

    }

    public function stopSnmpGatewayService(): bool
    {

        return $this->runSysctlProcess([self::COMMAND_SERVICE_STOP, $this->serviceName . '_snmp_gw' . '.service']);

    }

    public function restartSnmpGatewayService(): bool
    {

        return $this->runSysctlProcess([self::COMMAND_SERVICE_RESTART, $this->serviceName . '_snmp_gw' . '.service']);

    }

    /**
     * @param string $deviceId
     * @return int
     */
    public function runCustomizedAgentGenerator(string $deviceId): int
    {

        $process = new Process(array_merge([$this->customizedAgentCmd], [$deviceId]));

        return $process->run();

    }

    /**
     * @return int
     */
    public function runGrabLogs(): int
    {

        $process = new Process($this->getCommandArray($this->grabLogsCmd));

        return $process->run();

    }

    /**
     * @param string|null $fileName
     * @return Process
     */
    public function dumpLicense(string $fileName = null): Process
    {

        $parameters = [];
        $parameters[] = '-d';
        if ($fileName) {
            $parameters[] = $this->uploadFwDirPath . $fileName;
        }

        return new Process($this->getCommandArray($this->licmanCmd, $parameters));

    }

    /**
     * @param string|null $fileName
     * @return Process
     */
    public function checkLicense(string $fileName = null): Process
    {

        $parameters = [];
        $parameters[] = '-c';
        if ($fileName) {
            $parameters[] = $this->uploadFwDirPath . $fileName;
        }

        return new Process($this->getCommandArray($this->licmanCmd, $parameters));

    }

    /**
     * @param string $fileName
     * @param bool $checkLicense
     * @param bool $forceLicense
     * @return Process
     * @throws \Exception
     */
    public function updateLicense(string $fileName, bool $checkLicense = false, ?bool $forceLicense = false): Process
    {

        $file = $this->uploadFwDirPath . $fileName;

        if (!file_exists($file)) {
            throw new \Exception('License file not found.');
        }

        $parameters = [];
        $parameters[] = '-u';
        $parameters[] = $file;

        if ($checkLicense) {
            $parameters[] = '-c';
        }

        if ($forceLicense) {
            $parameters[] = '-f';
        }

        return new Process($this->getCommandArray($this->licmanCmd, $parameters));

    }

    /**
     * @param string $address
     * @return Process
     */
    public function setServerIp(string $address): Process
    {

        return new Process($this->getCommandArray($this->setServerIpCmd, [$address]));

    }

    /**
     * @param string $message
     * @return Process
     */
    public function sendSlackNotification(string $message): Process
    {

        return new Process($this->getConsoleCommandArray(self::COMMAND_NOTIFY_SLACK, [$message]));

    }

    /**
     * @param string $message
     * @return Process
     */
    public function sendEmailNotification(string $message): Process
    {

        return new Process($this->getConsoleCommandArray(self::COMMAND_NOTIFY_EMAIL, [$message]));

    }

    /**
     * @param string $command
     * @param array $parameters
     * @return array
     */
    private function getCommandArray(string $command, array $parameters = []): array
    {

        return array_merge([self::COMMAND_SUDO, '-u', $this->cmdUser, $command], $parameters);

    }

    /**
     * @param string $command
     * @param string $serviceName
     * @return string[]
     */
    private function getSysctlCommandArray(string $command, string $serviceName): array
    {

        return [self::COMMAND_SUDO, $this->sysctlCommandPath, $command, $serviceName];

    }

    /**
     * @param string $command
     * @param array $parameters
     * @return array
     */
    private function getConsoleCommandArray(string $command, array $parameters = []): array
    {

        return array_merge([self::COMMAND_SUDO, '-u', $this->cmdUser, self::COMMAND_PHP, $this->notifierConsoleCmd, $command], $parameters);

    }

    /**
     * @param array $command
     * @return bool
     */
    public function runSysctlProcess(array $command): bool
    {
        $process = new Process($this->getSysctlCommandArray($command[0], $command[1]));
        $process->run();

        if ($process->isSuccessful()) {
            return true;
        }

        return false;
    }
}
