<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Tool\AuthProfile;
use App\Form\AuthProfileType;
use App\Service\FfiService;
use Doctrine\DBAL\Exception\ForeignKeyConstraintViolationException;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Validator\ValidatorInterface;

#[Route(path: '/admin/ajax/auth-profile', name: 'ajax_auth_profile_')]
class AuthProfileController extends Controller
{

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param FfiService $ffiService
     * @param ValidatorInterface $validator
     * @return Response
     */
    #[Route(path: '/add', name: 'add', methods: ['POST'])]
    public function addAuthProfileAction(Request $request, ManagerRegistry $managerRegistry,
                                         FfiService $ffiService, ValidatorInterface $validator): Response {

        $authProfile = new AuthProfile();

        $form = $this->createForm(AuthProfileType::class, []);
        $form->handleRequest($request);
        $errors = [];

        if(!$form->isValid()){

            $formErrors = $form->getErrors(true,true);

            foreach ($formErrors as $error){

                $errors[] = $error->getMessage();
            }
        }

        if ($form->isSubmitted()) {

            $errorsValidator = [];
            foreach ($errorsValidator as $error) {
                $errors[] = $error->getMessage();
            }

            if (count($errors) === 0) {

                $formData = $form->getData();

                $em = $managerRegistry->getManager();
                try {

                    $authProfile->setProfileName($formData['profileName']);
                    $authProfile->setUsername($formData['username']);

                    if (!empty($formData['sshPwd'])) {

                        $encodedPassword = $ffiService->encode($formData['sshPwd']);

                        $authProfile->setSshPwd($encodedPassword[0]);
                        $authProfile->setPwdSalt($encodedPassword[1]);
                        $authProfile->setPwdPepper($encodedPassword[2]);
                        $authProfile->setPwdChilli($encodedPassword[3]);
                        $authProfile->setPwdVersion($encodedPassword[4]);

                    }

                    if (!empty($formData['sshKey'])) {

                        $encodedSshKey = $ffiService->encode($formData['sshKey']);
                        $authProfile->setSshKey($encodedSshKey[0]);
                        $authProfile->setKeySalt($encodedSshKey[1]);
                        $authProfile->setKeyPepper($encodedSshKey[2]);
                        $authProfile->setKeyChilli($encodedSshKey[3]);
                        $authProfile->setKeyVersion($encodedSshKey[4]);

                    }

                    $em->persist($authProfile);
                    $em->flush();

                }catch (UniqueConstraintViolationException $e){

                    //dump($e->getMessage());

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Authentication profile with this name already exists."]],
                        400);

                }catch (\Exception $e){

                    /* dump($e->getMessage());
                    die();*/

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Authentication profile."]],
                        400);

                }

                return new JsonResponse(['code' => 200, 'message' => 'success'],200);
            }

        }

        return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => $errors],
            400);
    }

    /**
     * Delete Auth Profile
     * @param Request $request
     * @param int $authProfileId
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/delete/{authProfileId}', name: 'delete', methods: ['GET'])]
    public function authProfileDeleteAction(Request $request, int $authProfileId, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $authProfileId = $managerRegistry->getRepository(AuthProfile::class)->findOneBy(['id' => $authProfileId]);

            if($authProfileId){

                try{

                    $em = $managerRegistry->getManager();
                    $em->remove($authProfileId);
                    $em->flush();

                }catch (ForeignKeyConstraintViolationException $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Auth profile cannot be deleted: ',
                        'success' => '', 'errors' => ['There are existing devices assigned.']],
                        200);

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Auth profile cannot be deleted: ',
                        'success' => '', 'errors' => ['Something bad happened.']],
                        200);

                }

            }

            return new JsonResponse(['code' => 200, 'message' => 'Success: ',
                'success' => ['Auth Profile was deleted.'], 'errors' => []],
                200);

    }

        return new Response('Bad request.', 400);

    }

}