<?php

namespace App\Controller\Admin;

use App\Entity\ProductVersion;
use App\Repository\ProductFlagsLicenseStatusRepository;
use App\Repository\ProductFlagsRepository;
use App\Repository\ProductVersionRepository;
use App\Service\ConfigurationService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class AboutController extends Controller
{
    /**
     * @param Request $request
     * @param ProductFlagsLicenseStatusRepository $productFlagsLicenseStatusRepository
     * @param ProductFlagsRepository $productFlagsRepository
     * @param ProductVersionRepository $productVersionRepository
     * @param ConfigurationService $configurationService
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '/admin/about', name: 'admin_about')]
    public function about(
        Request $request,
        ProductFlagsLicenseStatusRepository $productFlagsLicenseStatusRepository,
        ProductFlagsRepository $productFlagsRepository,
        ProductVersionRepository $productVersionRepository,
        ConfigurationService $configurationService
    ): Response {
        $aboutInfo = $productFlagsRepository->getAboutInfo()->getDescription();
        $changeLog = $productVersionRepository->getChangeLog();
        $productVersion = $productVersionRepository->getProductVersion();
        $supportLink = $configurationService->getSupportLink();
        $isSupportLinkAvailable = false;
        if (!empty($supportLink)) {
            $isSupportLinkAvailable = true;
        }

        $versionString = $this->getFormattedVersion($productVersion);

        return $this->render('about/about.html.twig', [

            'aboutInfo' => $this->formatTextToHtml($aboutInfo),
            'productVersion' => $versionString,
            'changeLog' => $this->formatChangeLog($changeLog),
            'isSupportLinkAvailable' => $isSupportLinkAvailable,

        ]);
    }

    /**
     * @param Request $request
     * @param UrlGeneratorInterface $router
     * @return Response
     */
    #[Route(path: '/admin/legal/eula', name: 'admin_legal_eula')]
    public function LegalEula(Request $request, UrlGeneratorInterface $router): Response
    {
        $referer = $request->headers->get('referer');
        $refererPathInfo = Request::create($referer)->getPathInfo();
        $routeInfos = $router->match($refererPathInfo);
        $refererRoute = $routeInfos['_route'] ?? '';

        // No route found (external URL for example)
        if (!$refererRoute) {
            $refererRoute = 'admin_dashboard';
        } else {
            unset($routeInfos['_route'], $routeInfos['_controller']);
        }

        /*try {
            $newUrl = $router->generate($refererRoute, $routeInfos); // new URL with foo bar
        } catch (\Exception $e) {
            $newUrl = 'Error when generating the new URL from the referer.';
        }*/

        return $this->render('legal/eula.html.twig', ['refererRoute' => $refererRoute, 'routeInfos' => $routeInfos]);
    }

    /**
     * @param string $text
     * @return string
     */
    private function formatTextToHtml(string $text): string
    {
        return str_replace(["\r", "\n"], '<br>', $text);
    }

    /**
     * @param array $changeLog
     * @return array
     */
    private function formatChangeLog(array $changeLog): array
    {
        $result = [];

        foreach ($changeLog as $item) {
            $row = [];
            $row['id'] = $item->getId();
            $row['version'] = $this->getFormattedVersion($item);
            $row['description'] = $item->description;
            $row['beta'] = $item->beta;
            $row['major'] = $item->major;
            $row['releaseDate'] = $item->releaseDate;

            $result[] = $row;
        }

        return $result;
    }

    /**
     * @param ProductVersion $productVersion
     * @return string
     */
    private function getFormattedVersion(ProductVersion $productVersion): string
    {
        return $productVersion->getVersionMajor() . '.' . $productVersion->getVersionMinor() . '.'
        . $productVersion->getVersionPatchlevel() . ' ' . $productVersion->getVersion()
        . ($productVersion->isBeta() ? ' Beta' : '');
    }
}
