#!/bin/bash
# deploy/destroy functions to be sourced by deployment, cleanup and upgrade
# Copyright (c) Dwarf Technologies, Jan Otte
# NOTE: while the file (itself) is copyrighted, you are invited to use the stuff you learn here anywhere you need

# turn on extended globbing (pattern matchin) in bash
shopt -s extglob

# drop all aliases
unalias -a
# unset all (potential) variables C_*
unset ${C_@}
# fix path to include admin stuff
export PATH=/sbin:/usr/sbin:$PATH

BASEDEFS=${BASEDEFS:-base_defs}

# deployment variables
TARGET_DEPLOY_NUM=${TARGET_DEPLOY_NUM:-0}
ORIG_DEPLOY_STR="deployed"
OFFSET_SHIFT=${OFFSET_SHIFT:-100}
DEPLOY_BLOCK=""
DEPLOY_SUCCESS=${DEPLOY_SUCCESS:-0}
UPGRADING=${UPGRADING:-0}
OLDSERVIP=""

# script parameter names
PARAM_FORCE="--yes"
PARAM_KEEPCACHE="--keepcache"
PARAM_HELP="--help"
PARAM_BR="--br"
PARAM_SERVID="--servid"
PARAM_BONLY="--bonly"
PARAM_RONLY="--ronly"
PARAM_DOMAIN="--domain"
PARAM_LICFILE="--license"
PARAM_OFFSET="--offset"
PARAM_SERVSSH="--servsshport"
PARAM_ADMPWD="--admpwd"
PARAM_ADMNAME="--admname"
PARAM_OPERPWD="--operpwd"
PARAM_OPERNAME="--opername"
PARAM_NOSSL="--nossl"
PARAM_DWARFGD_PORT="--port"
PARAM_UPGRADE="--upgrade"
PARAM_NOBACKUP="--nobackup"
PARAM_NOSTART="--nostart"

# define core commands (no point in doing anything if any of these is not available)
CMDS_CORE="C_MKDIR C_ID C_SYSTEMCTL C_DIRNAME C_BASENAME C_SED C_CP C_MV C_RM C_CHOWN C_CHMOD C_LN C_CHGRP C_TOUCH C_SLEEP C_USERMOD C_DELUSER C_GETENT C_ADDGROUP C_ADDUSER C_GREP C_CAT C_TAIL"
# try locating the core commands
C_MKDIR=$(command -v mkdir)
C_ID=$(command -v id)
C_SYSTEMCTL=$(command -v systemctl)
C_DIRNAME=$(command -v dirname)
C_BASENAME=$(command -v basename)
C_SED=$(command -v sed)
C_CP=$(command -v cp)
C_MV=$(command -v mv)
C_RM=$(command -v rm)
C_CHOWN=$(command -v chown)
C_CHMOD=$(command -v chmod)
C_LN=$(command -v ln)
C_LS=$(command -v ls)
C_CHGRP=$(command -v chgrp)
C_TOUCH=$(command -v touch)
C_SLEEP=$(command -v sleep)
C_USERMOD=$(command -v usermod)
C_DELUSER=$(command -v deluser)
C_GETENT=$(command -v getent)
C_ADDGROUP=$(command -v addgroup)
C_ADDUSER=$(command -v adduser)
C_GREP=$(command -v grep)
C_CAT=$(command -v cat)
C_TAIL=$(command -v tail)

for i in ${!C_@}; do # check all based on prefix, can expand the CMDS_CORE instead but opting to do full avail check here
	[ -z "${!i}" ] && {
		echo "Unable to locate required command (variable) $i"
		exit 1
	}
done

# define additional commands (empty now, it may require installing some packages to get these...
CMDS_DEPLOY="C_APACTL C_PYTHON C_SUDO C_SS C_MYSQL C_MYSQLDUMP C_A2ENMOD C_A2ENSITE C_A2DISSITE C_PHP C_RSYNC"
C_APACTL=""
C_PYTHON=""
C_SUDO=""
C_SS=""
C_MYSQL=""
C_MYSQLDUMP=""
C_A2ENMOD=""
C_A2ENSITE=""
C_A2DISSITE=""
C_PHP=""
C_RSYNC=""

# function to check (all additional) commands needed for messing with deploy once the time comes
validate_deploy_commands() {
	C_APACTL=$(command -v apachectl)
	C_A2ENMOD=$(command -v a2enmod)
	C_A2ENSITE=$(command -v a2ensite)
	C_A2DISSITE=$(command -v a2dissite)
	C_PYTHON=$(command -v python3)
	C_SUDO=$(command -v sudo)
	C_SS=$(command -v ss)
	C_MYSQL=$(command -v mysql)
	C_MYSQLDUMP=$(command -v mysqldump)
	C_PHP=$(command -v php)
	C_RSYNC=$(command -v rsync)
	for i in $CMDS_DEPLOY; do
		[ -z "${!i}" ] && {
			echo "Unable to locate deploy command (variable) $i"
			return 1
		}
	done
	return 0
}

prep_db() {
	USERS=$(mysql -Nse "SELECT COUNT(*) FROM mysql.user WHERE User='${SHORTNAME}' and Host='localhost'")
	if [ "$USERS" -gt 0 ] ; then
		echo "Users: $USERS, resetting $SHORTNAME user password to $DBPASS"
		echo -e "SET PASSWORD FOR ${SHORTNAME}@localhost = PASSWORD('$DBPASS');" | $C_MYSQL $MYSQOPT
		[ $? -eq 0 ] || fail_fat "Unable to set password for DB user ${SHORTNAME}@localhost"
	else
		echo "Setting $SHORTNAME user password to $DBPASS"
		echo -e "CREATE USER ${SHORTNAME}@localhost IDENTIFIED BY '$DBPASS';" | $C_MYSQL $MYSQOPT
		[ $? -eq 0 ] || fail_fat "Unable create DB user ${SHORTNAME}@localhost"
	fi
	echo -e "DROP DATABASE IF EXISTS \`$DWARFG_DBN\`;\n CREATE DATABASE IF NOT EXISTS \`$DWARFG_DBN\`;\n GRANT ALL ON ${DWARFG_DBN}.* to ${SHORTNAME}@localhost;" | $C_MYSQL $MYSQOPT
	[ $? -eq 0 ] || fail_fat "Error when preparing DB and basic permissions."
}

# read base_defs and app_funcs.sh
read_defs() {
	local MYDIR="$DIR"
	[[ $# -eq 1 && "$1" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Reading basedefs:\"$BASEDEFS\" from directory:$MYDIR ..."
	. "$MYDIR/$BASEDEFS" || { echo "Fatal: Definition file ($MYDIR/$BASEDEFS) not found." >&2; exit 1; }
	if [[ -n "$DWARFG_SITECONF" && -f "$DWARFG_SITECONF" ]] ; then
		echo "Found $NAME global site defs ($DWARFG_SITECONF), sourcing...";
		. "$DWARFG_SITECONF" || { echo "Fatal: Unable to source $NAME site config ($DWARFG_SITECONF)." >&2; exit 1; }
	fi
	echo "Reading basefuncs:\"$BASEFUNCS\" from directory:$MYDIR ..."
	. "$MYDIR/$BASEFUNCS" || { echo "Fatal: Basic functions ($MYDIR/$BASEFUNCS) not found." >&2; exit 1; }
}

# following functions serve mostly for updating variables in mid-deploy. It could be simplified considerably by reducing C+P
update_domain() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating domain in $MYDIR/$BASEDEFS..."
	$C_SED -i --follow-symlinks "s/^\(DOMAIN\)=.*/\1=\"$1\"/" "$MYDIR/$BASEDEFS"
}


update_stuff_here() {
	[ $# -eq 2 ] || {
		echo "String update function requires 2 parameters: string to replace and replacing string." >&2
		return 1
	}
	for f in *.conf *.txt *.service; do
		echo "Updating $f: $1 to $2..."
		$C_SED -i --follow-symlinks "s#\($1\)#$2#" "$f"
	done
}

update_install_paths() {
	[[ $# -eq 1 && -n "$1" && -d "$1" ]] || {
		echo "Install paths update: need path to target directory ($1 is not an accessible directory)." >&2
		return 1
	}
	echo "Updating INSTALL_PATH and INSTALL_DIR in $1..."
	opwd="$(pwd)"
	cd "$1" || {
		echo "Unable to change directory to $1" >&2
		return 1
	}
	update_stuff_here "INSTALL_PATH" "$BINDIR"
	update_stuff_here "INSTALL_DIR" "$DWARFG_NAM"
	cd "$opwd"
	return 0
}

update_sysctl() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating systemctl path in $MYDIR/$BASEDEFS... "
	$C_SED -i --follow-symlinks "s#^\(SYSCTLPATH\)=.*#\1=\"$1\"#" "$MYDIR/$BASEDEFS"
}

update_usessl() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating USE_SSL in $MYDIR/$BASEDEFS..."
	$C_SED -i --follow-symlinks "s/^\(USE_SSL\)=.*/\1=\"$1\"/" "$MYDIR/$BASEDEFS"
}

update_site_var() {
	[[ $# -ne 2 || -z "$1" || -z "$2" ]] && {
		echo "update_site_var(): need 2 parameters: which pass to update and its value. Got $#: 1:\"$1\" 2:\"$2\""
		return 1
	}
	UPDVAL="$2"
	case "$1" in
		DBPASS|GRDBPASS|GRADMPASS)
			if [ -f "$DWARFG_SITECONF" ] ; then
				if ($C_GREP "$1" "$DWARFG_SITECONF") ; then
					echo "Updating $1 in $DWARFG_SITECONF... to $UPDVAL"
					$C_SED -i --follow-symlinks "s/^\($1\)=.*/\1=\"$UPDVAL\"/" "$DWARFG_SITECONF"
				else
					echo "Adding $1 variable to existing $NAME site config..."
					echo "$1=\"$UPDVAL\"" >>"$DWARFG_SITECONF"
				fi
			else
				echo "Creting new $NAME site config..."
				echo "# Do not edit this file. It contains site-wide configuration for all your $NAME deployments and improper directive or syntax error here may result in partial or complete malfunction of one or all of your $NAME deployments" >"$DWARFG_SITECONF"
				echo "$1=\"$UPDVAL\"" >>"$DWARFG_SITECONF"
				chown $APPUSER "$DWARFG_SITECONF"
				chmod =600 "$DWARFG_SITECONF"
			fi
			;;
		*)
			echo "update_glob_pass(): unrecognized site-wide parameter"
			return 1
	esac
}

get_randpass() {
	psize=12
	[ -n "$1" ] && psize=$1
	tr -cd A-Za-z0-9 </dev/urandom | head -c $psize
}

new_dbpass() {
	DBPASS=$(get_randpass 12)
	update_site_var "DBPASS" "$DBPASS"
}

new_grdbpass() {
	GRDBPASS=$(get_randpass 12)
	update_site_var "GRDBPASS" "$GRDBPASS"
}

new_gradmpass() {
	GRADMPASS=$(get_randpass 12)
	update_site_var "GRADMPASS" "$GRADMPASS"
}

update_servid() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating servid in $MYDIR/$BASEDEFS to $1..."
	$C_SED -i --follow-symlinks "s/^SERVID=.*/SERVID=\"$1\"/" "$MYDIR/$BASEDEFS"
}

update_scdkey() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating scdkey in $MYDIR/$BASEDEFS to $1..."
	$C_SED -i --follow-symlinks "s/^SCDKEY=.*/SCDKEY=\"$1\"/" "$MYDIR/$BASEDEFS"
}

update_externurl() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating external URL in $MYDIR/$BASEDEFS to $1..."
	$C_SED -i --follow-symlinks "s/^EXTERNURL=.*/EXTERNURL=\"$1\"/" "$MYDIR/$BASEDEFS"
}

prep_servid() {
	local NEWSERVID
	if [ -z "$SERVID" ] ; then
		NEWSERVID="$(cat /proc/sys/kernel/random/uuid)"
	else
		NEWSERVID="$SERVID"
	fi
	echo "Updating servid in $BINDIR/$BASEDEFS (orig: \"$SERVID\"; new: \"$NEWSERVID\") ..."
	$C_SED -i --follow-symlinks "s/^SERVID=.*/SERVID=\"$NEWSERVID\"/" "$BINDIR/$BASEDEFS"
}

prep_scdkey() {
	local NEWSCDK
	if [ -z "$SCDKEY" ] ; then
		NEWSCDK="$(cat /proc/sys/kernel/random/uuid | tr -d '-')"
	else
		NEWSCDK="$SCDKEY"
	fi
	$C_SED -i --follow-symlinks "s/^SCDKEY=.*/SCDKEY=\"$NEWSCDK\"/" "$BINDIR/$BASEDEFS"
	echo $NEWSCDK
}

add_app_group() {
	"$C_GETENT" group "$APPGROUP" && return 0
	"$C_ADDGROUP" --system "$APPGROUP" || return 1
	return 0
}

add_app_user() {
	"$C_ID" -u "$APPUSER" >/dev/null 2>&1 && return 0
	"$C_ADDUSER" --system --ingroup "$APPGROUP" --disabled-password --no-create-home --shell=/bin/sh --home "$DWARFG_HOME" "$APPUSER" || return 1
	return 0
}

add_utg() {
	"$C_USERMOD" -a -G "$WMGROUP" "$APPUSER" || return 1
	return 0
}

add_app_home() {
	if [ ! -d "$DWARFG_HOME" ] ; then
		"$C_MKDIR" -p "$DWARFG_HOME" || return 1
	fi
	"$C_CHMOD" 755 "$DWARFG_HOME" || return 1
	"$C_CHOWN" -R "$APPUSER:$APPGROUP" "$DWARFG_HOME" || return 1
	"$C_MKDIR" -p -m 700 "$DWARFG_HOME/.ssh" || return 1
	return 0
}

add_app_service() {
	"$C_CP" "$BINDIR"/$SHORTNAME.service /etc/systemd/system/"${DWARFG_NAM}".service || return 1
#	"$C_SED" -i "s#/opt/${SHORTNAME}#$BINDIR#" /etc/systemd/system/"${DWARFG_NAM}".service || return 1
#	"$C_SED" -i "s#/srv/${SHORTNAME}#$SRVDIR#" /etc/systemd/system/"${DWARFG_NAM}".service || return 1
	return 0
}

add_app_sudoers() {
	echo -e "Cmnd_Alias NOTIFY_CMD_${DWARFG_DBN^^}=/usr/bin/php $BINDIR/notifier/bin/console app\:email-notification *,/usr/bin/php $BINDIR/notifier/bin/console app\:slack-notification *" >/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=NOPASSWD:${C_SYSTEMCTL} is-active ${DWARFG_NAM}.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=NOPASSWD:${C_SYSTEMCTL} restart ${DWARFG_NAM}.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=NOPASSWD:${C_SYSTEMCTL} start ${DWARFG_NAM}.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=($APPUSER) NOPASSWD:$BINDIR/prep_custom_agent.sh" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=($APPUSER) NOPASSWD:$BINDIR/exchange_server_ip.sh" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=($APPUSER) NOPASSWD:$BINDIR/licman" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$APPUSER\tALL=NOPASSWD:${C_SYSTEMCTL} restart apache2.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$APPUSER\tALL=NOPASSWD:${C_SYSTEMCTL} reload apache2.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$APPUSER\tALL=NOPASSWD:${C_SYSTEMCTL} restart ${DWARFG_NAM}.service" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$APPUSER\tALL=NOPASSWD:${C_SS}" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=($APPUSER) NOPASSWD:$BINDIR/grab_logs.sh" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	echo -e "$WWWUSER\tALL=($APPUSER) NOPASSWD:NOTIFY_CMD_${DWARFG_DBN^^}" >>/etc/sudoers.d/"${DWARFG_DBN}" || return 1
	return 0
}

add_apache_links() {
	"$C_LN" -sf "$DWARFG_DIR_WEB" "$DWARFG_APAWEBROOT" || return 1
	"$C_LN" -sf "$SRVDIR_FW" "$DWARFG_DIR_WEB/" || return 1
	"$C_LN" -sf "$SRVDIR_CONFIG" "$DWARFG_DIR_WEB/" || return 1
	"$C_CHGRP" "$APPGROUP" "$APACFGENDIR" "$APACFGAVDIR" || return 1
	"$C_CHMOD" g+w "$APACFGAVDIR" "$APACFGENDIR" || return 1
	return 0
}

apa_servsite() {
	$C_LN -sf "$BINDIR/$DWARFG_APACONF" "$APASITESDIR/$DWARFG_APACONF_AREF" || return 1
	$C_A2ENSITE -q "$DWARFG_APACONF_AREF" || return 1
}

create_app_fwdir() {
	if [ ! -d "$DWARFG_CACHE/$FW_FLAVORS" ] ; then # create system-wide cache directory if not existing
		"$C_MKDIR" -p "$DWARFG_CACHE/$FW_FLAVORS" || return 1
		"$C_CHOWN" -R "$APPUSER" "$DWARFG_CACHE" || return 1
	fi
	return 0
}

run_app_svc() {
	"$C_SYSTEMCTL" enable --now "${DWARFG_NAM}.service" || return 1
	"$C_SLEEP" 4 || return 1
	"$C_SYSTEMCTL" --output=short -n 0 --no-pager status "${DWARFG_NAM}.service" || return 1
	return 0
}

update_offset() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR=$BINDIR
	echo "Updating offset in $MYDIR/$BASEDEFS ..."
	$C_SED -i --follow-symlinks "s/^\(PORT_OFFSET\)=.*/\1=\"$1\"/" "$MYDIR/$BASEDEFS"
}

update_servssh() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR="$BINDIR"
	echo "Updating serv ssh port in $MYDIR/$BASEDEFS..."
	$C_SED -i --follow-symlinks "s/^\(SERV_TUNSSH_PORT\)=.*/\1=\"$1\"/" "$MYDIR/$BASEDEFS"
}

update_daemon_port() {
	local MYDIR="$DIR"
	[[ $# -eq 2 && "$2" == "$ORIG_DEPLOY_STR" ]] && MYDIR="$BINDIR"
	echo "Updating $NAME daemon local port in $MYDIR/$BASEDEFS to $1..."
	$C_SED -i --follow-symlinks "s/^\(DWARFG_PORT\)=.*/\1=\"$1\"/" "$MYDIR/$BASEDEFS"
}

# mid-deploy Dwarfg wipeout
wipeout() {
	local ORIGCLEANUP=0
	if [ 1 -eq "$INTERACTIVE" ] ; then
		echo "Running in interactive mode, wipeout check:"
		echo "Proceed with $NAME wipeout [Y/n]?"
		read -r RESP
		if [[ "$RESP" == "y" || "$RESP" == "Y" || -z "$RESP" ]] ; then
			echo "Wipeout confirmed."
		else
			fail_fat "Cannot continue as $NAME is already installed ($1)"
		fi
	fi
	if [ -f "$BINDIR/cleanup.sh" ] ; then
		[[ -e "$BINDIR/$BASEDEFS" && -n "$DOMAIN" ]] && update_domain "$DOMAIN" "$ORIG_DEPLOY_STR"
		[[ -e "$BINDIR/$BASEDEFS" && 0 -ne "$PORT_OFFSET" ]] && update_offset "$PORT_OFFSET" "$ORIG_DEPLOY_STR"
		"$BINDIR"/cleanup.sh --yes
		[ $? -eq 0 ] && ORIGCLEANUP=1
	fi
	if [ 0 -eq $ORIGCLEANUP ] ; then
		fail_nof "Original wipeout script failed, trying bundled one"
		"$DIR"/cleanup.sh --yes
		if [ $? -ne 0 ] ; then
			fail_fat "Bundled wipeout script failed, aborting."
		fi
	fi
	return 0
}

# mid-deploy light-backup (usable only when reinstalling SAME VERSION of Dwarfg)
cbackup() {
	echo "Reinstall (NOT UPGRADE): Performing deployment light-backup..."
	$C_MYSQLDUMP -t --insert-ignore --skip-opt "$DWARFG_DBN" >"$DWARFG_DUMP"
	OLDSERVIP=$($C_MYSQL -Ns $MYSQOPT "$DWARFG_DBN" << EOF
SELECT strval from product_conf where name='ExternalURL';
EOF
)
	if [ 0 -ne $? ] ; then
		fail_fat "Backup failed, try manually or reinstall from scratch (wipeout existing installation first ($BINDIR/cleanup.sh))"
	else
		echo "Backup to $DWARFG_DUMP finished."
	fi
	if [ -d "$BINDIR/ssl" ] ; then
		$C_CP -r "$BINDIR"/ssl ssl
		echo "Backed up ssl."
	fi
	$C_MKDIR bcert 2>/dev/null
	$C_CP "$BINDIR"/*.key "$BINDIR"/*.pem "$BINDIR"/*.srl "$BINDIR"/server.* bcert/ 2>/dev/null && echo "Backed up certs."
	$C_CP "$BINDIR/$DWARFG_APACONF" "$DWARFG_APACONF_BCK" && echo "Backed up $DWARFG_APACONF"
}

# mid-deploy restore (usable only when reinstalling the SAME VERSION of Dwarfg)
crestore_data() {
	echo "Reinstall (NOT UPGRADE): Performing deployment light-restore..."
	if [ ! -f "$DWARFG_DUMP" ] ; then
		fail_nof "Unable to locate $NAME data backup at ./$DWARFG_DUMP, skipping data restore."
	else
		$C_MYSQL $MYSQOPT "$DWARFG_DBN" <"$DWARFG_DUMP"
	fi
	if [ -n "$OLDSERVIP" ] ; then
		$C_MYSQL $MYSQOPT "$DWARFG_DBN"<<EOF
UPDATE product_conf SET strval="$OLDSERVIP" WHERE name='ExternalURL';
EOF
	fi
}

# mid-deploy pre-restore
crestore_preconfig() {
	echo "Performing pre-config restore..."
	[ -d bcert ] && echo "Restoring certificates..." && $C_CP bcert/* "$BINDIR"/ 2>/dev/null && $C_CP -r ssl "$BINDIR"/ 2>/dev/null
}

# mid-deploy post-restore
crestore_postconfig() {
	echo "Performing post-config restore..."
	[ -f "$DWARFG_APACONF_BCK" ] && echo "Overwriting $NAME apache config by backup copy" && $C_CP "$DWARFG_APACONF_BCK" "$BINDIR/$DWARFG_APACONF"
}

upgrade_deploy() {
	local CONFLICT DEPS
	[ -d "$DEPLOYMENTS" ] || return 1
	DEPS=$(ls -1 "$DEPLOYMENTS" | $C_GREP "$SHORTNAME")
	CONFLICT=$(echo -e "$DEPS" | $C_GREP "$DEPLOY_POSTFIX$")
	[ -z "$CONFLICT" ] && return 1
	# parse number from conflict
	TARGET_DEPLOY_NUM="${CONFLICT%%[^1-9]*}"
	[ -z "$TARGET_DEPLOY_NUM" ] && return 1
	return 0
}

list_deploys() {
	ls -1 "$DEPLOYMENTS" | $C_GREP "$SHORTNAME"
}

reserve_deploy() {
	local DEPS CONFLICT NUM
	NUM=0
	[ -d "$DEPLOYMENTS" ] || $C_MKDIR -p "$DEPLOYMENTS"
	DEPS=$(list_deploys)
	CONFLICT=$(echo -e "$DEPS" | $C_GREP "$DEPLOY_POSTFIX$")
	[ -n "$CONFLICT" ] && DEPLOY_BLOCK="$CONFLICT" && return 1
	echo -e "DEPS is $DEPS"
	for i in $(seq 9); do
		if (echo -e "$DEPS" | $C_GREP "${i}_") ; then
			echo "Deployment ${i}_ found in DEPS, continuing search..."
			:
		else
			echo "Deployment ${i}_ not found in DEPS, selecting..."
			NUM="$i"
			break
		fi
	done
	[ "$NUM" -eq 0 ] && {
		DEPLOY_BLOCK="Too many $NAME installations..."
		return 1
	}
	TARGET_DEPLOY_NUM=$NUM
	echo "Selected deployment number: $TARGET_DEPLOY_NUM"
	$C_TOUCH "$DEPLOYMENTS/${TARGET_DEPLOY_NUM}_$DEPLOY_POSTFIX" || {
		DEPLOY_BLOCK="Unable to reserve deployment..."
		return 1
	}
	return 0
}

remove_deploy() {
	if [[ 0 -eq "$DEPLOY_SUCCESS" && 0 -eq "$UPGRADING" ]] ; then
		[ "$TARGET_DEPLOY_NUM" -eq 0 ] && return 0 # nothing to remove as no deployment was chosen
		$C_RM "$DEPLOYMENTS/${TARGET_DEPLOY_NUM}_$DEPLOY_POSTFIX" 2>/dev/null
	fi
}

update_cmdline() {
	local MYDIR="$DIR"
	[[ -z "$DWARFG_CACHE" || -z "$MYDIR" || -z "$CMDLINE_DIR" ]] && return 1
	local cmd_script="$DWARFG_CACHE/$CMDLINE_DIR/dt_rmm_cmd"
	local new_cmd_script="$MYDIR/$CMDLINE_DIR/dt_rmm_cmd"
	[ -f "$new_cmd_script" ] || { echo "New cmdline script ($new_cmd_script) not found!" >&2; return 1; }
	ver_new=$(grep "^version=" "$new_cmd_script")
	ver_new=${ver_new##*=}
	ver_new=${ver_new#\"}
	ver_new=${ver_new%\"}
	ver_new_major=${ver_new%%.*}
	ver_new_minor=${ver_new##*.}
	if [ -f "$cmd_script" ] ; then
		ver_inst=$(grep "^version=" "$cmd_script")
		ver_inst=${ver_inst##*=}
		ver_inst=${ver_inst#\"}
		ver_inst=${ver_inst%\"}
		ver_inst_major=${ver_inst%%.*}
		ver_inst_minor=${ver_inst##*.}
		# compare versions - if new is newer, reinstall cmdline script
		if [ "$ver_new_major" -gt "$ver_inst_major" ] || [[ "$ver_new_major" -eq "$ver_inst_major" && "$ver_new_minor" -gt "$ver_inst_minor" ]] ; then
			echo "NOTE: updating $SHORTNAME cmdline at \"$DWARFG_CACHE/$CMDLINE_DIR\""
			rm -rf "$DWARFG_CACHE/$CMDLINE_DIR/"
			cp -r "$MYDIR/$CMDLINE_DIR" "$DWARFG_CACHE"/
		else
			echo "Skipping $SHORTNAME cmdline update because version is not higher (new: $ver_new_major.$ver_new_minor X old: $ver_inst_major.$ver_inst_minor)"
		fi
	else
		echo "NOTE: copying cmdline into \"$DWARFG_CACHE/$CMDLINE_DIR\""
		cp -r "$MYDIR/$CMDLINE_DIR" "$DWARFG_CACHE"/
	fi
}

integrate_cmdline() {
	local linked_cmds=0
	for i in "$DWARFG_CACHE/$CMDLINE_DIR/"* ; do
		target="$(basename "$i")"
		[ -f "/usr/bin/$target" ] || { ln -s -v "$i" "/usr/bin/$target"; linked_cmds=$((linked_cmds+1)); }
	done
	[ "$linked_cmds" -gt 0 ] && echo "$linked_cmds $SHORTNAME commands were linked to /usr/bin/"
}

enable_ffi() {
	ffi_ext="extension=ffi"
	ffi_enable="ffi.enable="
	enable_true="true"
	for i in $(find /etc/php/ -iname "php.ini" | grep apache2); do
		echo -n "Resolving FFI state in $i ... "
		if grep "^$ffi_ext" $i >/dev/null; then
			echo -n "extension already present, ... "
		else
			# if extension not enabled, check if commented out
			if grep "^;$ffi_ext" $i >/dev/null; then
				if sed -i "0,/^;$ffi_ext/s/^;$ffi_ext.*/$ffi_ext/" $i 2>/dev/null; then
					echo -n "uncommented extension, ... "
				else
					echo "extention uncomment failed"
					return 1
				fi
			else
				# add the extension
				if echo "$ffi_ext" >>$i 2>/dev/null; then
					echo -n "extension added, ..."
				else
					echo "extension add failed"
					return 1
				fi
			fi
		fi
		if grep "^$ffi_enable" $i >/dev/null; then
			if grep "^$ffi_enable" $i | grep "$enable_true" >/dev/null; then
				echo "FFI enabled already"
			else
				# ffi.enable is set to something else than true, switch to true
				if sed -i "0,/^$ffi_enable/s/^$ffi_enable.*/${ffi_enable}\"${enable_true}\"/" $i 2>/dev/null; then
					echo "FFI switched to true"
				else
					echo "FFI switch failed"
					return 1
				fi
			fi
		else
			# there is no ^ffi.enable=... line in cfg file
			# check if ffi section is there
			if grep "^\[ffi\]" $i >/dev/null; then
				if sed -i "/^\[ffi\]/a ${ffi_enable}\"${enable_true}\"" $i 2>/dev/null; then
					echo "enabled FFI"
				else
					echo "adding FFI enable failed"
					return 1
				fi
			else
				# add both section and enabling line to the file
				if echo -e "[ffi]\n${ffi_enable}\"${enable_true}\"" >>$i 2>/dev/null ; then
					echo "added FFI section and enabled ffi"
				else
					echo "Failed to add FFI section and enabling line"
					return 1
				fi
			fi
		fi
	done
}

set_phpini_var() {
	[ "$#" -eq 2 ] || return 1
	[[ -n "$1" && -n "$2" ]] || return 1
	for i in $(find /etc/php/ -iname "php.ini" | grep apache2); do
		if grep "^[ \t]*$1[ \t]*=" "$i" >/dev/null; then
			# present, update the value...
			if sed -i "0,/^[ \t]*;[ \t]*$1[ \t]*=/s/^[ \t]*;[ \t]*$1[ \t]*=.*/$1 = $2/" $i 2>/dev/null; then
				echo "Replaced PHP configuration value $1 to value $2 in $i ..."
			else
				echo "Failed to replace PHP configuration value $1 in $i ..."
				return 1
			fi
		else
			# check if commented out
			if grep "^[ \t]*;[ \t]*$1[ \t]*=" "$i" >/dev/null; then
				# uncomment and replace the value...
				if sed -i "0,/^[ \t]*;[ \t]*$1[ \t]*=/s/^[ \t]*;[ \t]*$1[ \t]*=.*/$1 = $2/" $i 2>/dev/null; then
					echo "Replaced (default) PHP configuration value $1 to value $2 in $i ..."
				else
					echo "Failed to replace (default) PHP configuration value $1 in $i ..."
					return 1
				fi
			else
				# add it to the bottom...
				if echo "$1 = $2" >>$i; then
					echo "Added PHP configuration value $1 set to value $2 at the end of $i ..."
				else
					echo "Failed to add PHP configuration value $1 at the end of $i ..."
					return 1
				fi
			fi
		fi
	done
	return 0
}
