<?php

namespace App\DataTable;

use App\DataTable\Column\CustomSecretDataColumn;
use App\Entity\Tool\AuthProfile;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Editable\TextEditable;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;
use App\Bundles\Sg\DatatablesBundle\Datatable\Filter\CustomSelectFilter;
use Sg\DatatablesBundle\Datatable\Column\Column;

/**
 * Class AuthManagementDatatable
 * @package App\Datatables
 */
class AuthManagementDatatable extends BaseDatatable
{

    protected string $entity = AuthProfile::class;
    protected string $name = 'auth_profile_datatable';

    /**
     * @param array $options
     * @throws \Exception
     */
    public function buildDatatable(array $options = array()): void
    {

        parent::buildDatatable($options);

        $this->extensions->set(array(
            'buttons' => array(
                'create_buttons' => array(  // custom buttons
                    array(
                        'action' => array(
                            'template' => '/datatable/tool/addAuthProfile.js.twig',
                        ),
                        'class_name' => 'btn-add',
                        'name' => 'addAuthProfile',
                        'text' => '<i class=\"fa fa-plus\"></i> Add Auth Profile',
                    ),
                    $this->getRefreshButton()
                ),
            )
        ));

        $this->addIdColumn();

        $this->columnBuilder
            ->add('profileName', Column::class,[
                'title' => 'Profile Name',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
                'editable' => array(TextEditable::class,
                    array(
                        'url' => 'app_datatables_edit',
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                if(true === $this->isAdmin()) {
                                    return true;
                                };
                            }

                            return false;
                        },
                        'mode' => 'inline',
                    ),
                ),
            ])
            ->add('username', Column::class, [
                'title' => 'Username',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class, [
                        'cancel_button' => true,
                    ],
                ],
            ])

            ->add('sshPwd', CustomSecretDataColumn::class, [
                'title' => 'SSH Password',
                'width' => '100%;',
                'filter' => [
                    CustomSelectFilter::class, [
                        'multiple' => false,
                        'cancel_button' => true,
                        'select_search_types' => [
                            '' => null,
                            'send_isNull' => 'isNull',
                            'send_isNotNull' => 'isNotNull'
                        ],
                        'select_options' => [
                            '' => 'Any',
                            'send_isNull' => 'Undefined',
                            'send_isNotNull' => 'Password defined'
                        ]
                    ],
                ],
            ])

            //CustomSecretDataColumn processed in the first column no need to do it again
            ->add('sshKey', Column::class, [
                'title' => 'SSH Key',
                'width' => '100%;',
                'filter' => [
                    CustomSelectFilter::class, [
                        'multiple' => false,
                        'cancel_button' => true,
                        'select_search_types' => [
                            '' => null,
                            'send_isNull' => 'isNull',
                            'send_isNotNull' => 'isNotNull'
                        ],
                        'select_options' => [
                            '' => 'Any',
                            'send_isNull' => 'Undefined',
                            'send_isNotNull' => 'Key defined'
                        ]
                    ],
                ],
            ])
            ->add('devices_count', Column::class, [
                'title' => 'Devices count',
                'width' => '100%;',
                //authprofile.id on the end of the subquery must be in small case I do not know why????
                'dql' => '(SELECT COUNT({p}) FROM App:Device\Device {p} WHERE {p}.authProfile = authprofile.id)',
                'searchable' => false,
                'orderable' => true,
            ])

            ->add(null, ActionColumn::class, [
                'title' => $this->translator->trans('sg.datatables.actions.title'),
                'width' => '100%;',
                'actions' => [
                    [
                        'route' => 'admin_tool_auth_profile_edit',
                        'route_parameters' => [
                            'authProfileId' => 'id',
                        ],
                        'label' => $this->translator->trans('sg.datatables.actions.edit'),
                        'icon' => self::ICON_EDIT,
                        'attributes' => [
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.edit'),
                            'class' => 'btn btn-primary btn-xs mb-1 mr-sm-2',
                            'role' => 'button',
                            /*'data-toggle' => 'modal',
                            'id' => 'edit',
                            'data-target' => "#add-sim-card",*/
                        ],
                        'start_html' => '<div class="input-group">',
                    ],
                    [
                        'route' => 'ajax_auth_profile_delete',
                        'route_parameters' => [
                            'authProfileId' => 'id',
                        ],
                        'label' => $this->translator->trans('sg.datatables.actions.delete'),
                        'icon' => self::ICON_DELETE,
                        'attributes' => [
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.delete'),
                            'class' => 'btn btn-danger btn-xs mb-1',
                            'role' => 'button',
                            'data-toggle' => 'modal',
                            'id' => 'delete',
                            'data-target' => "#confirm-delete",
                        ],
                        'render_if' => function ($row) {
                            return $row['devices_count'] === 0 && $this->authorizationChecker->isGranted('ROLE_ADMIN');
                            },
                        'end_html' => '</div>',
                    ],
                ],
                ]
            )
        ;
    }

}