<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Label;
use App\Form\LabelType;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/ajax/label', name: 'ajax_label_')]
class LabelController extends Controller
{

    /**
     * Get All Labels
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/all', name: 'get_all', methods: ['GET'])]
    public function getAllLabelsAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $labels = $managerRegistry->getRepository(Label::class)->findAll();
            $result = [];

            foreach ($labels as $label) {
                $result[$label->getId()] = ['value' => $label->getLabel()];
            }

            return new JsonResponse($result);

        }

        return new Response('Bad request.', 400);

    }

    /**
     * Create new label
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/create', name: 'create', methods: ['POST'])]
    public function createLabelAction(Request $request, ManagerRegistry $managerRegistry): Response {

        $label = new Label();
        $form = $this->createForm(LabelType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {

            $formData = $form->getData();
            $em = $managerRegistry->getManager();
            try {

                $label->setLabel($formData['label']);
                $em->persist($label);
                $em->flush();

            }catch (UniqueConstraintViolationException){

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Label already exists."]],
                    400);

            }catch (\Exception){

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the label."]],
                    400);

            }

            return new JsonResponse(['code' => 200, 'message' => 'Label successfully created.', 'success' => ''],
                200);
        }

        return new Response('Bad request.', 400);
    }

    /**
     * Delete Label
     * @param Request $request
     * @param int $labelId
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/delete/{labelId}', name: 'delete', methods: ['GET'])]
    public function labelDeleteAction(Request $request, int $labelId, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $label = $managerRegistry->getRepository(Label::class)->findOneBy(['id' => $labelId]);

            if($label){

                try {
                    $em = $managerRegistry->getManager();
                    $em->remove($label);
                    $em->flush();

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Label cannot be deleted: ',
                        'errors' => ['Something bad happened.']],
                        200);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Label "'. $label->getLabel() .'" was deleted.', 'success' => ''],
                    200);

            }

        }

        return new Response('Bad request.', 400);

    }

}