<?php

declare(strict_types=1);

namespace App\Controller\Admin\System;

use App\DataTable\DeviceDatatable;
use App\Form\Settings\DclSettingsType;
use App\Form\SettingsType;
use App\Repository\DataAtomRepository;
use App\Repository\DeviceAgentRepository;
use App\Repository\ProductFlagsRepository;
use App\Repository\TypesValueTypeRepository;
use App\Service\Api\Configuration;
use App\Service\Command;
use App\Service\ConfigurationService;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Routing\Annotation\Route;

class SettingsController extends Controller {

    /**
     * @param ParameterBagInterface $params
     * @param Command $commandService
     * @param LoggerInterface $logger
     * @param Configuration $apiConfiguration
     * @param TypesValueTypeRepository $typesValueTypeRepository
     */
    public function __construct(private readonly ParameterBagInterface $params, private readonly Command $commandService,
                                private readonly LoggerInterface $logger, private readonly Configuration $apiConfiguration,
                                private readonly TypesValueTypeRepository $typesValueTypeRepository)
    {
    }

	/**
	 * @param Request $request
	 * @param ConfigurationService $configurationService
	 * @param Command $commandService
	 * @return RedirectResponse|Response
	 */
    #[Route(path: '/admin/settings', name: 'admin_settings')]
    public function default(Request $request, ConfigurationService $configurationService, Command $commandService): RedirectResponse|Response
	{

        $settings = $this->getSettings($configurationService);

        $deviceTableColumnsChoices = [];
        $availableDeviceTableColumns = DeviceDatatable::availableColumns;
        foreach ($availableDeviceTableColumns as $key => $column) {
            $deviceTableColumnsChoices[$column] = $key;
        }

        // Sort array of choices by actual order stored in db
        $sortingArr = $settings['deviceTableColumns'];
        uasort($deviceTableColumnsChoices, fn($a, $b) => array_search($a, $sortingArr) <=> array_search($b, $sortingArr));

        $settings['deviceTableColumnsChoices'] = $deviceTableColumnsChoices;
        $configFormData = $settings['configFormData'];
        $settingsForm = $this->createForm(SettingsType::class, $settings);

        $settingsForm->handleRequest($request);

        if ($settingsForm->isSubmitted() && $settingsForm->isValid()) {
            $inputDataFormNames = array_map(static fn($ar) => ($ar['writable'] === true) ? $ar['name'] : null, $configFormData);
            $formData = $settingsForm->getData();

            foreach ($inputDataFormNames as $key => $inputName) {
                if ($inputName === null) {
                    continue;
                }

                $dataApi = $configFormData[$key];
                $isUserDefined = $formData[$inputName . '_enabled'];
                $result = $this->apiConfiguration->setConfiguration($inputName,  ($isUserDefined) ? $formData[$inputName] : null);
                if($result){
                    if ($result->operation_result->Code < 0) {

                        $this->addFlash(
                            'danger',
                            $result->operation_result->Description ,
                        );

                    }
                }else{

                    $this->addFlash(
                        'danger',
                        'Cannot update: ' . $dataApi['label'] ,
                    );

                }

            }

            $sendSlackTestNotification = false;
            $sendEmailTestNotification = false;

            if ($settingsForm->offsetExists('sendSlackTestNotification') && $sendSlackTestNotification = $settingsForm->get('sendSlackTestNotification')->isClicked()) {
                $serverSettings = $settingsForm->getData();
                $configurationService->updateSlackToken($serverSettings['slackToken']);
                $configurationService->updateSlackChannel($serverSettings['slackChannel']);

                $this->sendSlackTestNotification();
            }

            if ($settingsForm->offsetExists('sendEmailTestNotification') && $sendEmailTestNotification = $settingsForm->get('sendEmailTestNotification')->isClicked()) {
                $serverSettings = $settingsForm->getData();
                $configurationService->updateSenderEmail($serverSettings['senderEmail']);
                $configurationService->updateSmtpServer($serverSettings['smtpServer']);
                $configurationService->updateSmtpPort((int)$serverSettings['smtpPort']);
                $configurationService->updateSmtpSecurity($serverSettings['smtpSecurity']);
                $configurationService->updateSmtpLogin($serverSettings['smtpLogin']);
                $configurationService->updateSmtpPassword($serverSettings['smtpPassword']);

                $this->sendEmailTestNotification();
            }

            if (!$sendSlackTestNotification && !$sendEmailTestNotification) {

                return $this->saveSettings($settingsForm, $configurationService, $settings['serverIpAddress'], $commandService);
            }
        }

		$configNames = [];
		//Get group builder which is part of the form named 'config'
		foreach ($settingsForm->all()['config'] as $item){
			//Used to generate js to disable and enable items in the form
			$configNames[] = $item->getName();
		}

        return $this->render('system/settings.html.twig', [
            'settingsForm' => $settingsForm->createView(),
            'configNames' => $configNames ?? null,
            'nonWritableConfigData' => $this->getNonWritableConfigData($configFormData),
            'deviceTableColumnsChoices' => $deviceTableColumnsChoices,
        ]);
    }

	#[Route(path: '/admin/settings/dcl', name: 'admin_settings_dcl_atom')]
	public function dclAtomConfiguration(Request $request, DataAtomRepository $dataAtomRepository): Response
	{

		$atomTypes = $dataAtomRepository->findBy(['visible' => true]);
		$settingsForm = $this->createForm(DclSettingsType::class, ['configFormData' => $atomTypes]);

		$settingsForm->handleRequest($request);

		if ($settingsForm->isSubmitted() && $settingsForm->isValid()) {

			$formData =  $settingsForm->getData();
			//we do not want these data for iteration
			unset($formData['configFormData']);
			$updateData = [];
			foreach ($formData as $name => $value) {

				$items = explode('_', $name);
				$updateData[$items[0]][$items[1]] = $value;

			}

			foreach ($updateData as $id => $values) {
				$result = $dataAtomRepository->update($id, $values);

			}

			if($result){

				$this->addFlash(
					'success',
					'Data was updated.',
				);

			}else{

				$this->addFlash(
					'danger',
					'Cannot update data.' ,
				);

			}

			return $this->redirectToRoute('admin_settings_dcl_atom');

		}

		return $this->render('system/settings/dcl.html.twig', [
			'settingsForm' => $settingsForm->createView(),
			//Get group builder which is part of the form named 'config'
			'configItems' => $settingsForm->all()['config'] ?? null,
		]);

	}

	/**
	 * @param Request $request
	 * @param ConfigurationService $configurationService
	 * @param ProductFlagsRepository $productFlagsRepository
	 * @param DeviceAgentRepository $deviceAgentRepository
	 * @return Response
	 * @throws \Exception
	 */
    #[Route(path: '/admin/device-agents', name: 'admin_device_agents')]
    public function deviceAgents(
        Request                $request,
        ConfigurationService   $configurationService,
        ProductFlagsRepository $productFlagsRepository,
        DeviceAgentRepository  $deviceAgentRepository
    ): Response
    {
        $agents = $deviceAgentRepository->findAll();
        $serverIpAddress = $configurationService->getServerIpAddress();
        $processServerIpUpdate = $productFlagsRepository->getProcessServerIpUpdate();

        return $this->render('system/deviceAgents.html.twig', [
            'agents' => $agents,
            'serverIpAddress' => $serverIpAddress,
            'processServerIpUpdate' => false, // $processServerIpUpdate->version,
        ]);
    }

    /**
     * @return void
     */
    private function sendSlackTestNotification(): void
    {
        try {
            $process = $this->commandService->sendSlackNotification('Testing application notification');
            $process->mustRun();
            $output = $process->getOutput();

            $this->addFlash(
                'success',
                'Notification to Slack was sent.',
            );
        } catch (ProcessFailedException $exception) {
            $this->logger->error($exception->getMessage());
            $this->addFlash(
                'danger',
                'Cannot send notification to Slack. ' . $exception->getProcess()->getErrorOutput(),
            );
        }

        $this->redirectToRoute('admin_settings');
    }

    /**
     * @return void
     */
    private function sendEmailTestNotification(): void
    {
        try {
            $process = $this->commandService->sendEmailNotification('Testing application notification');
            $process->setTimeout(10);
            $process->mustRun();
            $output = $process->getOutput();

            $this->addFlash(
                'success',
                'Notification to email was sent.',
            );
        } catch (ProcessFailedException $exception) {
            $this->logger->error($exception->getMessage());
            $this->addFlash(
                'danger',
                'Cannot send notification to email. ' . $exception->getProcess()->getErrorOutput(),
            );
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
            $this->addFlash(
                'danger',
                'Cannot send notification to email. ' . $exception->getProcess()->getErrorOutput(),
            );
        }

        $this->redirectToRoute('admin_settings');
    }

    /**
     * @return array
     */
    private function getConfigFormData(): array {

        $config = $this->apiConfiguration->getConfiguration();
        $types = $this->typesValueTypeRepository->getAllTypes();

        // Cannot get data from API
        if (!$config) {
            return [];
        }

        $configFormData = [];

        foreach ($config->configuration as $configItem) {
            // dump($configItem);
            $configFormData[] = [
                'name' => $configItem->Option,
                'label' => $configItem->Caption,
                'description' => $configItem->Description,
                'type' => $types[$configItem->Type]['name'],
                'value' => $configItem->{'User Value'},
                'actualValue' => $configItem->{'Actual Value'},
                'unit' => $configItem->Unit,
                'writable' => (bool)$configItem->Writable,
                'default' => $configItem->Default,
                'min' => $configItem->Minimum,
                'max' => $configItem->Maximum,
            ];
        }

        return $configFormData;
    }

    private function getSettings(ConfigurationService $configurationService): array
    {

        $smtpServer = $configurationService->getSmtpServer();
        $smtpPort = $configurationService->getSmtpPort();
        $smtpSecurity = $configurationService->getSmtpSecurity();
        $smtpLogin = $configurationService->getSmtpLogin();

        $slackToken = $configurationService->getSlackToken();
        $slackChannel = $configurationService->getSlackChannel();

        $deviceTableColumns = $configurationService->getDeviceTableColumns();

        $isSlackSettingsComplete = $slackToken !== null && $slackChannel !== null;
        $isEmailSettingsComplete = $smtpServer !== null && $smtpPort !== null && $smtpSecurity !== 0 && $smtpLogin !== null;

        $configFormData = $this->getConfigFormData();

        return [
            'serverIpAddress' => $configurationService->getServerIpAddress(),
            'enableEmailNotifications' => $configurationService->isEnabledEmailNotifications(),
            'enableEmailNotificationToAddress' => $configurationService->isEnabledEmailNotificationToAddress(),
            'notificationEmail' => $configurationService->getNotificationEmail(),
            'senderEmail' => $configurationService->getSenderEmail(),
            'smtpServer' => $smtpServer,
            'smtpPort' => $smtpPort,
            'smtpSecurity' => $smtpSecurity,
            'smtpLogin' => $smtpLogin,
            'smtpPassword' => $configurationService->getSmtpPassword(),
            'slackToken' => $slackToken,
            'slackChannel' => $slackChannel,
            'isIpAddressEditEnabled' => $configurationService->isSaasMode(),
            'isSlackSettingsComplete' => $isSlackSettingsComplete,
            'isEmailSettingsComplete' => $isEmailSettingsComplete,
            'deviceTableColumns' => $deviceTableColumns,
            'snmpGatewayThreads' => $configurationService->getSnmpGatewayThreads(),
            'snmpGatewayBatch' => $configurationService->getSnmpGatewayBatch(),
            'snmpGatewayTimeout' => $configurationService->getSnmpGatewayTimeout(),
            'snmpGatewayRetries' => $configurationService->getSnmpGatewayRetries(),
            'snmpGatewayInterval' => $configurationService->getSnmpGatewayInterval(),
            'snmpLogLevel' => $configurationService->getSnmpLogLevel(),
            'snmpSslOverride' => $configurationService->getSnmpSslOverride(),
            'configFormData' => $configFormData,
        ];

    }

	/**
	 * @param array $configFormData
	 * @return array
	 */
    private function getNonWritableConfigData(array $configFormData): array {

        return array_filter($configFormData, fn($v) => $v['writable'] === false);

    }

	/**
	 * @param FormInterface $settingsForm
	 * @param ConfigurationService $configurationService
	 * @param $serverIpAddress
	 * @return RedirectResponse
	 */
    private function saveSettings(FormInterface $settingsForm, ConfigurationService $configurationService, $serverIpAddress): RedirectResponse
    {
        $serverSettings = $settingsForm->getData();

        $configurationService->updateIsEnabledEmailNotifications($serverSettings['enableEmailNotifications']);
        $configurationService->updateIsEnabledEmailNotificationToAddress($serverSettings['enableEmailNotificationToAddress']);
        $configurationService->updateNotificationEmail($serverSettings['notificationEmail']);
        $configurationService->updateSenderEmail($serverSettings['senderEmail']);
        $configurationService->updateSmtpServer($serverSettings['smtpServer']);
        $configurationService->updateSmtpPort((int)$serverSettings['smtpPort']);
        $configurationService->updateSmtpSecurity($serverSettings['smtpSecurity']);
        $configurationService->updateSmtpLogin($serverSettings['smtpLogin']);
        $configurationService->updateSmtpPassword($serverSettings['smtpPassword']);
        $configurationService->updateSlackToken($serverSettings['slackToken']);
        $configurationService->updateSlackChannel($serverSettings['slackChannel']);
        $configurationService->updateSnmpGatewayThreads($serverSettings['snmpGatewayThreads']);
        $configurationService->updateSnmpGatewayBatch($serverSettings['snmpGatewayBatch']);
        $configurationService->updateSnmpGatewayTimeout($serverSettings['snmpGatewayTimeout']);
        $configurationService->updateSnmpGatewayRetries($serverSettings['snmpGatewayRetries']);
        $configurationService->updateSnmpGatewayInterval($serverSettings['snmpGatewayInterval']);
        $configurationService->updateSnmpLogLevel($serverSettings['snmpLogLevel']);
        $configurationService->updateSnmpSslOverride($serverSettings['snmpSslOverride']);

        if ($serverSettings['serverIpAddress'] !== $serverIpAddress) {
            // Update Server IP Address
            try {
                $process = $this->commandService->setServerIp($serverSettings['serverIpAddress']);
                $process->mustRun();
                $exitCode = $process->getExitCode();

                if ($exitCode === 0) {
                    $this->addFlash(
                        'success',
                        'Settings where saved.',
                    );
                } else {
                    $this->addFlash(
                        'danger',
                        'Server IP address or FQDN cannot be set, see the event log detail.',
                    );
                }
            } catch (ProcessFailedException $exception) {
                $this->logger->error($exception->getMessage());
                $this->addFlash(
                    'danger',
                    'Server IP address or FQDN cannot be set, see the event log detail.',
                );
            }
        } else {
            $this->addFlash(
                'success',
                'Settings were saved.',
            );
        }

        return $this->redirectToRoute('admin_settings');
    }

}
