#ifndef DWARFLIB_LENC
#define DWARFLIB_LENC
// Dwarf Technologies - Dwarf Lenc library

/* The Light ENCoding library is an implementation of lightweight encoding/decoding functions.
 * The purpose is just to provide some encoding able to encode/decode VERY fast, and providing
 * an option to use it to not store sensitive data as plaintext.
 *
 * There is absolutely no claim about cryptography strength of ciphers as the implementation is
 * more of an encode/decode than encrypt/decrypt.
 *
 * There are basically 2 ways how to use the library based on use-case/context:
 *
 * [1] Occasional single calls to encode/decode
 *    - when called e.g. via FFI from PHP where there is no context remembered between the calls for encode/decode
 *    - in this case you must not call init function at all as all the encode/decode will be treated separately
 *    - in this mode, only one encode/decode can be run in parallell (locking implemented inside library)
 *    - encoding call example:
 *       result = dwarflenc_encode(DWARFLENC_MODE_RELAXED, bytearray_to_encode, bytearray_length, key_string, NULL, salt, pepper, chilli, DWARFLENC_VERSION_LAST);
 *          - the relaxed mode means that if the library does not have a key fixed, it will use the provided key
 *          - for at least some 'security', you can provide the salt, pepper and chilli parameters or let them be filled (pseudo-)randomly - use DWARFLENC_SALT_RANDOM for that. NOTE that you absolutely must provide the same salt, pepper and chilli (all positive integers only) for the decoding later on - if not, the decoding operation will output different data than the original data before encode.
 *          - use DWARFLENC_VERSION_LAST to use the latest known algorithm
 *          - the resulting data, and actually used values for salt, pepper, chilli and version are stored in the T_dwarflenc_packed structure (see return value below). You need to remember all these and provide them back when decoding.
 *          - to avoid dynamic allocation inside library you may pass a pointer to the allocated T_dwarflenc_packed structure with correct-lenght allocated data inside (struct.data is char * byte array holding the encoded text which is the same length as the data to be encoded)
 *          - the return value is pointer to the T_dwarflenc_packed structure which is either NULL in case of an error, a newly allocated memory block (when NULL is passed in as pointer) or pointer equal to the pre-allocated passed-in structure.
 *     - example decoding call:
 *        result = dwarflenc_decode(DWARFLENC_MODE_RELAXED, bytearray_to_decode, bytearray_length, key_string, NULL, salt, pepper, chilli, version);
 *          - the decoding function is very similar to the encoding call. It returns resulting structure pointer. Do NOT pass the input structure pointer inside. If you want to avoid dynamic allocation, pass empty allocated structure pointer with pre-allocated structure.data memory block of correct size
 *
 * [2] Repeated calls to encode/decode
 *    - when called repeatedly after first call, meaning the caller has context
 *    - when you USE THE SAME KEY for multiple encode/decode operations. If you use different key for each and every call, this way of using the library has no advantage over the first mode
 *    - in this case you first initialize the library, providing a key like this
 *       dwarflenc_init(T_dwarflenc_mode mode, const char * key);
 *    - afterwards you can use the same calls as above
 *    - depending on the mode requested, you may or may not use different then the fixed key in further encode/decode calls:
 *       DWARFLENC_MODE_STRICT - allows to encode/decode with only the key provided to dwarflenc_init()
 *       DWARFLENC_MODE_RELAXED - allows to encode/decode with either the fixed key (using DWARFLENC_MODE_STRICT or (DWARFLENC_MODE_RELAXED && providing NULL key)) or providing specific key for encode/decode (using DWARFLENC_MODE_OWN)
 *
 *  NOTES:
 *    - dwarflenc_setkey() can be used to change the fixed key unless DWARFLENC_MODE_STRICT has been used during initialization
 *    - dwarflenc_teardown() can be used to tear down the existing initialization. You can call dwarflenc_init() afterwards
 *    - if you access encode/decode functions via e.g. an FFI interface from another language, you can free the in-library allocated structures by calling dwarflenc_free_packed_data()
 */

typedef enum {
	DWARFLENC_VERSION_LAST = 0,
} T_dwarflenc_version;

#define MAX_DWARFLENC_KEYLEN 1024
	// Maximal key length. Key characters after this length are ignored.
#define MAX_ALLOC_ON_STACK 1024
	// If you DON'T provide pre-allocated buffer, internal calls use own buffers that are allocated on stack if they do not exceed this value
	// From performance point of view, pre-allocated buffers rock. If not viable option, run the encode/decode with blocks smaller or equal to MAX_ALLOC_ON_STACK. If you do no care about top performance, use as you want.
	// Bigger buffer gets allocated by malloca().

typedef enum {
	DWARFLENC_SALT_FAKE = 0, // no salt/pepper/chilli
	DWARFLENC_SALT_RANDOM = -1, // generate (pseudo-)random salt/pepper/chilli (returned in T_dwarflenc_packed)
} T_dwarflenc_salt;

typedef enum {
	DWARFLENC_MODE_RELAXED = 0, // setup: prep fixed key, do not lock it; usage: use fixed key if there or mandatory, provided key otherwise
	DWARFLENC_MODE_STRICT = 1, // setup: prep and lock fixed key - do not allow use of own key in consecutive encode/decode calls; usage: always use the fixed key - if not there, return error
	DWARFLENC_MODE_OWN = 2, // setup: error (no meaning); usage: always use the provided key, return error if fixed key is locked
} T_dwarflenc_mode;

typedef struct dwarflenc_packed {
	char * data;
	int len;
	int salt;
	int pepper;
	int chilli;
	int version;
} T_dwarflenc_packed;

T_dwarflenc_packed * dwarflenc_encode(T_dwarflenc_mode mode, const char * string_in, int len, const char * key, T_dwarflenc_packed * result_out, int salt, int pepper, int chilli, int version);

T_dwarflenc_packed * dwarflenc_decode(T_dwarflenc_mode mode, const char * string_in, int len, const char * key, T_dwarflenc_packed * result_out, int salt, int pepper, int chilli, int version);

int dwarflenc_init(T_dwarflenc_mode mode, const char * key);
int dwarflenc_teardown();
int dwarflenc_setkey(const char * new_key);
void dwarflenc_free_packed_data(T_dwarflenc_packed * data);
#endif
