<?php
declare(strict_types=1);

namespace App\Service;

use App\Datatables\DeviceDatatable;
use App\Entity\ProductConf;
use App\Entity\TypesValueType;
use App\ValueObject\Configuration;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;

class ConfigurationService {

    public const RESULT_STR_VAL = 'strval';
    public const RESULT_INT_VAL = 'intval';
    public const RESULT_BOOL_VAL = 'boolval';

    /**
     * @var EntityManagerInterface
     */
    private EntityManagerInterface $entityManager;

    /**
     * @var EntityRepository|\Doctrine\Persistence\ObjectRepository
     */
    private EntityRepository $configRepository;

    /**
     * @var EntityRepository|\Doctrine\Persistence\ObjectRepository
     */
    private EntityRepository $valueTypeRepository;

    /**
     * Configuration constructor.
     * @param EntityManagerInterface $entityManager
     */
    public function __construct(EntityManagerInterface $entityManager) {
        $this->entityManager = $entityManager;
        $this->configRepository = $this->entityManager->getRepository(ProductConf::class);
        $this->valueTypeRepository = $this->entityManager->getRepository(TypesValueType::class);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getProductRootDir(): string {
        return $this->getConfigItem(Configuration::CONFIG_PRODUCT_ROOT_DIR, self::RESULT_STR_VAL);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getServerIpAddress(): string {
        return $this->getConfigItem(Configuration::CONFIG_SERVER_IP_ADDRESS, self::RESULT_STR_VAL);
    }

    /**
     * @return int
     * @throws \Exception
     */
    public function getServerLocalPort(): int {
        return $this->getConfigItem(Configuration::CONFIG_SERVER_LOCAL_PORT, self::RESULT_INT_VAL);
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isFirstRun(): bool {
        return (bool)$this->getConfigItem(Configuration::CONFIG_IS_FIRST_RUN, self::RESULT_INT_VAL);
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isSaasMode(): bool {
        return (bool)$this->getConfigItem(Configuration::CONFIG_SAAS_MODE, self::RESULT_INT_VAL);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getSupportLink(): string {
        return $this->getConfigItem(Configuration::CONFIG_SUPPORT_LINK, self::RESULT_STR_VAL);
    }

    /**
     * @return bool
     */
    public function isEnabledEmailNotifications(): bool {
        try {
            return (bool)$this->getConfigItem(Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATIONS, self::RESULT_INT_VAL);
        } catch (\Exception $exception) {
            return false;
        }
    }

    public function isEnabledEmailNotificationToAddress(): bool {
        try {
            return (bool)$this->getConfigItem(Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATION_TO_ADDRESS, self::RESULT_INT_VAL);
        } catch (\Exception $exception) {
            return false;
        }
    }

    /**
     * @return string|null
     * @throws \Exception
     */
    public function getWebTunnelUrl(): ?string {
        return $this->getConfigItem(Configuration::CONFIG_WEB_TUNNEL_URL, self::RESULT_STR_VAL);
    }

    /**
     * @return string|null
     */
    public function getNotificationEmail(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_NOTIFICATION_EMAIL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSenderEmail(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SENDER_EMAIL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpServer(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_SERVER, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return int
     */
    public function getSmtpPort(): int {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_PORT, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return 25;
        }
    }

    /**
     * @return int
     */
    public function getSmtpSecurity(): int {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_SECURITY, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return 0; // None
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpLogin(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_LOGIN, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpPassword(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_PASSWORD, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSlackToken(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SLACK_TOKEN, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSlackChannel(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SLACK_CHANNEL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSysctlCommandPath(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SYSCTLPATH_COMMAND, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getServiceName(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SERVICE_CORE_NAME, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getAggregatedLogsFileName(): ?string {
        try {
            return $this->getConfigItem(Configuration::CONFIG_AGGREGATED_LOGS_FILE_NAME, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDeviceTableColumns(): array {
        try {
            $data = $this->getConfigItem(Configuration::CONFIG_DEVICE_TABLE_COLUMNS, self::RESULT_STR_VAL);
            if (!is_null($data)){
                return explode(',', $data);
            }else{
                return DeviceDatatable::showColumns;
            }
        } catch (\Exception $e) {
            return DeviceDatatable::showColumns;
        }
    }

    public function setDeviceTableColumns(array $columns){

        $data = implode(',', $columns);
        $this->setConfigItem($data, Configuration::CONFIG_DEVICE_TABLE_COLUMNS, self::RESULT_STR_VAL);

    }

    /**
     * @param bool $isFirstRun
     * @return void
     */
    public function updateIsFirstRun(bool $isFirstRun) {
        $this->setConfigItem($isFirstRun, Configuration::CONFIG_IS_FIRST_RUN, self::RESULT_BOOL_VAL);
    }

    /**
     * @param bool $isEnabled
     * @return void
     */
    public function updateIsEnabledEmailNotifications(bool $isEnabled) {
        $this->setConfigItem($isEnabled, Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATIONS, self::RESULT_BOOL_VAL);
    }

    /**
     * @param bool $isEnabled
     * @return void
     */
    public function updateIsEnabledEmailNotificationToAddress(bool $isEnabled) {
        $this->setConfigItem($isEnabled, Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATION_TO_ADDRESS, self::RESULT_BOOL_VAL);
    }

    /**
     * @param string|null $email
     * @return void
     */
    public function updateNotificationEmail(?string $email) {
        $this->setConfigItem($email, Configuration::CONFIG_NOTIFICATION_EMAIL, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $email
     * @return void
     */
    public function updateSenderEmail(?string $email) {
        $this->setConfigItem($email, Configuration::CONFIG_SENDER_EMAIL, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $server
     * @return void
     */
    public function updateSmtpServer(?string $server) {
        $this->setConfigItem($server, Configuration::CONFIG_SMTP_SERVER, self::RESULT_STR_VAL);
    }

    /**
     * @param int $port
     * @return void
     */
    public function updateSmtpPort(int $port) {
        $this->setConfigItem($port, Configuration::CONFIG_SMTP_PORT, self::RESULT_INT_VAL);
    }

    /**
     * @param int $type
     * @return void
     */
    public function updateSmtpSecurity(int $type) {
        $this->setConfigItem($type, Configuration::CONFIG_SMTP_SECURITY, self::RESULT_INT_VAL);
    }

    /**
     * @param string|null $login
     * @return void
     */
    public function updateSmtpLogin(?string $login) {
        $this->setConfigItem($login, Configuration::CONFIG_SMTP_LOGIN, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $password
     * @return void
     */
    public function updateSmtpPassword(?string $password) {
        if ($password !== null) {
            $this->setConfigItem($password, Configuration::CONFIG_SMTP_PASSWORD, self::RESULT_STR_VAL);
        }
    }

    /**
     * @param string|null $slackToken
     * @return void
     */
    public function updateSlackToken(?string $slackToken) {
        $this->setConfigItem($slackToken, Configuration::CONFIG_SLACK_TOKEN, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $slackChannel
     * @return void
     */
    public function updateSlackChannel(?string $slackChannel) {
        $this->setConfigItem($slackChannel, Configuration::CONFIG_SLACK_CHANNEL, self::RESULT_STR_VAL);
    }

    /**
     * @param string $itemName
     * @param string $resultType
     * @return mixed
     * @throws \Exception
     */
    private function getConfigItem(string $itemName, string $resultType) {
        $result = $this->configRepository->findOneBy(['name' => $itemName]);

        if (!$result) {
            throw new \Exception('No ' . $itemName . ' configuration found.');
        }

        return $result->{'get' . $resultType}();
    }

    /**
     * @param $value
     * @param $itemName
     * @param $resultType
     */
    private function setConfigItem($value, $itemName, $resultType) {
        $config = $this->configRepository->findOneBy(['name' => $itemName]);

        if (!$config) {
            $config = new ProductConf();
            $config->setName($itemName);
            $config->setValType($this->valueTypeRepository->findOneBy(['id' => 3]));
        }

        $config->{'set' . $resultType}($value);

        $this->entityManager->persist($config);
        $this->entityManager->flush();
    }

}
