<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Device\Device;
use App\Entity\Tunnel\TunnelOperation;
use App\Entity\Tunnel\Tunnels;
use App\Repository\TunnelWebRepository;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/ajax/tunnel', name: 'ajax_tunnel_')]
class TunnelController extends Controller
{

    /**
     * Assign Device to Web Tunnel
     * @return Response
     */
    #[Route(path: '/web/assign-device', name: 'assign_device', methods: ['POST'])]
    public function assignDeviceToTunnel(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {
            return new Response('Bad Request', 400);
        }

        if(isset($request->request)) {

            // Request has request data ?
            if (!isset($request->request))
                return new JsonResponse(['status' => 'Error'],400);

            $item = (int) $request->get('item');
            $tunnelId = (int) $request->get('tunnelId');

            if (!isset($item) && !isset($tunnelId))
                return new JsonResponse(['status' => 'Error'],400);

            $em = $managerRegistry->getManager();

            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $item]);
            $tunnel = $managerRegistry->getRepository(Tunnels::class)->findOneBy(['id' => $tunnelId]);
            $requestedOperation = $managerRegistry
                ->getRepository(TunnelOperation::class)->findOneBy(['id' => 1]); //Setup

            if($device && $tunnel){

                $tunnel->setDevice($device);
                $tunnel->setRequestedOperation($requestedOperation);
                $em->persist($tunnel);
                $em->flush();

                return new JsonResponse(['code' => 200, 'message' => 'Devices sucessfully inserted to the group.', 'errors' => ['errors' => ['']]],
                    200);

            }

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Error'],
            400);

    }

    /**
     * Web Tunnel restart
     * @return JsonResponse|Response
     */
    #[Route(path: '/web/{tunnelId}/restart', name: 'restart', methods: ['GET'])]
    public function tunnelRestart(Request $request, int $tunnelId, TunnelWebRepository $tunnelWebRepository): Response {

        if (!$request->isXmlHttpRequest()) {
            return new Response('Bad Request', 400);
        }

        if(isset($request->request)) {

            if (!isset($tunnelId))
                return new JsonResponse(['status' => 'Error'],400);

            $result = $tunnelWebRepository->processRequestOperation($tunnelId, TunnelOperation::OPERATION_RESTART);

            if($result){

                return new JsonResponse(['code' => 200, 'message' => 'Web tunnel was restarted.', 'errors' => ['errors' => ['']]],
                    200);

            }

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Error'],
            400);

    }

    /**
     * Web Tunnel shutdown
     * @return JsonResponse|Response
     */
    #[Route(path: '/web/{tunnelId}/shutdown', name: 'shutdown', methods: ['GET'])]
    public function tunnelShutdown(Request $request, int $tunnelId, TunnelWebRepository $tunnelWebRepository): Response {

        if (!$request->isXmlHttpRequest()) {
            return new Response('Bad Request', 400);
        }

        if(isset($request->request)) {

            if (!isset($tunnelId))
                return new JsonResponse(['status' => 'Error'],400);

            $result = $tunnelWebRepository->processRequestOperation($tunnelId, TunnelOperation::OPERATION_SHUTDOWN);

            if($result){

                return new JsonResponse(['code' => 200, 'message' => 'Web tunnel was shut down.', 'errors' => ['errors' => ['']]],
                    200);

            }

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Error'],
            400);

    }

}