<?php

namespace App\Repository;

use App\Entity\Device\Device;
use App\Entity\Device\DeviceStats;
use App\Entity\Device\DeviceSyncStatus;
use App\Entity\MonitoringGroup\MonitoringGroupMapping;
use App\Entity\Tunnel\Tunnels;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\ParameterType;
use Doctrine\ORM\Query;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @method Device|null find($id, $lockMode = null, $lockVersion = null)
 * @method Device|null findOneBy(array $criteria, array $orderBy = null)
 * @method Device[]    findAll()
 * @method Device[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class DeviceRepository extends ServiceEntityRepository
{

    /**
     * @param ManagerRegistry $registry
     */
    public function __construct(ManagerRegistry $registry){

        parent::__construct($registry, Device::class);

    }

    /**
     * @param Device $device
     * @return int|mixed|string
     */
    public function getDevicesExceptItself(Device $device): mixed
    {

        return $this->createQueryBuilder('p')
            ->where('p.id != :val')
            ->setParameter('val', $device)
            ->getQuery()
            ->getResult()
            ;

    }

    /**
     * @return array
     * @throws Exception
     */
    public function getDevices(): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid, name FROM devices
            WHERE deleted = 0
            ORDER BY devid
            ';

        return $conn->executeQuery($sql)->fetchAllAssociative();

    }

    /**
     * @param array $deviceIds
     * @param int $devTypeId
     * @return array
     * @throws Exception
     */
    public function getDevicesExcept(array $deviceIds, int $devTypeId): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid, name FROM devices
            WHERE id_device not IN(:deviceIds) AND dev_type = :devTypeId AND deleted = 0
            ORDER BY devid
            ';

        return $conn->executeQuery($sql, array('deviceIds' => $deviceIds, 'devTypeId' => $devTypeId), ['deviceIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY])->fetchAllAssociative();

    }

    /**
     * @param int $id
     * @param string $name
     * @return Device|null
     */
    public function updateName(int $id, string $name): ?Device
    {

        $entityManager = $this->getEntityManager();

        $device = $this->findOneBy(array('id'=> $id));
        $device->name = $name;

        $entityManager->persist($device);
        $entityManager->flush();

        return $device;

    }

    /**
     * @param array $order
     * @param string|null $filter
     * @return Query
     */
    public function findByOrder(array $order, string $filter = null): Query
    {

        $query =  $this->createQueryBuilder('p')
            // ->from('device', 'd')
            ->join('p.devType', 'devt')
            ->join('p.osVer', 'devosv')
            ->join('devosv.osTypes', 'ost')
            //->orderBy('p.id', 'ASC')
            ->orderBy($order[0], $order[1]);

        if(isset($filter)){

            $query->where('p.name LIKE :filter')
            ->setParameter('filter', '%'. $filter . '%');

        }

      return $query->getQuery();

    }

    /**
     * @param int $mntGroupId
     * @return int|mixed|string
     */
    public function findAllNotInMonitoringGroup(int $mntGroupId): mixed
    {

        // get an ExpressionBuilder instance, so that you
        $expr = $this->_em->getExpressionBuilder();

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->where(
                $expr->notIn(
                    'd.id',
                    $this->_em->createQueryBuilder()
                        ->select('IDENTITY(mgm.device)')
                        ->from(MonitoringGroupMapping::class, 'mgm')
                        ->where('mgm.monitoringGroup = :mntGroupId')
                        ->getDQL()
                )
            )
            ->andWhere('d.deleted = :deleted')

            ->setParameter('deleted', '0')
            ->setParameter('mntGroupId', $mntGroupId)
        ;

        /*
        // create a subquery in order to take all address records for a specified user id
        $sub = $this->_em->createQueryBuilder()
            ->select('mgm.device')
            ->from(MonitoringGroupMapping::class, 'mgm')
            ->where('mgm.monitoringGroup = :mntGroupId');

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->where($expr->not($expr->exists($sub->getDQL())))
            ->setParameter('mntGroupId', $mntGroupId);
        */
        return $qb->getQuery()->getResult();

    }

    /**
     * @param int $mntGroupId
     * @return array
     * @throws Exception
     */
    public function findAllNotInMonitoringGroupPlainSql(int $mntGroupId): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = "SELECT id_device, devid, name
                    FROM ". Device::ENTITY_TABLE_NAME ."
                    WHERE id_device NOT IN (
                    SELECT id_device FROM monitoring_group_device_map WHERE mongrp_id = :mntGroupId
                    ) AND deleted = 0";

        return $conn->executeQuery($sql, ['mntGroupId' => $mntGroupId], ['mntGroupId' => \PDO::PARAM_INT])->fetchAllAssociative();

    }

    /**
     * @return array
     * @throws Exception
     */
    public function getDevicesReadyToSendSms(): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = "SELECT d.id_device, d.devid, d.name
                FROM
                    system_labels_map s
                        LEFT JOIN
                    devices d ON s.id_device = d.id_device
                        LEFT JOIN
                    auth_profiles a ON d.auth_profile = a.id
                        LEFT JOIN
                    device_netdevs n ON d.id_device = n.id_device
                WHERE
                    s.id_system_label = :systemLabelId
                    AND a.username IS NOT NULL
                    AND (a.ssh_pwd IS NOT NULL OR a.ssh_key IS NOT NULL)
                    AND n.is_linkable IS TRUE
                    AND (n.ipaddr IS NOT NULL OR n.ip6addr IS NOT NULL)
                GROUP BY d.id_device
                HAVING COUNT(d.id_device) = 1";

        return $conn->executeQuery($sql, ['systemLabelId' => 3], ['systemLabelId' => \PDO::PARAM_INT])->fetchAllAssociative();

    }

    /**
     * @return int|mixed|string
     */
    public function findAllNotAssignedToWebTunnel(): mixed
    {

        $expr = $this->_em->getExpressionBuilder();

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->join('d.stats', 'stats')
            ->join('stats.syncStatus', 'status')
            ->where(
                $expr->notIn(
                    'd.id',
                    $this->_em->createQueryBuilder()
                        ->select('IDENTITY(tw.device)')
                        ->from(Tunnels::class, 'tw')
                        ->where('tw.device is not null')
                        ->getDQL()
                )
            )
            ->andWhere('status.id != :statusId')
            ->andWhere('d.deleted = :deleted')

            ->setParameter('statusId', '1')
            ->setParameter('deleted', '0')
        ;

        return $qb->getQuery()->getResult();

    }

    /**
     * @return array[]
     * @throws Exception
     */
    public function findAllNotAssignedToWebTunnelPlainSql(): array
    {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT d.id_device, d.devid, d.name
                FROM '. Device::ENTITY_TABLE_NAME .' d 
                INNER JOIN device_stats ds ON d.id_device = ds.id_device 
                INNER JOIN device_sync_statuses dss ON ds.sync = dss.id 
                WHERE d.id_device NOT IN (
                    SELECT id_device FROM tunnels WHERE id_device IS NOT NULL
                ) AND dss.id <> 1 AND d.deleted = 0
            ';

        return $conn->executeQuery($sql)->fetchAllAssociative();

    }

    /**
     * @param int $deviceId
     * @return Device|null
     * @throws Exception
     */
    public function deleteDevice(int $deviceId): ?Device
    {

        $entityManager = $this->getEntityManager();
        $conn = $this->getEntityManager()->getConnection();

        $entityManager->getConnection()->beginTransaction(); // suspend auto-commit

        try{

            $device = $this->findOneBy(array('id'=> $deviceId));

            $sql = '
            UPDATE '. Device::ENTITY_TABLE_NAME .' SET config_profile = NULL 
            WHERE config_profile = (
                SELECT automatic_profile FROM (SELECT * FROM '. Device::ENTITY_TABLE_NAME .') a WHERE id_device = :deviceId)
            ';
            $stmt = $conn->prepare($sql);
            $stmt->executeQuery(array('deviceId' => $deviceId));

            $sql = 'DELETE FROM '. MonitoringGroupMapping::ENTITY_TABLE_NAME .' WHERE id_device = :deviceId';
            $stmt = $conn->prepare($sql);
            $stmt->executeQuery(array('deviceId' => $deviceId));

            $device->setDeleted(true);

            $entityManager->persist($device);
            $entityManager->flush();

            $entityManager->getConnection()->commit();

        } catch (\Exception $e) {

            $entityManager->getConnection()->rollBack();
            throw $e;

        }

        return $device;

    }

    /**
     * @param array $devicesId
     * @return array[]
     * @throws Exception
     */
    public function getByDevIds(array $devicesId): array
    {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid FROM '. Device::ENTITY_TABLE_NAME .'
            WHERE id_device IN (?) 
            ';

        return $conn->executeQuery($sql, array($devicesId), array(\Doctrine\DBAL\Connection::PARAM_INT_ARRAY))->fetchAllAssociative();

    }

    /**
     * @param int|null $configProfileId
     * @param array $devicesId
     * @return array[]
     * @throws Exception
     */
    public function updateDevicesConfigProfile(?int $configProfileId, array $devicesId): array
    {

        $conn = $this->getEntityManager()->getConnection();

        //Increment Gui version for device and set status to beeing processed
        $sql = '
            UPDATE '. Device::ENTITY_TABLE_NAME .' d 
            INNER JOIN '. DeviceStats::ENTITY_TABLE_NAME .' ds ON ds.id_device = d.id_device
            SET d.config_profile = ?, d.gui_version = d.gui_version + 1, ds.sync = ?
            WHERE d.id_device IN (?) 
            ';

        return $conn->executeQuery($sql, array($configProfileId, DeviceSyncStatus::BEING_PROCESSED, $devicesId),
            array((is_int($configProfileId)) ? ParameterType::INTEGER : ParameterType::NULL,
                ParameterType::INTEGER, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY))->fetchAllAssociative();


    }

    /**
     * @return array
     * @throws Exception
     */
    public function getDevicesToClonedFrom(): array {

        $conn = $this->getEntityManager()->getConnection();

        //WHERE d.automatic is not null AND d.dev_type = :devTypeId AND d.deleted = 0
        $sql = '
            SELECT d.id_device, d.devid, d.name FROM devices as d
            WHERE d.automatic_profile is not null AND d.deleted = 0
            ';

        return $conn->executeQuery($sql, [], ['deviceIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY])->fetchAllAssociative();

    }

	public function getDevicesWithAutomaticProfile(): array
	{

		$conn = $this->getEntityManager()->getConnection();

		//WHERE d.automatic is not null AND d.dev_type = :devTypeId AND d.deleted = 0
		$sql = '
            SELECT d.id_device, d.devid, d.name FROM devices as d
            WHERE d.automatic_profile is not null AND d.deleted = 0
            ';

		return $conn->executeQuery($sql, [], ['deviceIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY])->fetchAllAssociative();


	}

}