<?php

namespace App\Controller\Admin\System;

use App\Exception\FileUploadException;
use App\Form\SystemLicenseApplyType;
use App\Form\SystemLicenseUploadType;
use App\Repository\ConfigRepository;
use App\Service\Command;
use App\Service\ConfigurationService;
use App\Service\FileUploader;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\Response;

class LicenseController extends Controller
{

    final public const MLERR_NONE = 0;
    final public const MLERR_FATAL = 1;
    final public const MLERR_OPFAIL = 2;
    final public const MLERR_LICINVALID = 3;
    final public const MLERR_LICSAME = 4;
    final public const MLERR_LICWARN = 5;

    final public const MLERR_MESSAGE_CODE_1 = 'general failure (license file provided cannot be accessed. operation not understood etc.)';
    final public const MLERR_MESSAGE_CODE_2 = 'operation failure (requested operation attempted and failed)';
    final public const MLERR_MESSAGE_CODE_3 = 'a license is invalid (either provided one or actual one depending on operation requested)';
    final public const MLERR_MESSAGE_CODE_4 = 'provided and actual licenses are identical (in all tested parameters - NOTE: e.g. the textual licensee field is not tested - user may proceed upgrade with using the force flag, you may join action with the case below, this is just a more subtle warning)';
    final public const MLERR_MESSAGE_CODE_5 = 'warning that by upgrading to the new license, some is lost. E.g. new license has less seats, worse flagset, earlier expiration, shorter MAMAS version set support, is no longer a production version -- any or all of these. More info on standard error output.';

    /**
     * @var string
     */
    private readonly string $serviceName;

    public function __construct(ConfigurationService $configurationService, private readonly FileUploader $fileUploader){

        $this->serviceName = $configurationService->getServiceName();

    }

    /**
     * @return RedirectResponse|Response
     * @throws \Exception
     */
    #[Route(path: '/admin/system/license', name: 'admin_system_license')]
    public function default(Request $request, ConfigRepository $configRepository,
                            Command $commandService, ConfigurationService $configurationService){

        try{

            $processDumpLicense = $commandService->checkLicense();
            //$processDumpLicense->mustRun();
            //changed to not throw exception when other exit code returned than 0
            //get standard error codes when licence is invalid
            $returnCode = $processDumpLicense->run();
            if($returnCode === self::MLERR_FATAL){
                $licenseText = 'Cannot get the license info.';
            }else{
                $license = $processDumpLicense->getOutput();
                $licenseText = $license; //$this->formatLicenseTextToHtml($license);
            }

        }catch (ProcessFailedException $exception){

            $licenseText = 'Cannot get the license info.';

        }

        $licenseUpdateEnabled = !$configurationService->isSaasMode();

        $formLicenseUpload = $this->createForm(SystemLicenseUploadType::class);
        $formLicenseUpload->handleRequest($request);

        $output = null;
        $outputUpdateCheck = null;
        $processUpdateErrorOutput = null;
        $outputUpdateCheckErrorCode = null;

        if ($formLicenseUpload->isSubmitted() && $formLicenseUpload->isValid()) {

            /** @var UploadedFile $licenseFile */
            $licenseFile = $formLicenseUpload->get('license')->getData();

            if ($licenseFile) {

                try {

                    $licenseFileName = $this->uploadFile($licenseFile);

                }catch (FileUploadException $e){

                    $this->addFlash(
                        'danger',
                        $e->getMessage() . ' Reason: ' . $e->getErrorMessage()
                    );

                    return $this->redirectToRoute('admin_system_license');

                }catch (\Exception $e){

                    $this->addFlash(
                        'danger', $e->getMessage()
                    );

                    return $this->redirectToRoute('admin_system_license');

                }

                try {

                    $processDumpLicense = $commandService->dumpLicense($licenseFileName);
                    $processDumpLicense->mustRun();
                    $output = $processDumpLicense->getOutput();
                    //$output = 'test';

                } catch (ProcessFailedException $exception) {

                    $this->addFlash(
                        'danger',
                        'License update failed. Reason: ' . $processDumpLicense->getErrorOutput()
                    );

                    return $this->redirectToRoute('admin_system_license');

                }

                try {

                    $processUpdateCheck = $commandService->updateLicense($licenseFileName, true);
                    $processUpdateCheck->mustRun();
                    $outputUpdateCheck = $processUpdateCheck->getOutput(); //$this->formatLicenseTextToHtml($processUpdateCheck->getOutput());

                } catch (ProcessFailedException) {

                    $processUpdateErrorOutput = $processUpdateCheck->getErrorOutput(); //$this->formatLicenseTextToHtml($processUpdateCheck->getErrorOutput());
                    $outputUpdateCheckErrorCode = $processUpdateCheck->getExitCode();

                }

            }

        }

        $isUpgradable = false;
        $useForce = false;

        if(isset($outputUpdateCheckErrorCode)) {

            if($outputUpdateCheckErrorCode === self::MLERR_NONE){

                $isUpgradable = true;

            }elseif($outputUpdateCheckErrorCode === self::MLERR_LICWARN){

                //If LICWARN show stderror and ask "Are you sure?"
                $isUpgradable = true;
                $useForce = true;

            }

        }

        $formApplyUpdate = $this->createForm(SystemLicenseApplyType::class,
            ['licenseFileName' => $licenseFileName ?? null, 'useForce' => $useForce]);
        $formApplyUpdate->handleRequest($request);

        if ($formApplyUpdate->isSubmitted()) {

            $licenseFile = $formApplyUpdate->get('file')->getData();
            $applyForce = $formApplyUpdate->get('useForce')->getData();

            try{

                $processUpdate = $commandService->updateLicense($licenseFile, false, $applyForce);
                $processUpdate->mustRun();

                $outputUpdate = $processUpdate->getOutput();


            } catch (ProcessFailedException $exception) {

                $this->addFlash(
                    'danger',
                    'License update failed. Reason:' . $processUpdate->getErrorOutput()
                );
                return $this->redirectToRoute('admin_system_license');


            } catch (\Exception){

                $this->addFlash(
                    'danger',
                    'License update failed.'
                );

                return $this->redirectToRoute('admin_system_license');

            }

            try{

                $processServiceRestart = $commandService->restartService();

            } catch (ProcessFailedException) {

                $this->addFlash(
                    'danger',
                    'License update failed. Reason:' . $processUpdate->getErrorOutput()
                );

                return $this->redirectToRoute('admin_system_license');

            }

            $this->addFlash(
                'success',
                'License update successfull'
            );

            return $this->redirectToRoute('admin_system_license');

        }

        return $this->render("system/license.html.twig", [
            'licenseUpdateEnabled' => $licenseUpdateEnabled,
            'form' => $formLicenseUpload->createView(),
            'formApplyUpdate' => (isset($formApplyUpdate)) ? $formApplyUpdate->createView() : null,
            'licenseText' => $licenseText,
            'output' => $output, //$this->formatLicenseTextToHtml($output),
            'outputUpdateCheck' => $outputUpdateCheck, //$this->formatLicenseTextToHtml($outputUpdateCheck),
            'processUpdateErrorOutput' => $processUpdateErrorOutput,
            'outputUpdateCheckErrorCode' => $outputUpdateCheckErrorCode,
            'isUpgradable' => $isUpgradable,
        ]);

    }

    /**
     * @param $licenseFile
     * @return string
     * @throws FileUploadException
     */
    private function uploadFile($licenseFile){

        try{

            return $this->fileUploader->upload($licenseFile);

        }catch (\Exception $exception){

            throw new FileUploadException($exception->getMessage());

        }

    }

    /**
     * @param $licenseText
     * @return array|string|string[]
     */
    private function formatLicenseTextToHtml($licenseText){

        $licenseText = str_replace(["\r", "\n"], '<br>', (string) $licenseText);
        $licenseText = str_replace('Dwarfg license dump:<br>', '', $licenseText);

        return $licenseText;

    }

}