<?php

namespace App\Form;

use App\Validator\Constraints\IsUniqueInDb;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Constraints\Type;

class SimCardType extends AbstractType
{

    /**
     * @param FormBuilderInterface $builder
     * @param array $options
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {

        $simCard = (key_exists('simCard', $options['data'])) ? $options['data']['simCard'] : null;
        $devices = $options['data']['devices'];

        $devicesChoices = [];
        $deviceChoicesAttr = [];
        foreach ($devices as $item){

            $devicesChoices[$item['devid']] = $item['id_device'];
            $deviceChoicesAttr[$item['devid']] = ['data-description' => $item['name']];

        }

        $builder
            ->add('iccid', TextType::class, [
                'constraints' => [
                    new NotBlank(),
                    new Type('string'),
                    new Regex('/^[0-9]{13,20}$/', 'Iccid must be in valid format'),
                    new IsUniqueInDb('iccid', (key_exists('iccid', $options['data'])) ? $options['data']['iccid'] : '', 'ICCID value must be unique.'),

                ],
                'label' => 'ICCID <strong>*</strong>',
                'label_html' => true,
                'help' => '<p style="padding-left: 20px;">ICCID number is a single string of 18–22 digits, an example is: 891004234814455936F</p>',
                'help_html' => true,
                'required' => true,
                'block_prefix' => 'modal_body'
            ])
            ->add('msisdn', TextType::class, [
                'constraints' => [
                    new Type('string'),
                    new Regex('/^[0-9]{3,17}$/', 'Msisdn must be in valid format'),
                    new IsUniqueInDb('msisdn', (key_exists('msisdn', $options['data'])) ? $options['data']['msisdn'] : '', 'MSISDN value must be unique.'),
                ],
                'label' => 'MSISDN',
                'help' => '<p style="padding-left: 20px;">MSISDN is a phone number in a format 00420_________</p>',
                'help_html' => true,
                'required' => false,
                'block_prefix' => 'modal_body'
            ])
            ->add('label', TextType::class, [
                'block_prefix' => 'modal_body',
                'required' => false,
            ])
            ->add('device', ChoiceType::class, [
                    'label' => 'Manual Device Selection',
                    'required' => false,
                    'choices' => $devicesChoices,
                    'choice_attr' => $deviceChoicesAttr,
                    'placeholder' => 'Select a Device',
                    'data' => (isset($options['data']['device'])) ? $options['data']['device'] : null,
                    'mapped' => false,
                    'auto_initialize' => false,
                    'attr' => ['data-row-class' => 'sim_card[device]'], //Important to setup to send data from the form
                    'block_prefix' => 'modal_body'
            ])
            ->add('save', SubmitType::class, [
                'label' => (isset($simCard) && $simCard->getId()) ? '<i class="fas fa-floppy-disk"></i> Update' : '<i class="fas fa-plus"></i> Add SIM Card',
                'label_html' => true,
                'block_prefix' => 'modal_footer',
                'attr' => ['data-row-class' => 'alert_targetDevice', 'class' => 'btn btn-primary float-right']
            ])
        ;
    }

}