<?php

namespace App\Controller\Admin\Firmware;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Controller\Admin\BaseController;
use App\DataTable\FirmwareDatatable;
use App\DataTable\Type\FwVersionDatatableType;
use App\Entity\Device\DeviceOsType;
use App\Entity\Device\DeviceOsVersion;
use App\Entity\Device\FwFileName;
use App\Exception\FileUploadException;
use App\Exception\FirmwareNotFoundException;
use App\Form\Firmware\FwFileUploadType;
use App\Form\Firmware\FwType;
use App\Repository\DeviceOsVersionRepository;
use App\Repository\EventLogRepository;
use App\Service\FileUploader;
use App\ValueObject\Firmware;
use Doctrine\Persistence\ManagerRegistry;
use Kreyu\Bundle\DataTableBundle\DataTableFactoryAwareTrait;
use Kreyu\Bundle\DataTableBundle\Query\ArrayProxyQuery;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route(path: '/admin/firmware', name: 'admin_firmware')]
class FirmwareController extends BaseController
{
    use DataTableFactoryAwareTrait;

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @param DeviceOsVersionRepository $osVersionRepository
     * @return JsonResponse|Response
     * @throws \Exception
     */
    #[Route(path: '', name: '')]
    public function firmware(
        Request $request,
        ManagerRegistry $managerRegistry,
        DatatableFactory $factory,
        DatatableResponse $responseService,
        DeviceOsVersionRepository $osVersionRepository
    ): JsonResponse|Response {

        $deviceOsTypes = $managerRegistry->getRepository(DeviceOsType::class)->findAll();
        $osVersions = $osVersionRepository->getAvailableVersions();

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $datatable = $factory->create(FirmwareDatatable::class);
        $datatable->buildDatatable(['osTypes' => $deviceOsTypes, 'osVersions' => $osVersions]);

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        $form = $this->createForm(FwType::class, ['osTypes' => $deviceOsTypes]);

        return $this->render('firmware/index.html.twig', [
            'datatable' => $datatable, 'filter' => $this->filter, 'form' => $form->createView()
        ]);
    }

    /**
     * @throws FirmwareNotFoundException
     */
    #[Route(path: '/detail/{fwId}', name: '_detail')]
    public function firmwareDetail(
        int $fwId,
        Request $request,
        ManagerRegistry $managerRegistry,
        ParameterBagInterface $parameterBag
    ): Response {

        $fw = $managerRegistry->getRepository(DeviceOsVersion::class)->findOneBy(['id' => $fwId]);

        if (!$fw /*|| $fw->isDeleted()*/) {
            throw new FirmwareNotFoundException();
        }

        $directoryName = $fw->getOsType()->getDirectoryName();
        $rootPath = $parameterBag->get('kernel.project_dir');

        //dump($rootPath . '/../firmware/' . $directoryName . '/' . $fw->getVersion());

        $finder = new Finder();
        $firmwares = [];
        $id = 0;

        try {
            // find all files in the current directory
            $finder->files()->in($rootPath . '/../firmware/' . $directoryName . '/' . $fw->getVersion())
                //->depth(0)
                ->name('*.bin');

            // check if there are any search results
            if ($finder->hasResults()) {
                //dump($finder);
                foreach ($finder as $key => $file) {
                    $absoluteFilePath = $file->getRealPath();
                    $fileNameWithExtension = $file->getRelativePathname();
                    $firmwares[] = new Firmware(
                        $id++,
                        $fileNameWithExtension,
                        $file->getSize(),
                        date('Y-m-d H:i:s', $file->getCTime())
                    );
                }
            }
        } catch (\Exception) {
            //When folder does not exist
        }

        $dataTable = $this->createDataTable(
            FwVersionDatatableType::class,
            new ArrayProxyQuery($firmwares),
            ['fwId' => $fwId]
        );
        $dataTable->handleRequest($request);

        return $this->render('firmware/list.html.twig', [
            'firmware' => $fw,
            'firmwares' => $dataTable->createView(),
        ]);
    }

    /**
     * @throws FirmwareNotFoundException
     */
    #[Route(path: '/{fwId}/upload', name: '_upload', methods: ['GET', 'POST'])]
    public function uploadFirmwareAction(
        int $fwId,
        Request $request,
        FileUploader $fileUploader,
        ManagerRegistry $managerRegistry,
        EventLogRepository $eventLogRepository
    ): Response {

        $fw = $managerRegistry->getRepository(DeviceOsVersion::class)->findOneBy(['id' => $fwId]);

        if (!$fw /*|| $fw->isDeleted()*/) {
            throw new FirmwareNotFoundException();
        }

        $formFwUpload = $this->createForm(FwFileUploadType::class, ['fwId' => 1]);
        $formFwUpload->handleRequest($request);

        if ($formFwUpload->isSubmitted() && $formFwUpload->isValid()) {

            /** @var UploadedFile $fwFile */
            $fwFile = $formFwUpload->get('firmware')->getData();

            if ($fwFile) {
                try {
                    [$message, $filename] = $fileUploader->uploadFwFile($fwFile, $fw->getId());
                    $eventLog = $eventLogRepository->addEventLog(
                        FwFileName::class, $this->getUser()->getId(),
                        EventLogRepository::ACTION_CREATE, $filename, $fwId
                    );
                    $eventLogRepository->persist($eventLog);
                    $this->addFlash('success', $message);
                    return $this->redirectToRoute('admin_firmware_upload', ['fwId' => $fwId]);
                } catch (FileUploadException $e) {
                    $this->addFlash(
                        'danger',
                        $e->getMessage() . ' Reason: ' . $e->getErrorMessage()
                    );

                    return $this->redirectToRoute('admin_firmware_upload', ['fwId' => $fwId]);
                } catch (\Exception $e) {
                    $this->addFlash(
                        'danger',
                        $e->getCode() . ':' . $e->getMessage()
                    );

                    return $this->redirectToRoute('admin_firmware_upload', ['fwId' => $fwId]);
                }
            }
        }

        return $this->render('firmware/upload.html.twig', [
            'form' => $formFwUpload->createView(),
            'fw' => $fw
        ]);
    }
}
