<?php

declare(strict_types=1);

namespace App\DataTable;

use App\Bundles\Sg\DatatablesBundle\Datatable\Column\MultiActionColumn;
use App\Bundles\Sg\DatatablesBundle\Datatable\Editable\CustomTextEditable;
use App\Bundles\Sg\DatatablesBundle\Datatable\Filter\CustomSelectFilter;
use App\DataTable\Column\CustomArrayColumn;
use App\DataTable\Column\CustomDeviceFreshnessColumn;
use App\DataTable\Column\CustomDeviceStatusColumn;
use App\DataTable\Column\CustomIpAddressesArrayColumn;
use App\DataTable\Column\CustomMonitoringGroupArrayColumn;
use App\Entity\Device\VDevice;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Column\Column;
use Sg\DatatablesBundle\Datatable\Column\LinkColumn;
use Sg\DatatablesBundle\Datatable\Column\MultiselectColumn;
use Sg\DatatablesBundle\Datatable\Filter\SelectFilter;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;

/**
 * Class DeviceDatatable
 * @package App\Datatables
 */
class DeviceDatatable extends BaseDatatable
{
    public const tableColumnName_SnmpName = 'snmpname';

    public const username_SnmpName = 'SNMP Name';

    public const showColumns = ['devId', 'name', 'devTypeName', 'OsVerOsTypeName', 'OsVerVersion', 'Labels',
                        'MonitoringGroups', 'StatsSyncStatusDescription', 'StatLiveMissCountDescription'];

    public const availableColumns = ['devId' => 'DevId', 'name' => 'Name', 'devTypeName' => 'Type', 'SnpmDevTypeName' => 'SNMP subtype',
        'OsVerOsTypeName' => "OS", 'OsVerVersion' => 'Version', 'Labels' => 'Labels', 'SystemLabels' => 'System Labels',
        'IpAddresses' => 'Ip Addresses', 'MonitoringGroups' => 'MonitoringGroups',
        'ConfigProfileAutomaticDevId' => 'Cloned From', 'StatsSyncStatusDescription' => 'Sync Status',
        'StatLiveMissCountDescription' => 'Freshness', 'SerialNumber' => 'Serial Number',
        'MacAddress' => 'Mac Address', self::tableColumnName_SnmpName => self::username_SnmpName,
        'Model' => 'Model', 'HostName' => 'Host Name', 'Uuid' => 'UuId',
        'Iccid' => 'IccId', 'ActiveSim' => 'Active Sim', 'SignalStrength' => 'Signal Strength', 'SignalQuality' => 'Signal Quality',
		'SignalCsq' => 'Signal CSQ', 'SignalRssi' => 'Signal RSSI',
        'Temperature' => 'Temperature', 'AgentProfile' => 'Agent Profile', 'AuthProfile' => 'Auth Profile',
		'simIpaddr' => 'SIM IPv4', 'simIp6addr' => 'SIM IPv6',
        'Note1' => 'Note 1', 'Note2' => 'Note 2', 'Note3' => 'Note 3',
        '_Number1' => 'Number 1', '_Number2' => 'Number 2', '_Number3' => 'Number 3', '_Number4' => 'Number 4', '_Number5' => 'Number 5',
        '_Number6' => 'Number 6', '_Number7' => 'Number 7', '_Number8' => 'Number 8', '_Number9' => 'Number 9',
        '_Float1' => 'Float 1', '_Float2' => 'Float 2', '_Float3' => 'Float 3', '_Float4' => 'Float 4', '_Float5' => 'Float 5',
        '_Float6' => 'Float 6', '_Float7' => 'Float 7', '_Float8' => 'Float 8', '_Float9' => 'Float 9',
        '_Text1' => 'Text 1', '_Text2' => 'Text 2', '_Text3' => 'Text 3', '_Text4' => 'Text 4', '_Text5' => 'Text 5',
        '_Text6' => 'Text 6', '_Text7' => 'Text 7', '_Text8' => 'Text 8', '_Text9' => 'Text 9'
    ];

    protected string $entity = VDevice::class;
    protected string $name = 'device_datatable';

    protected array $devices;
    protected array $syncStatuses;
    protected array $freshnessStatuses;
    protected array $deviceTypes;
    protected array $osTypes;
    protected array $osVersions;
    protected array $deviceLabels;
    protected array $deviceSystemLabels;
    protected array $deviceMonitoringGroups;
    protected array $agentProfiles;
    protected array $authProfiles;
    protected array $columns;
    protected array $deviceSimCards;
    protected array $models;

    /**
     * {@inheritdoc}
     */
    public function buildDatatable(array $options = array(), bool $parentCall = false): void
    {

        $this->initData($options);

        parent::buildDatatable($options);

        $this->options->setOrder([1,'asc']); //this is fixed by custom template to inject it before table data

        if (!$parentCall) {
            $this->extensions->set(
                [
                    'buttons' => [
                        'create_buttons' => [
                            [
                                'action' => [
                                    'template' => '/datatable/device/dataExport.js.twig',
                                    'vars' => array('link' => 'admin_devices_data_download'),

                                ],
                                'class_name' => 'btn-add export',
                                'name' => 'refresh',
                                'text' => '<i class=\"fas fa-file-export\"></i> Export',
                                'title_attr' => 'Export',
                            ],
                            $this->getRefreshButton(),
                        ],
                    ],
                ]
            );

			$actions = $this->getMassActions();
            $this->getMultiselectActions($actions);
            $this->getColumns($this->columns);
            $this->columnBuilder

                // ->add('id', new ChildRow(), array('ajax' =>array('route' =>'ci_supply_by_chem', 'args' => array('id' => 'id')) ))
                /* ->add('id', Column::class, array(
                     'title' => 'Id',
                     'width' => '100%;',
                     'filter' => array(TextFilter::class,
                         array(
                             'cancel_button' => true,
                         ),
                     ),
                 ))*/

                ->add(null, MultiActionColumn::class, array(
                    'title' => $this->translator->trans('sg.datatables.actions.column_name'),
                    'width' => '100%;',
                    'class_name' => 'row_device_actions',
                    'actions' => array(
                        array(
                            'route' => 'admin_devices_detail',
                            'route_parameters' => array(
                                'deviceId' => 'id',
                            ),
                            'label' => '' . $this->translator->trans('sg.datatables.actions.detail'),
                            'icon' => self::ICON_DETAIL,
                            'attributes' => array(
                                'rel' => 'tooltip',
                                'title' => $this->translator->trans('sg.datatables.actions.detail'),
                                'class' => 'btn btn-primary btn-sm',
                                'role' => 'button',
                                'style' => 'white-space: nowrap' //icon on the same line as title
                            ),

                        ),

                    ),
                    'menu' => [
                        [
                            'label' => 'Reboot',
                            'icon' => 'fa fa-refresh',
                            'route' => 'ajax_device_reboot',
                                'route_parameters' => [
                                    'deviceId' => 'id',
                                ],
                            'attributes' => [
                                'id' => 'reboot-button',
                                'data-toggle' => 'modal',
                                'data-target' => "#confirm-reboot",
                            ]
                        ]
                    ]
                ))

            ;
        }
    }

    public function initData(array $options): void
    {

        $this->columns = $options['columns'];
        $this->devices = $options['devices'];
        $this->syncStatuses = $options['syncStatuses'];
        $this->freshnessStatuses = $options['freshnessStatuses'];
        $this->deviceTypes = $options['deviceTypes'];
        $this->osTypes = $options['osTypes'];
        $this->osVersions = $options['osVersions'];
        $this->deviceLabels = $options['deviceLabels'];
        $this->deviceSystemLabels = $options['deviceSystemLabels'];
        $this->deviceSimCards = $options['deviceSimCards'];
        $this->deviceMonitoringGroups = $options['deviceMonitoringGroups'];
        $this->agentProfiles = $options['agentProfiles'];
        $this->authProfiles = $options['authProfiles'];
        $this->models = $options['models'];
    }

    public function getMultiselectActions(array $actions, array $options = []): void
    {

        $this->columnBuilder->add(null, MultiselectColumn::class, [
                'start_html' => '<div class="start_checkboxes">',
                'end_html' => '</div>',
                'width' => '10px',
                'add_if' => function () {
                    return $this->authorizationChecker->isGranted('ROLE_ADMIN');
                },
                'value' => 'id',
                'value_prefix' => true,
                //'render_actions_to_id' => 'sidebar-multiselect-actions',
                'actions' => $actions,
            ]);
    }

    private function getDeviceToMonitoringGroupAction(): array
    {

        return [
            'icon' => 'fa fa-layer-group',
            'label' => 'Add to Monitoring group',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Add to Monitoring Group',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-devices-to-monitoring-group', //Element ID of modal dialog
            ],
            'start_html' => '<div class="input-group">',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

	protected function getMassActions(): array
	{

		$actions = [];

		$actions[] = $this->getDeviceToMonitoringGroupAction();
		$actions[] = $this->getAssignCustomLabelsAction();
		$actions[] = $this->getAssignSystemLabelsAction();
		$actions[] = $this->getAssignFwVersion();
		$actions[] = $this->getAssignConfigProfile();
		$actions[] = $this->getAssignAgentProfile();
		//Remove buttons
		$actions[] = $this->getRemoveConfigProfile();
		$actions[] = $this->getRemoveCustomLabel();
		$actions[] = $this->getRemoveSystemLabel();

		return $actions;

	}

    protected function getAssignCustomLabelsAction(): array
    {

        return [
            'icon' => 'fa fa-tags',
            'label' => 'Assign Custom Labels',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign labels',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-custom-labels',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

	protected function getAssignSystemLabelsAction(): array
	{

		return [
			'icon' => 'fa fa-tags',
			'label' => 'Assign System Labels',
			'attributes' => [
				'rel' => 'tooltip',
				'title' => 'Assign System Labels',
				'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
				'role' => 'button',
				'id' => '#add-device-system-labels',
			],
			'start_html' => '',
			'end_html' => '',
			'render_if' => function () {
				return $this->authorizationChecker->isGranted('ROLE_ADMIN');
			},
		];
	}

    protected function getAssignFwVersion(): array
    {

        return [
            'icon' => 'fa fa-file',
            'label' => 'Assign Fw Version',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Fw',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-fw',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignConfigProfile(): array
    {

        return [
            'icon' => 'fa fa-cog',
            'label' => 'Assign Config Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Config Profile',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-config',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignAgentProfile(): array
    {

        return [
            'icon' => 'fa fa-file-lines',
            'label' => 'Assign Agent Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Agent Profile',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-agent-profile',
            ],
            'start_html' => '',
            'end_html' => '</div>',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getRemoveConfigProfile(): array
    {

        return [
            'route' => 'ajax_device_remove_config_profile',
            //'route_parameters' => [],
            'icon' => 'fa fa-times',
            'label' => 'Remove Config Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Remove Config Profile',
                'class' => 'btn btn-danger btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#remove-device-config',
            ],
            'start_html' => '<div class="input-group">',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

	protected function getRemoveCustomLabel(): array
	{

		return [
			'route' => 'ajax_device_remove_custom_label',
			//'route_parameters' => [],
			'icon' => 'fa fa-times',
			'label' => 'Remove Custom Label',
			'attributes' => [
				'rel' => 'tooltip',
				'title' => 'Remove Custom Label',
				'class' => 'btn btn-danger btn-xs mb-1 mr-sm-2',
				'role' => 'button',
				'id' => '#remove-device-custom-label',
			],
			'start_html' => '',
			'end_html' => '',
			'render_if' => function () {
				return $this->authorizationChecker->isGranted('ROLE_ADMIN');
			},
		];
	}

	protected function getRemoveSystemLabel(): array
	{

		return [
			'route' => 'ajax_device_remove_system_label',
			//'route_parameters' => [],
			'icon' => 'fa fa-times',
			'label' => 'Remove System Label',
			'attributes' => [
				'rel' => 'tooltip',
				'title' => 'Remove System Label',
				'class' => 'btn btn-danger btn-xs mb-1 mr-sm-2',
				'role' => 'button',
				'id' => '#remove-device-system-label',
			],
			'start_html' => '',
			'end_html' => '</div>',
			'render_if' => function () {
				return $this->authorizationChecker->isGranted('ROLE_ADMIN');
			},
		];
	}

    public function getColumns($showColumnNames): void
    {

        foreach ($showColumnNames as $columnName) {

            if(str_starts_with($columnName, '_Number')) {

                $number = (int)str_replace('_Number', '', $columnName);
                $this->getColumnNumber($number);

            }elseif(str_starts_with($columnName, '_Float')){

                $number = (int)str_replace('_Float', '', $columnName);
                $this->getColumnFloat($number);

            }elseif(str_starts_with($columnName, '_Text')){

                $number = (int)str_replace('_Text', '', $columnName);
                $this->getColumnText($number);

            }else{

                $this->getColumnByName($columnName);

            }

        }

       /* $this->getColumnNote1();
        $this->getColumnNote2();
        $this->getColumnNote3();*/

        /*$this->getColumnByName('devId');
        $this->getColumnByName('name');
        $this->getColumnByName('devTypeName');
        $this->getColumnByName('OsVerOsTypeName');
        $this->getColumnByName('OsVerVersion');
        $this->getColumnByName('Labels');
        $this->getColumnMonitoringGroups();*/

        //$this->getColumnByName('MacAddr');
        //$this->getColumnByName('Serial');
        //$this->getColumnByName('HostName');
        //$this->getColumnByName('Uuid');

       /* $this->getColumnByName('ConfigProfileAutomaticDevId');
        $this->getColumnByName('StatsSyncStatusDescription');
        $this->getColumnByName('StatLiveMissCountDescription');*/
    }

    private function getColumnByName(string $name): void
    {

        if (method_exists($this, 'getColumn' . $name)) {
            $this->{'getColumn' . $name}();
        }
    }

    private function getColumnDevId(): void
    {

        $this->columnBuilder->add(
            'devId',
            Column::class,
            [
                'name' => 'devId',
                'title' => 'DevId',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'search_type' => 'like',
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnName(): void
    {

        $this->columnBuilder->add(
            'name',
            Column::class,
            [
                'name' => 'name',
                'title' => 'Name',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
                'editable' => [
                    CustomTextEditable::class,[
                        'url' => 'app_datatables_edit',
                        'maxlength' => 32,
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                //if ($row['createdBy']['id'] == $this->getUser()->getId() or true === $this->isAdmin()) {
                                return true;
                                //};
                            }

                            return true;
                        },
                        'mode' => 'inline',
                    ],
                ],
            ]
        );
    }

    private function getColumnDevTypeName(): void
    {
        $this->addDefaultSelectableFilterColumns('devType.name', 'Type', $this->getOptionsArrayFromEntities($this->deviceTypes, 'name', 'name'));
    }

	private function getColumnSnpmDevTypeName(): void
	{
		$this->addDefaultSelectableFilterColumns('snmpSubDevType.name', 'SNMP subtype', $this->getOptionsArrayFromEntities($this->deviceTypes, 'name', 'name'));
	}

    private function getColumnOsVerOsTypeName(): void
    {
        $this->addDefaultSelectableFilterColumns('osVer.osType.name', 'OS', $this->getOptionsArrayFromEntities($this->osTypes, 'name', 'name'));
    }

    private function getColumnOsVerVersion(): void
    {
        //$this->addDefaultSelectableFilterColumns('osVer.version', 'Version', $this->getOptionsArrayFromEntities($this->osVersions, 'version', 'version'));
        $this->addDefaultSelectableFilterColumns('osVer.version', 'Version', $this->osVersions);
    }

    private function getColumnLabels(): void
    {

        $this->columnBuilder->add(
            'label',
            CustomArrayColumn::class,
            [
                'name' => 'labels',
                'title' => 'Labels',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({l}.label) FROM App\Entity\Label\LabelDeviceMap {p}
                                JOIN App\Entity\Label\Label {l} with {l}.id = {p}.labelId
                                WHERE {p}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntitiesWithNeg($this->deviceLabels, 'id', 'label'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnIpAddresses(){

        /*$this->columnBuilder->add(
            'ipAddresses', CustomIpAddressesArrayColumn::class, [
                'name' => 'ipAdresses',
                'title' => 'IP Addresses',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({net}.label, '_', {net}.ipAddr, '_', COALESCE({net}.ip6Addr, 'NULL'))
                            FROM App:DeviceNetDev {net}
                            WHERE {net}.device = device AND {net}.active = true AND {net}.isLinkable = true
                             AND {net}.ipAddr is not NULL)",
                'type_of_field' => 'string',
                'searchable' => true,
                'orderable' => false,
                'filter' => [
                    TextFilter::class,[
                        'classes' => '',
                        'search_type' => 'like',
                       'cancel_button' => true,
                    ],
                ],
            ]
        );*/

        $this->columnBuilder->add(
            'netDevs', CustomIpAddressesArrayColumn::class, [
                'name' => 'ipAdresses',
                'title' => 'IP Addresses',
                'width' => '100%;',
                'type_of_field' => 'string',
                'searchable' => true,
                'orderable' => false,
                'filter' => [
                    TextFilter::class,[
                        'classes' => '',
                        'search_type' => 'like',
                        'cancel_button' => true,
                    ],
                ],
            ]
        );


    }

    private function getColumnSystemLabels(): void
    {

        $this->columnBuilder->add(
            'systemLabel',
            CustomArrayColumn::class,
            [
                'name' => 'systemLabels',
                'title' => 'System Labels',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({sl}.label) FROM App\Entity\Label\SystemLabelDeviceMap {pa}
                                JOIN App\Entity\Label\SystemLabel {sl} with {sl}.id = {pa}.labelId
                                WHERE {pa}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntities($this->deviceSystemLabels, 'id', 'label'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnNote1(): void
    {

        $this->getColumnNotes(1);
    }

    private function getColumnNote2(): void
    {

        $this->getColumnNotes(2);
    }

    private function getColumnNote3(): void
    {

        $this->getColumnNotes(3);
    }

    private function getColumnNumber1(): void
    {

        $this->getColumnNumber(1);
    }

    private function getColumnNotes(int $noteNum): void
    {

        $this->columnBuilder->add(
            'noteDevice.note' . $noteNum,
            Column::class,
            [
                'name' => 'note' . $noteNum,
                'title' => $this->translator->trans('Note' . $noteNum, domain: 'messages'), //'Note ' . $noteNum,
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
                'editable' => [
                    CustomTextEditable::class,[
                        'url' => 'app_datatables_edit',
                        'maxlength' => 64, //maxlength custom implementation
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                //if ($row['createdBy']['id'] == $this->getUser()->getId() or true === $this->isAdmin()) {
                                return true;
                                //};
                            }

                            return true;
                        },
                        'mode' => 'inline',
                    ],
                ],
            ]
        );
    }

    private function getColumnNumber(int $numberNum): void
    {
        $this->addDefaultTextColumn('customFieldDevice.number' . $numberNum, 'Number ' . $numberNum);
    }

    private function getColumnFloat(int $numberNum): void
    {
        $this->addDefaultTextColumn('customFieldDevice.float' . $numberNum, 'Float ' . $numberNum);
    }

    private function getColumnText(int $numberNum): void
    {
        $this->addDefaultTextColumn('customFieldDevice.text' . $numberNum, 'Text ' . $numberNum);
    }

    private function getColumnMonitoringGroups(): void
    {

        $this->columnBuilder->add(
            'monitoringGroups',
            CustomMonitoringGroupArrayColumn::class,
            [
                'router' => $this->router,
                'name' => 'monitoringGroups',
                'title' => 'Monitoring Groups',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT(CONCAT_WS('_', {mg}.id, {mg}.name)) FROM App\Entity\MonitoringGroup\MonitoringGroupMapping {mgmapping}
                                JOIN App\Entity\MonitoringGroup\MonitoringGroup {mg} with {mg}.id = {mgmapping}.monitoringGroup
                                WHERE {mgmapping}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] + $this->getOptionsArrayFromEntities($this->deviceMonitoringGroups),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnMacAddress(): void
    {
        $this->addDefaultTextColumn('macAddr', 'MAC');
    }

    private function getColumnSerialNumber(): void
    {
        $this->addDefaultTextColumn('serial', 'SN');
    }

    private function getColumnHostName(): void
    {
        $this->addDefaultTextColumn('hostname', 'Hostname');
    }

    private function getColumnModel(): void
    {
        $this->addDefaultSelectableFilterColumns('modelDevice.model', 'Model', $this->getOptionsArrayFromEntities($this->models, 'model', 'model'));
    }

    private function getColumnSnmpname(): void
    {
        $this->addDefaultTextColumn(self::tableColumnName_SnmpName, self::username_SnmpName);
    }

    private function getColumnUuid(): void
    {
        $this->addDefaultTextColumn('uuid', 'UUID');
    }

    private function getColumnIccid(): void
    {

        $this->columnBuilder->add(
            'iccid',
            CustomArrayColumn::class,
            [
                'name' => 'deviceSimCards',
                'title' => 'Iccid',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({sc}.iccid) FROM App\Entity\Tool\SimCard {sc}
                                WHERE {sc}.deviceUser = vdevice OR {sc}.deviceLastSeen = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsArrayFromEntities($this->deviceSimCards, 'id', 'iccid'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnActiveSim(): void
    {
        $this->addDefaultTextColumn('activeSim', 'Active Sim');
    }

    private function getColumnSignalStrength(): void
    {
        $this->addDefaultTextColumn('stats.signalStrength', 'Signal signalStrength');
    }

    private function getColumnSignalQuality()
    {
        $this->addDefaultTextColumn('stats.signalQuality', 'Signal Quality');
    }

	private function getColumnSignalCsq()
    {
        $this->addDefaultTextColumn('stats.signalCsq', 'Signal CSQ');
    }

	private function getColumnSignalRssi()
	{
		$this->addDefaultTextColumn('stats.signalRssi', 'Signal RSSI');
	}

    private function getColumnTemperature(): void
    {
        $this->addDefaultTextColumn('stats.temperature', 'Temperature');
    }

	private function getColumnSimIpaddr(): void
	{
		$this->addDefaultTextColumn('stats.simIpaddr', 'SIM IPv4');
	}

	private function getColumnSimIp6addr(): void
	{
		$this->addDefaultTextColumn('stats.simIp6addr', 'SIM IPv6');
	}

    private function getColumnConfigProfileAutomaticDevId(): void
    {

        $this->columnBuilder->add(
            'configProfile.automatic.devId',
            LinkColumn::class,
            [
                'name' => 'clonedFrom',
                'title' => 'Cloned from',
                'default_content' => '',
                'width' => '100%;',
                'route' => 'admin_devices_detail',
                'route_params' => function ($row) {
                    return [
                        'deviceId' => $row['configProfile']['automatic']['id'],
                    ];
                },
                'filter' => [
                    SelectFilter::class,[
                        'search_type' => 'eq',
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsFromArray($this->devices, 'devid', 'devid'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnStatsSyncStatusDescription(): void
    {
        $this->addDefaultSelectableFilterColumns('stats.syncStatus.description', 'Sync Status',
            $this->getOptionsArrayFromEntities($this->syncStatuses, 'description', 'description'), '', CustomDeviceStatusColumn::class);
    }

    private function getColumnStatLiveMissCountDescription(): void
    {
        $this->addDefaultSelectableFilterColumns('stats.liveMissCount.description', 'Freshness',
            $this->getOptionsArrayFromEntities($this->freshnessStatuses, 'description', 'description'), '', CustomDeviceFreshnessColumn::class);
    }

    private function getColumnAgentProfile(): void
    {
        $this->addDefaultSelectableFilterColumns('agentProfile.profileName', 'Agent Profile',
            $this->getOptionsArrayFromEntities($this->agentProfiles, 'profileName', 'profileName'));
    }

    private function getColumnAuthProfile(): void
    {
        $this->addDefaultSelectableFilterColumns('authProfile.profileName', 'Auth Profile',
            $this->getOptionsArrayFromEntities($this->authProfiles, 'profileName', 'profileName'));

    }

    private function addDefaultSelectableFilterColumns(string $dqlColumn, string $title, array $options, string $defaultContentValue = '', string $columnType = Column::class): void
    {

        $this->columnBuilder->add(
            $dqlColumn,
            $columnType,
            [
                'name' => strtolower(str_replace(" ", "", $title)),
                'title' => $title,
                'default_content' => $defaultContentValue,
                //'searchable' => true, //is it needed?
                //'orderable' => true, //is it needed?
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] + $options,
                        'cancel_button' => true,
                    ],
                ],
            ]
        );

    }

    /**
     * @throws \Exception
     */
    private function addDefaultTextColumn(string $dqlColumn, string $title, string $defaultContentValue = ''): void
    {

        $this->columnBuilder->add(
            $dqlColumn,
            Column::class,
            [
                'name' => strtolower(str_replace(" ", "", $title)),
                'title' => $title,
                'default_content' => $defaultContentValue,
                'type_of_field' => 'string',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

}
