#!/bin/bash

BASEDEFS="base_defs"
ERR=0

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR/$BASEDEFS" || . "$BASEDEFS" || { echo "Fatal: Definition file ($BASEDEFS) not found."; exit 1; }
. "$DIR/$BASEFUNCS" || . "$BASEFUNCS" || { echo "Fatal: Basic functions ($BASEFUNCS) not found."; exit 1; }

# you can specify extra variables using ENVIRONMENTAL variable ARGS_EXTRA
# supported variables are:
# -d <generic-area-debuglevel-override-value>
# -a <all-specific-areas-debuglevel-override-value>
# these two variables override generic or all specific areas values
# that are usually specified in /opt/${SHORTNAME}_<your_domain>/dwarflib_cfg.txt
# note that you can force $NAME reloading the debug values from that file
# at anytime by calling this script with --reload-cfg param
ARGS_EXTRA=${ARGS_EXTRA:-}
ARG_START="--start"
ARG_FG="--fg"
ARG_VG="--vg"
ARG_STOP="--stop"
ARG_PID="--pid"
ARG_STATUS="--status"
ARG_CFGRELOAD="--reload-cfg"
ARG_FORCE="--force"
ARG_DBGREOPEN="--reopen-dbg"
ARG_STOPWEBSSH="--stop-webssh"
ARG_STARTWEBSSH="--start-webssh"
ARG_NOROOT="--groot"
WEBSSH_PIDFILE_PREFIX="sshwifty_"
WEBSSH_PIDFILE_POSTFIX=".pid"

# globvars
g_loopin_systemd=""
g_fn_pidfile="$LOGDIR/$PIDLOG"
g_runpid=0
g_force=0
g_fg=""
g_vg=""
# globconsts
g_cnum_wdelay=1
g_cnum_slackpid=7 # in seconds, slack time for Dwarfg to initialize and write out pid into PID file
g_cnum_maxwcount=5
g_ctxt_vallog="/tmp/${SHORTNAME}_valgrind_$$.txt"

print_help() {
	echo
	echo "$NAME control script"
	echo
	echo "   Run with one of the commands:"
	echo "      $ARG_START ... starts $DGDAEMON"
	echo "      $ARG_STOP ... stops $DGDAEMON"
	echo "      $ARG_PID ... print $DGDAEMON PID (0 means not running). Note: can be run under nonroot."
	echo "      $ARG_STATUS ... print short status message and PID (if running)"
	echo "      $ARG_CFGRELOAD ... reload logging configuration"
	echo "      $ARG_DBGREOPEN ... reopen debug file (used when rotating log)"
	echo "      $ARG_STOPWEBSSH <n> ... stop ssh tunnel <n> web GW (WebSSH/SSHWifty). Use 0 to stop all. Note: does not touch Apache config."
	echo "      $ARG_STARTWEBSSH <n> <port> <pwd> ... start ssh tunnel <n> web GW (WebSSH/SSHWifty). Specify local port of already connected device SSH tunnel and password for web user to authenticate. Note: Apache CFG link handled separately."
	echo "   For debug purposes you can use these but put them *BEFORE* command:"
	echo "      $ARG_NOROOT ... attempt to run the requested action even when non root. Must come as first option if at all."
	echo "      $ARG_FORCE ... perform start/stop op outside systemd (must come before required op arg)"
	echo "      $ARG_FG ... (DEBUG) run in foreground (implies $ARG_FORCE)"
	echo "      $ARG_VG ... (DEBUG) run inside valgrind (implies $ARG_FG and $ARG_FORCE)"
	echo
	fail_fat "No action performed."
}

stop_webssh() {
	#WSSHPID=$(pgrep -u "$APPUSER" wssh)
	serialno="$1"
	silent="$2"
	tarid=$(id -u "$APPUSER")
	[[ -z "$serialno" || "$serialno" != "${serialno//[^0-9]}" ]] && { echo "$FUNCNAME: need serialno of ssh tunnel or 0 to terminate all of them." >&2; return 1; }
	if [ 0 -eq "$serialno" ]; then
		echo "$FUNCNAME: Killing all ssh tunnel gateways running under user $APPUSER was requested..."
		pgrep -u "$APPUSER" sshwifty && pkill -u "$APPUSER" sshwifty
		sleep 1
		pgrep -u "$APPUSER" sshwifty && pkill -u "$APPUSER" -9 sshwifty
		if pgrep -u "$APPUSER" sshwifty; then
			echo "$FUNCNAME: Unable to terminate all sshwifty instances.">&2
		else
			echo "$FUNCNAME: All sshwifty instances were terminated."
		fi
		if [ "$tarid" = "$MYID" ]; then
			rm -f $BINDIR/${WEBSSH_PIDFILE_PREFIX}*${WEBSSH_PIDFILE_POSTFIX}
		else
			sudo -u "$APPUSER" rm -f $BINDIR/${WEBSSH_PIDFILE_PREFIX}*${WEBSSH_PIDFILE_POSTFIX}
		fi
	else
		pidtokill=$(<"$BINDIR/$WEBSSH_PIDFILE_PREFIX$serialno$WEBSSH_PIDFILE_POSTFIX")
		if [[ -z "$pidtokill" || "$pidtokill" != "${pidtokill//[^0-9]}" ]]; then
			[ -n "$silent" ] || echo "$FUNCNAME: PID for ssh tunnel $serialno is not known or invalid. Rerun with 0 to kill all tunnel web gws.">&2
			return 1
		else
			if ps -p "$pidtokill"; then
				kill "$pidtokill" || kill -9 "$pidtokill"
				if ps -p "$pidtokill"; then
					echo "$FUNCNAME: Unable to kill the bugger (PID is $pidtokill)">&2
					return 1
				else
					echo "$FUNCNAME: process $pidtokill was killed."
					if [ "$tarid" = "$MYID" ] ; then
						rm -f "$BINDIR/${WEBSSH_PIDFILE_PREFIX}$serialno${WEBSSH_PIDFILE_POSTFIX}"
					else
						sudo -u "$APPUSER" rm -f "$BINDIR/${WEBSSH_PIDFILE_PREFIX}$serialno${WEBSSH_PIDFILE_POSTFIX}"
					fi
				fi
			else
				[ -n "$silent" ] || echo "$FUNCNAME: SSH tunnel gw with serialno $serialno was not running."
			fi
		fi
	fi
}

start_webssh() {
	local lisport locport serialno userpwd
	[ $# -ne 3 ] && {
		echo "$FUNCNAME: invalid number of arguments (need: locport, serialno, userpwd)."
		return 1
	}
	serialno=$1
	locport=$2
	userpwd=$3
	[[ "$locport" = "${locport//[^0-9]}" && "$serialno" = "${serialno//[^0-9]}" ]] || {
		echo "$FUNCNAME: local port ($locport), serialno ($serialno) must all be numbers." >&2
		return 1
	}
	[ -n "$userpwd" ] || { echo "$FUNCNAME: web ssh user password ($userpwd) must be non-empty." >&2; return 1; }
	[[ "$serialno" -gt 0 && $serialno -lt 10 ]] || { echo "$FUNCNAME: serialno ($serialno) must be between 1 and 9."; return 1; }
	lisport=$((WEBSSH_PORT-1+serialno))
	stop_webssh "$serialno"
	[ -f "$SSHWIFTY_TEMPLATE" ] || {
		echo "$FUNCNAME: sshwifty config template ($SSHWIFTY_TEMPLATE) is missing."
		return 1
	}
	cat "$SSHWIFTY_TEMPLATE" | sed "s/SWGW_PASSWORD/$userpwd/g" | sed "s/SWGW_SERVPORT/$lisport/g" | sed "s/SWGW_LOCPORT/$locport/g" | sed "s/SERIALNO/$serialno/g" >"$BINDIR/sshwifty_gw_$serialno.conf.json"
	chown "$APPUSER" "$BINDIR/sshwifty_gw_$serialno.conf.json"
	tarid=$(id -u "$APPUSER")
	if [ "$tarid" = "$MYID" ]; then
		SSHWIFTY_CONFIG="$BINDIR/sshwifty_gw_$serialno.conf.json" nohup "$BINDIR/sshwifty" &
	else
		sudo -u "$APPUSER" SSHWIFTY_CONFIG="$BINDIR/sshwifty_gw_$serialno.conf.json" nohup "$BINDIR/sshwifty" &
	fi
	echo $! >"$BINDIR/$WEBSSH_PIDFILE_PREFIX$serialno$WEBSSH_PIDFILE_POSTFIX"
	#nohup sudo -u $APPUSER $DWARFG_HOME/.local/bin/wssh --address=127.0.0.1 --fbidhttp=False --policy=autoadd --port=8483 --redirect=False --timeout=20 --encoding='utf-8' &
}


check_running() {
	# check if daemon is running, setting g_runpid global variable to match the pid (0 means it is not running)
	local l_otherpid
	l_otherpid=0
	g_runpid=0
	if [[ -n "$g_loopin_systemd" ]] ; then
		g_runpid=$(systemctl show --property MainPID --value "$DWARFG_NAM")
	else
		if [ -f "$g_fn_pidfile" ] ; then
			read -r input <"$g_fn_pidfile"
			case $input in
				'') # empty line
					fail_nof "Pidfile $g_fn_pidfile does not contain $DGDAEMON PID."
					;;
				*[!0-9]*) # contains a non-number character
					fail_nof "Pidfile $g_fn_pidfile contains non-number characters."
					;;
				*) # otherwise it is a number
					l_otherpid=$input
			esac
		fi
		if [ 0 -ne "$l_otherpid" ] ; then
			if [ -d "/proc/$l_otherpid" ] ; then
				g_runpid=$input
				return 0
			else
				echo "0" >"$g_fn_pidfile" # correct the value in pidfile (no longer running)
				fail_nof "$DGDAEMON crashed/was killed"
			fi
		fi
	fi
}

dirok_or_die() {
	[ $# -eq 1 ] || fail_fat "Internal call error: dirok_or_die() must have 1 parameter."
	if [ ! -d "$1" ] ; then
		mkdir "$1" || fail_fat "Unable to create directory $1"
	fi
}

eval_mode() {
	[[ -n "$g_systemd_present" && -z "$g_from_systemd" && 0 -eq "$g_force" ]] && {
		# not started from systemd but systemd is present and caller does not force us
		g_loopin_systemd="yes"
	}
}


# are we run from systemd?
g_from_systemd=${FROM_SYSTEMD:-}
# check if root
MYID=$(id -u)
if [[ $MYID -ne 0 && "$1" != "$ARG_PID" ]] ; then
	if [ "$ARG_NOROOT" == "$1" ]; then
		shift
	else
		fail_fat "$NAME ctl script must be run under root ($DGDAEMON daemon then runs under $APPUSER)."
	fi
fi
# check if on systemd-enabled system
case "$(readlink -f /proc/1/exe)" in
	*systemd*)
		g_systemd_present="yes"
		;;
	*)
		g_systemd_present=""
		;;
esac

myoldpwd="$(pwd)"
while [ $# -gt 0 ] ; do
	case $1 in
		"$ARG_PID")
			eval_mode
			check_running
			echo "$g_runpid"
			exit 0
			;;
		"$ARG_STATUS")
			eval_mode
			check_running
			if [ 0 -eq "$g_runpid" ] ; then
				echo "$DGDAEMON is *NOT* running"
			else
				echo "$DGDAEMON is running with PID $g_runpid"
			fi
			exit 0
			;;
		"$ARG_START")
			# check if not running already!
			eval_mode
			check_running
			if [ 0 -ne "$g_runpid" ] ; then
				fail_fat "$DGDAEMON already running with PID $g_runpid. Bailing out. Stop $DGDAEMON using $ARG_STOP first."
			fi
			if [ -n "$g_loopin_systemd" ] ; then
				echo "Loopin' systemd..."
				systemctl start "$DWARFG_NAM"
				systemctl start "${DWARFG_NAM}_${SNMP_GW}"
			elif [ -n "$g_from_systemd" ] ; then
				[ ! -f "$BINDIR/user.ini" ] && {
					touch "$BINDIR/user.ini"
					chown "$APPUSER" "$BINDIR/user.ini"
				}
				# started from systemd - daemonized already, exec on foreground
				echo "Executing daemon..."
				cd "$BINDIR" && exec setpriv --reuid "$APPUSER" --regid "$APPGROUP" --init-groups "$BINDIR/$DGDAEMON" $ARGS_EXTRA
				exit $? # never get here
			else
				# classic startup
				echo "Non-systemd startup"
				if [ ! -f "$g_fn_pidfile" ] ; then
					touch "$g_fn_pidfile"
				fi
				chown "$APPUSER" "$g_fn_pidfile"
				if [ -n "$g_vg" ] ; then
					# start inside valgrind (DEBUG) - do not start webssh
					echo "NOTE that valgrind log is stored in $g_ctxt_vallog"
					cd "$BINDIR" && setpriv --reuid "$APPUSER" --regid "$APPGROUP" --init-groups valgrind --log-file="$g_ctxt_vallog" --leak-check=full "$BINDIR/$DGDAEMON" $ARGS_EXTRA || fail_fat "Unable to start $NAME daemon: $BINDIR/$DGDAEMON"
					cd "$myoldpwd"
				elif [ -n "$g_fg" ] ; then
					# start on foreground (DEBUG)
					cd "$BINDIR" && setpriv --reuid "$APPUSER" --regid "$APPGROUP" --init-groups "$BINDIR/$DGDAEMON" $ARGS_EXTRA || fail_fat "Unable to start $NAME daemon: $BINDIR/$DGDAEMON"
					cd "$myoldpwd"
				else
					# daemonize via nohup+BG (regular when not on systemd)
					cd "$BINDIR" && touch "$LOGDIR/$NHLOG" && chown "$APPUSER" "$LOGDIR/$NHLOG"
					nohup sudo -u "$APPUSER" "$BINDIR/$DGDAEMON" $ARGS_EXTRA >>"$LOGDIR/$NHLOG" &
					cd "$myoldpwd"
					# give Dwarfg time to write out PID correctly
					sleep "$g_cnum_slackpid"
					check_running
					if [ "$g_runpid" -eq 0 ] ; then
						fail_fat "Unable to start $NAME daemon: $BINDIR/$DGDAEMON"
					else
						echo "Start operation finished."
					fi
				fi
			fi
			exit 0
			;;
		"$ARG_FG")
			g_force=1
			g_fg="yes"
			;;
		"$ARG_VG")
			g_force=1
			g_fg="yes"
			g_vg="yes"
			;;
		"$ARG_STOP")
			stop_webssh 0
			eval_mode
			if [ -n "$g_loopin_systemd" ] ; then
				echo "Loopin' systemd..."
				systemctl stop "${DWARFG_NAM}_${SNMP_GW}"
				systemctl stop "$DWARFG_NAM"
			elif [ -n "$g_from_systemd" ] ; then
				echo "Called with $ARG_STOP argument from systemd, which is invalid, bailing out."
			else
				check_running
				if [ 0 -eq "$g_runpid" ] ; then
					echo "$DGDAEMON is not running."
					exit 0
				else
					# stop daemon outside systemd
					echo "Stopping $DGDAEMON ..."
					kill "$g_runpid"
					WCOUNT=0
					ORUNPID=$g_runpid
					sleep "$g_cnum_wdelay"
					check_running
					while [[ "$g_runpid" -eq "$ORUNPID" && "$WCOUNT" -lt "$g_cnum_maxwcount" ]] ; do
						WCOUNT=$((WCOUNT+1))
						echo "$DGDAEMON is still running, waiting loop ($WCOUNT/$g_cnum_maxwcount)"
						sleep "$g_cnum_wdelay"
						check_running
					done
					if [ 0 -ne "$g_runpid" ] ; then
						echo "Force-kill $DGDAEMON ..."
						kill -9 "$g_runpid"
						echo "0" >"$g_fn_pidfile" # make sure PIDFILE contains 0 now
					else
						echo "$DGDAEMON process terminated properly"
					fi
					echo "Stop operation finished."
					event_log 303 MSG="Intended $NAME stop was performed."
				fi
			fi
			exit 0
			;;
		"$ARG_FORCE")
			echo "Using the force..."
			g_force=1
			;;
		"$ARG_CFGRELOAD")
			check_running
			if [ 0 -ne "$g_runpid" ] ; then
				echo "Signalling config reload to $NAME daemon ($DGDAEMON)..."
				kill -s SIGUSR1 "$g_runpid"
				sleep 1
			else
				fail_fat "$DGDAEMON is not running, ignoring request to reload config."
			fi
			exit 0
			;;
		"$ARG_DBGREOPEN")
			check_running
			if [ 0 -ne "$g_runpid" ] ; then
				echo "Signalling config reload to reopen log file..."
				kill -s SIGHUP "$g_runpid"
				sleep 1
			else
				fail_fat "$DGDAEMON is not running, ignoring request to reopen debug."
			fi
			exit 0
			;;
		"$ARG_STARTWEBSSH")
			if [ $# -ge 4 ]; then
				start_webssh "$2" "$3" "$4"
				shift 3
			else
				echo "Insufficient number of arguments for $1..."
			fi
			;;
		"$ARG_STOPWEBSSH")
			if [ $# -gt 1 ]; then
				stop_webssh "$2"
				shift
			else
				stop_webssh 0
			fi
			;;
		*)
			print_help
			exit 0
			;;
	esac
	shift
done
