<?php

namespace App\DataTable;

use App\Bundles\Sg\DatatablesBundle\Datatable\Column\CustomLinkColumn;
use App\Entity\Snmp\Device;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Column\Column;
use Sg\DatatablesBundle\Datatable\Column\DateTimeColumn;
use Sg\DatatablesBundle\Datatable\Filter\DateRangeFilter;
use Sg\DatatablesBundle\Datatable\Filter\NumberFilter;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;

/**
 * Class SnmpDatatable
 * @package App\Datatables
 */
class SnmpDatatable extends BaseDatatable
{

    protected string $entity = Device::class;
    protected string $name = 'snmp_device_datatable';

    /**
     * @param array $options
     * @throws \Exception
     */
    public function buildDatatable(array $options = []): void
    {

        parent::buildDatatable($options);

        $this->extensions->set([
            'buttons' => [
                'create_buttons' => [  // custom buttons
                    [
                        'action' => [
                            'template' => '/datatable/snmp/addSnmpDevice.js.twig',
                        ],
                        'class_name' => 'btn-add',
                        'name' => 'addLabel',
                        'text' => '<i class=\"fa fa-plus\"></i> Add new SNMP Device',
                    ],
                    $this->getRefreshButton()
                ]
            ]
        ]);

        $this->addIdColumn();

        $this->columnBuilder
            ->add('address', Column::class, array(
                'title' => 'Address',
                'width' => '80%;',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                ),

            ))

            ->add('communityString', Column::class, array(
                'title' => 'Community String',
                'width' => '80%;',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                ),

            ))

            ->add('protocolVersion', Column::class, array(
                'title' => 'Protocol Version',
                'width' => '80%;',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                ),

            ))

            ->add('lastSuccess', DateTimeColumn::class, array(
                'title' => 'Last Success',
                'searchable' => true,
                'filter' => array(DateRangeFilter::class, array(
                    'cancel_button' => true,
                )),
                'date_format' => self::DATETIME_FORMAT,
            ))

            ->add('lastFailure', DateTimeColumn::class, array(
                'title' => 'Last Failure',
                'searchable' => true,
                'filter' => array(DateRangeFilter::class, array(
                    'cancel_button' => true,
                )),
                'date_format' => self::DATETIME_FORMAT,
            ))

            ->add('errorsSinceSuccess', Column::class, array(
                'title' => 'Errors Since Success',
                'width' => '80%;',
                'filter' => array(NumberFilter::class,
                    array(
                        'min' => '0',
                        'cancel_button' => true,
                    ),
                ),

            ))

            ->add('registeredAsDevId', CustomLinkColumn::class, array(
                'title' => 'Device',
                'width' => '80%;',
                'route' => 'admin_devices_detail',
                'route_params' => function($row) {
                    if(isset($row['registeredAs'])){
                        return array(
                            'deviceId' => $row['registeredAs'],
                        );
                    }
                    return [];
                },
                'filter' => array(NumberFilter::class,
                    array(
                        'min' => '0',
                        'cancel_button' => true,
                    ),
                ),

            ))

            //used by link to device
            ->add('registeredAs', Column::class, [
                'visible' => false
            ])

            ->add(null, ActionColumn::class, options: [
                'title' => $this->translator->trans('sg.datatables.actions.title'),
                'width' => '100%;',
                'actions' => [
                    [
                        'route' => 'ajax_snmp_device_delete',
                        'route_parameters' => [
                            'deviceId' => 'id',
                            ],
                        'label' => $this->translator->trans('sg.datatables.actions.delete'),
                        'icon' => self::ICON_DELETE,
                        'attributes' => [
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.delete'),
                            'class' => 'btn btn-danger btn-xs mb-1',
                            'role' => 'button',
                            'data-toggle' => 'modal',
                            'id' => 'delete',
                        ],
                        'render_if' => function ($row) {
                            return $this->authorizationChecker->isGranted('ROLE_ADMIN');
                        },
                        'end_html' => '</div>',
                    ],
                ],
            ]);

    }

}