<?php

namespace App\DataTable;

use App\DataTable\Column\CustomDateTimeColumn;
use App\Entity\Tunnel\TunnelOperation;
use App\Entity\Tunnel\TunnelOperationStatusMapping;
use App\Entity\Tunnel\TunnelStatus;
use App\Entity\Tunnel\Tunnels;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Column\BooleanColumn;
use Sg\DatatablesBundle\Datatable\Column\Column;
use Sg\DatatablesBundle\Datatable\Filter\DateRangeFilter;
use Sg\DatatablesBundle\Datatable\Filter\SelectFilter;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;

/**
 * Class TunnelDatatable
 * @package App\Datatables
 */
class TunnelDatatable extends BaseDatatable
{

    protected string $entity = Tunnels::class;
    protected string $name = 'tunnels_datatable';

    /**
     * @param array $options
     * @throws \Exception
     */
    public function buildDatatable(array $options = array()): void
    {

        $tunnelStatuses = $options['tunnelStatuses'];
        $tunnelOperations = $options['tunnelOperations'];
        $isPythonComponentsInstalled = $options['isPythonComponentsInstalled'];

        parent::buildDatatable($options);

        $this->extensions->set(array(
            'buttons' => array(
                'create_buttons' => array(
                    $this->getRefreshButton(),
                    $this->getAutoRefreshButton()
                ),
            )
        ));

        $this->addIdColumn();

        $this->columnBuilder
            ->add('device.devId', Column::class, array(
                'title' => 'Device',
                'width' => '100%;',
                'default_content' => 'Not Assigned',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                ),
            ))
            ->add('device.name', Column::class, array(
                'title' => 'Name',
                'width' => '100%;',
                'default_content' => 'None',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                )))
            ->add('url', Column::class, array(
                'visible' => false,
                'title' => 'Url',
                'width' => '100%;',
                'default_content' => 'N/A',
                'filter' => array(TextFilter::class,
                    array(
                        'cancel_button' => true,
                    ),
                ),
            ))
            ->add('status.name', Column::class, array(
                'title' => 'Status',
                'width' => '100%;',
                'filter' => array(SelectFilter::class, array(
                    'classes' => '',
                    'search_type' => 'eq',
                    'multiple' => false,
                    'select_options' => array('' => 'Any') + $this->getOptionsArrayFromEntities($tunnelStatuses, 'name', 'name'),
                    'cancel_button' => true,
                ))
            ))
            ->add('requestedOperation.name', Column::class, array(
                'title' => 'Operation',
                'width' => '100%;',
                'filter' => array(SelectFilter::class, array(
                    'classes' => '',
                    'search_type' => 'eq',
                    'multiple' => false,
                    'select_options' => array('' => 'Any') + $this->getOptionsArrayFromEntities($tunnelOperations, 'name', 'name'),
                    'cancel_button' => true,
                ))
            ))
            ->add('expectedTime', CustomDateTimeColumn::class, array(
                'title' => 'Waiting Time',
                'searchable' => true,
                'filter' => array(DateRangeFilter::class, array(
                    'cancel_button' => true,
                )),
            ))
            ->add('ssh', BooleanColumn::class, array(
                'title' => 'Type',
                'width' => '100%;',
                'true_label' => 'SSH',
                'false_label' => 'Web',
                //'false_icon' => 'fa fa-circle yellow',
                'filter' => array(SelectFilter::class,
                    array(
                        'search_type' => 'eq',
                        'select_options' => array(
                            '' => 'Any',
                            '1' => 'SSH',
                            '0' => 'Web',
                        ),
                        'cancel_button' => true,
                    )
                ),
            ))

            ->add(null, ActionColumn::class, array(
                'title' => $this->translator->trans('sg.datatables.actions'),
                'width' => '100%;',
                'class_name' => 'row_web_tunnel_actions',
                'actions' => array(
                    array(
                        'button' => true,
                        'button_value' => 'id',
                        'label' => $this->translator->trans('sg.datatables.actions.tunnel_setup'),
                        'icon' => 'fas fa-cog',
                        'attributes' => array(
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.tunnel_setup'),
                            'class' => 'btn btn-success btn-xs mb-1',
                            'role' => 'button',
                            'data-toggle' => 'modal',
                            'id' => 'assign',
                            'data-target' => "#assign-device",
                        ),
                        'render_if' => function ($row) {

                            $allowedOperations = $this->getAllowedOperations($row);
                            $allowed = false;

                            if($row['requestedOperation']['name'] === TunnelOperation::OPERATION_NONE &&
                                in_array(TunnelOperation::OPERATION_SETUP, $allowedOperations)){
                                $allowed = true;
                            }

                            return $allowed && $this->authorizationChecker->isGranted('ROLE_ADMIN');
                        },
                    ),
                    array(
                        'route' => 'admin_tunnel_connect',
                        'route_parameters' => array(
                            'tunnelId' => 'id',

                        ),
                        'label' => $this->translator->trans('sg.datatables.actions.tunnel_connect'),
                        'icon' => 'fa fa-plug',
                        'attributes' => array(
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.tunnel_connect'),
                            'class' => 'btn btn-primary btn-xs mb-1 mr-sm-2',
                            'role' => 'button',
                            'target' => '_blank',
                        ),
                        'render_if' => function ($row) use ($isPythonComponentsInstalled) {

                            $statusId = $row['status']['id'];
                            $url = $row['url'];

                            $allowed = false;
                            if($statusId === 2 && $url !== null && $isPythonComponentsInstalled){ //Status Ready

                                $allowed = true;

                            }

                            return $allowed && $row['requestedOperation']['name'] === TunnelOperation::OPERATION_NONE &&
                                $this->authorizationChecker->isGranted('ROLE_ADMIN');
                        },
                    ),
                    array(
                        'route' => 'ajax_tunnel_restart',
                        'route_parameters' => array(
                            'tunnelId' => 'id',
                        ),
                        'label' => $this->translator->trans('sg.datatables.actions.tunnel_restart'),
                        'icon' => 'fas fa-sync',
                        'attributes' => array(
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.tunnel_restart'),
                            'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                            'role' => 'button',
                            'data-toggle' => 'modal',
                            'id' => 'restart',
                            'data-target' => "#confirm-restart",
                        ),
                        'render_if' => function ($row) {

                            $allowedOperations = $this->getAllowedOperations($row);
                            $allowed = false;

                            if($row['requestedOperation']['name'] === TunnelOperation::OPERATION_NONE &&
                                in_array(TunnelOperation::OPERATION_RESTART, $allowedOperations)){
                                $allowed = true;
                            }

                            return $allowed && $this->authorizationChecker->isGranted('ROLE_ADMIN');
                        },
                    ),
                    array(
                        'route' => 'ajax_tunnel_shutdown',
                        'route_parameters' => array(
                            'tunnelId' => 'id',
                        ),
                        'label' => $this->translator->trans('sg.datatables.actions.tunnel_shutdown'),
                        'icon' => 'fas fa-window-close',
                        'attributes' => array(
                            'rel' => 'tooltip',
                            'title' => $this->translator->trans('sg.datatables.actions.tunnel_shutdown'),
                            'class' => 'btn btn-danger btn-xs mb-1 mr-sm-2',
                            'role' => 'button',
                            'data-toggle' => 'modal',
                            'id' => 'shutdown',
                            'data-target' => "#confirm-shutdown",
                        ),
                        'render_if' => function ($row) {

                            $allowedOperations = $this->getAllowedOperations($row);
                            $allowed = false;

                            if($row['requestedOperation']['name'] === TunnelOperation::OPERATION_NONE &&
                                in_array(TunnelOperation::OPERATION_SHUTDOWN, $allowedOperations)){
                                $allowed = true;
                            }

                            return $allowed && $this->authorizationChecker->isGranted('ROLE_ADMIN');
                        },
                    ),

                ),
            ))
        ;
    }

    /**
     * @param array $row
     * @return array
     */
    private function getAllowedOperations(array $row): array {

        $statusId = $row['status']['id'];
        $tunnelStatusRepository = $this->em->getRepository(TunnelStatus::class);
        $status = $tunnelStatusRepository->findOneBy(['id' => $statusId]);
        $tunnelOperationStatusMappingRepository = $this->em->getRepository(TunnelOperationStatusMapping::class);

        $mappings = $tunnelOperationStatusMappingRepository->findBy(['status' => $status]);

        $allowedOperations = [];

        foreach ($mappings as $mapping){

            $allowedOperations[] = $mapping->getOperation()->getName();

        }

        return $allowedOperations;

    }

}