<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\AgentProfile\AgentProfile;
use App\Entity\ConfigProfile\ConfigProfile;
use App\Entity\Device\Device;
use App\Entity\Device\DeviceOsVersion;
use App\Repository\ConfigProfileRepository;
use App\Repository\DeviceRepository;
use App\Service\Api\AgentProfiles;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\ParameterType;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use App\Service\Api\Device as DeviceApi;

#[Route(path: '/admin/ajax/device', name: 'ajax_device_')]
class DeviceController extends Controller
{

    /**
     * @param Request $request
     * @param DeviceRepository $deviceRepository
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/not-assigned-tunnel', name: 'get_not_tunnel', methods: ['GET'])]
    public function getDevicesNotAssignedTunnel(Request $request, DeviceRepository $deviceRepository): Response {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->findAllNotAssignedToWebTunnelPlainSql();
            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new Response('Bad request.', 400);

    }

    /**
     * @param Request $request
     * @param DeviceRepository $deviceRepository
     * @param int $mntGroupId
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/get/all-not/{mntGroupId}', name: 'get_all_not_in_monitoring_group', methods: ['GET'])]
    public function getAllDevicesNotInMonitoringGroup(Request          $request,
                                                      DeviceRepository $deviceRepository,
                                                      int $mntGroupId): Response {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->findAllNotInMonitoringGroupPlainSql($mntGroupId);

            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new Response('Bad request.', 400);

    }

	/**
	 * @throws Exception
	 */
	#[Route(path: '/get/all-not-config-profile-assigned/{configProfileId}', name: 'get_all_not_config_profile_assigned', methods: ['GET'])]
	public function getAllNotConfigProfileAssigned(Request          $request,
													  DeviceRepository $deviceRepository,
													  int $configProfileId): Response {

		if ($request->isXmlHttpRequest()) {

			$devices = $deviceRepository->findAllNotConfigProfileAssignedPlainSql($configProfileId);

			$result = [];

			foreach ($devices as $device) {
				$result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
			}

			return new JsonResponse($result);
		}

		return new Response('Bad request.', 400);

	}

	/**
	 * @throws Exception
	 */
	#[Route(path: '/get/all-with-configuration-profile', name: 'get_all_with_config_profile', methods: ['GET'])]
	public function getAllDevicesWithAutomaticProfile(Request          $request,
												   DeviceRepository $deviceRepository): Response {

		if ($request->isXmlHttpRequest()) {

			$devices = $deviceRepository->getDevicesWithAutomaticProfile();

			$result = [];

			foreach ($devices as $device) {
				$result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
			}

			return new JsonResponse($result);
		}

		return new Response('Bad request.', 400);

	}

	#[Route(path: '/get/all-to-copy-configuration-profile', name: 'get_all_to_copy_config_profile', methods: ['GET'])]
	public function getAllDevicesToCopyProfile(Request          $request,
													  ConfigProfileRepository $configProfileRepository): Response {

		if ($request->isXmlHttpRequest()) {

			$profiles = $configProfileRepository->getProfilesToMakeCopy();

			$result = [];

			foreach ($profiles as $profile) {
				$result[$profile['id']] = ['value' => $profile['name']/*, 'description' => $profile['name']*/];
			}

			return new JsonResponse($result);
		}

		return new Response('Bad request.', 400);

	}

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/add/monitoring-group', name: 'add_to_monitoring_group', methods: ['POST'])]
    public function addDevicesToMonitoringGroupAction(Request $request, ManagerRegistry $managerRegistry): Response
    {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $monitoringGroups = (array)$request->get('monitoringGroups');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($monitoringGroups) || !isset($selectedDevices)) {
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();
            $connection->beginTransaction();

			if(empty($monitoringGroups)){
				return new JsonResponse(['status' => 'Error', 'message' => 'No monitoring group given.'],400);
			}

            foreach ($monitoringGroups as $monitoringGroupId) {

                $placeholders = [];
                $values = [];
                $types = [];

                foreach ($deviceIds as $deviceId) {

                    $placeholders[] = '(?)';
                    $values[] = [$monitoringGroupId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;

                }

                try{
                    $connection->executeStatement(
                        'INSERT IGNORE `monitoring_group_device_map` (`mongrp_id`, `id_device`)  VALUES ' .
                        implode(', ', $placeholders),
                        $values,
                        $types
                    );
                }catch (\Exception $e){

                    $connection->rollBack();

                    return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                        400);

                }

            }

            $connection->commit();

            return new JsonResponse(['code' => 200, 'message' => 'Devices were successfully assigned to Monitoring Groups.', 'errors' => ['errors' => ['']]],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Add Labels to Devices
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/add/label', name: 'add_label', methods: ['POST'])]
    public function addLabelToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $labelIds = (array)$request->get('labels');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($labelIds) || !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            //$labels = $managerRegistry->getRepository(Label::class)->findBy(['id' => $labelIds]);
            //$deviceRepository = $managerRegistry->getRepository(Device::class);

            $placeholders = [];
            $values = [];
            $types = [];

            foreach ($deviceIds as $deviceId) {

                foreach ($labelIds as $labelId) {
                    $placeholders[] = '(?)';
                    $values[] = [$labelId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;
                }

                /*$device = $deviceRepository->findOneBy(['id' => $device['id']]);

                foreach ($labels as $label) {

                    $existingLabel = $managerRegistry->getRepository(LabelDeviceMap::class)
                        ->findOneBy(['device' => $device, 'label' => $label]);

                    if(!$existingLabel){

                        try {

                            $labelDeviceMapping = new LabelDeviceMap();
                            $labelDeviceMapping->setLabel($label);
                            $labelDeviceMapping->setDevice($device);
                            $managerRegistry->getManager()->persist($labelDeviceMapping);
                            $managerRegistry->getManager()->flush();

                        } catch (UniqueConstraintViolationException) {

                            //Do nothing, label already assigned

                        } catch (\Exception) {

                        }

                    }

                }*/

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->executeStatement(
                    'INSERT IGNORE `labels_map` (`id_label`, `id_device`)  VALUES ' .
                    implode(', ', $placeholders),
                    $values,
                    $types
                );

            }catch (\Exception $e){

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);

            }

            return new JsonResponse(['code' => "200", 'message' => 'Custom Label was successfully assigned to the devices.', 'errors' => ['errors' => ['']]],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

	/**
	 * Add System Labels to Devices
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @return Response
	 */
	#[Route(path: '/add/system-label', name: 'add_system_label', methods: ['POST'])]
	public function addSystemLabelToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

		if (!$request->isXmlHttpRequest()) {

			return new Response('Bad request.', 400);

		}

		if (isset($request->request)) {

			$labelIds = (array)$request->get('labels');
			$selectedDevices = $request->get('selectedDevices');

			if (!isset($labelIds) || !isset($selectedDevices))
				return new JsonResponse(['status' => 'Error'], 400);

			$deviceIds = [];

			foreach ($selectedDevices as $device) {
				$deviceIds[] = $device['id'];
			}

			$placeholders = [];
			$values = [];
			$types = [];

			foreach ($deviceIds as $deviceId) {

				foreach ($labelIds as $labelId) {
					$placeholders[] = '(?)';
					$values[] = [$labelId, $deviceId];
					$types[] = ArrayParameterType::INTEGER;
				}

			}

			/** @var Connection $connection */
			$connection =  $managerRegistry->getConnection();

			try{

				$connection->executeStatement(
					'INSERT IGNORE `system_labels_map` (`id_system_label`, `id_device`)  VALUES ' .
					implode(', ', $placeholders),
					$values,
					$types
				);

			}catch (\Exception $e){

				dump($e->getMessage());

				return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
					400);

			}

			return new JsonResponse(['code' => "200", 'message' => 'System Label was successfully assigned to the devices.', 'errors' => ['errors' => ['']]],
				200);

		}

		// If we reach this point, it means that something went wrong
		return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
			400);

	}

	/**
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @return Response
	 * @throws Exception
	 */
    #[Route(path: '/add/fw', name: 'add_fw', methods: ['POST'])]
    public function addFwToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $fws = (array)$request->get('fws');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($fws) || !isset($selectedDevices)) {
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $fws = $managerRegistry->getRepository(DeviceOsVersion::class)->findBy(['id' => $fws]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

			if (!$fws){
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $errors = [];
            $success = [];
            $values = [];

            foreach ($selectedDevices as $device) {

                $device = $deviceRepository->findOneBy(['id' => $device['id']]);

                $deviceType = $device->getDevType();

                foreach ($fws as $fw) {

                    $fwDeviceOsType = $fw->getOsType()->getDevType();

                    //Check if there is the same OS type on device as selected FW
                    if ($deviceType === $fwDeviceOsType) {

                        $values[] = $device->getId();
                        $success[] = $device->getDevId();

                    } else {

                        $errors[] = $device->getDevId();

                    }

                }

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$values], [ArrayParameterType::INTEGER]);

                $connection->executeStatement(
                    'UPDATE `devices` SET `os_ver_desired` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [$fw->getId(), $globalVersion + 1,  $values],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Fw version successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param AgentProfiles $agentProfilesApi
     * @return Response
     */
    #[Route(path: '/add/agent-profile', name: 'add_agent_profile', methods: ['POST'])]
    public function addAgentProfileToDeviceAction(Request $request, ManagerRegistry $managerRegistry,
                                                  AgentProfiles $agentProfilesApi): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $agentProfiles = (array)$request->get('agentProfiles');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($agentProfiles) && !isset($selectedDevices)) {
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findBy(['id' => $agentProfiles]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

			if (!$agentProfiles){
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $errors = [];
            $success = [];

            foreach ($agentProfiles as $agentProfile) {

                $deviceIds = [];
				$deviceDevIds = [];
                $agentProfileId = $agentProfile->getId();

                foreach ($selectedDevices as $device) {

                    $device = $deviceRepository->findOneBy(['id' => $device['id']]);
					if(!$device){
						$deviceIds = [];
						$errors[] = 'Invalid device given.';
					}
                    $deviceIds[] = $device->getId();
                    $deviceDevIds[] = $device->getDevId();

                }

				if(!empty($deviceIds)){

					try {

						$result = $agentProfilesApi->assignDeviceAgentProfile($agentProfileId, $deviceIds);
						if($result->operation_result->Code < 0){
							throw new \Exception('Cannot call api');
						}
						/*$device->setAgentProfile($agentProfile);
						$managerRegistry->getManager()->persist($device);
						$managerRegistry->getManager()->flush();*/

						$success[] = implode(',', $deviceDevIds);

					} catch (\Exception $e) {

						$errors[] = $e->getMessage(); //$device->getDevId();

					}

				}

            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Agent Profile was successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * @param Request $request
     * @param DeviceRepository $deviceRepository
     * @param ManagerRegistry $managerRegistry
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/get/config-profiles', name: 'get_config_profiles', methods: ['GET'])]
    public function getConfigProfilesAction(Request $request, DeviceRepository $deviceRepository, ManagerRegistry $managerRegistry): Response
    {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->getDevicesToClonedFrom();

            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/get/agent-profiles', name: 'get_agent_profiles', methods: ['GET'])]
    public function getAgentProfilesAction(Request $request, ManagerRegistry $managerRegistry): Response
    {

        if ($request->isXmlHttpRequest()) {

            $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findAll();

            $result = [];

            foreach ($agentProfiles as $agentProfile) {
                $result[$agentProfile->getId()] = ['value' => $agentProfile->getProfileName(), 'description' => ''];
            }

            return new JsonResponse($result);
        }

        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/add/config-profile', name: 'add_config_profile', methods: ['POST'])]
    public function addConfigProfileToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $devices = (array)$request->get('devices');
            $selectedDevices = $request->get('selectedDevices');
            $overwrite = $request->get('overwrite'); //retype to bool did not work for false

            if (!isset($devices) && !isset($selectedDevices)) {
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $devices = $managerRegistry->getRepository(Device::class)->findBy(['id' => $devices]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

			if (!$devices){
				return new JsonResponse(['status' => 'Error'], 400);
			}

            $errors = [];
            $success = [];

            foreach ($selectedDevices as $device) {

                $device = $deviceRepository->findOneBy(['id' => $device['id']]);

                $deviceType = $device->getDevType();

                foreach ($devices as $deviceCloneFrom) {

                    $deviceTypeCloneFrom = $deviceCloneFrom->getDevType();
                    $deviceConfigProfileFrom = $deviceCloneFrom->getConfigProfileAutomatic();

                    //Check if there is the same OS type on device as selected FW
                    if (($deviceType === $deviceTypeCloneFrom && $device !== $deviceCloneFrom) && (
                        ($device->getConfigProfile() instanceof ConfigProfile && $overwrite == 'true') ||
                        ($device->getConfigProfile() === null)) ) {

                        $values[] = $device->getId();
                        $success[] = $device->getDevId();

                    } else {

                        $errors[] = $device->getDevId();

                    }

                }

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$values], [ArrayParameterType::INTEGER]);

                $connection->executeStatement(
                    'UPDATE `devices` SET `config_profile` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [$deviceConfigProfileFrom->getId(), $globalVersion + 1,  $values],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => $e->getMessage()],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Config Profile was successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

	/**
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @return Response
	 */
	#[Route(path: '/remove/custom-label', name: 'remove_custom_label', methods: ['POST'])]
	public function removeCustomLabelDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response
	{

		if (!$request->isXmlHttpRequest()) {

			return new Response('Bad request.', 400);

		}

		if (isset($request->request)) {

			$labelIds = (array)$request->get('labels');
			$selectedDevices = $request->get('selectedDevices');

			if (!isset($labelIds) || !isset($selectedDevices))
				return new JsonResponse(['status' => 'Error'], 400);

			$errors = [];
			$success = [];

			$deviceIds = [];

			foreach ($selectedDevices as $device) {
				$deviceIds[] = $device['id'];
			}

			$placeholders = [];
			$values = [];
			$types = [];

			foreach ($deviceIds as $deviceId) {

				foreach ($labelIds as $labelId) {
					$placeholders[] = '(id_label = ? AND id_device = ?)';
					$values[] = [$labelId];
					$values[] = [$deviceId];
					$types[] = ArrayParameterType::INTEGER;
					$types[] = ArrayParameterType::INTEGER;
				}
			}

			/** @var Connection $connection */
			$connection =  $managerRegistry->getConnection();

			try{

				$connection->executeStatement(
					'DELETE FROM `labels_map` WHERE ' . implode(' OR ', $placeholders),
					$values,
					$types
				);

			}catch (\Exception $e){

				return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
					400);

			}

			return new JsonResponse(['code' => "200", 'message' => 'Custom Labels was successfully removed from the devices.', 'errors' => ['errors' => ['']]],
				200);

		}

	}

	/**
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @return Response
	 */
	#[Route(path: '/remove/system-label', name: 'remove_system_label', methods: ['POST'])]
	public function removeSystemLabelDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response
	{

		if (!$request->isXmlHttpRequest()) {

			return new Response('Bad request.', 400);

		}

		if (isset($request->request)) {

			$labelIds = (array)$request->get('labels');
			$selectedDevices = $request->get('selectedDevices');

			if (!isset($labelIds) || !isset($selectedDevices))
				return new JsonResponse(['status' => 'Error'], 400);

			$errors = [];
			$success = [];

			$deviceIds = [];

			foreach ($selectedDevices as $device) {
				$deviceIds[] = $device['id'];
			}

			$placeholders = [];
			$values = [];
			$types = [];

			foreach ($deviceIds as $deviceId) {

				foreach ($labelIds as $labelId) {
					$placeholders[] = '(id_system_label = ? AND id_device = ?)';
					$values[] = [$labelId];
					$values[] = [$deviceId];
					$types[] = ArrayParameterType::INTEGER;
					$types[] = ArrayParameterType::INTEGER;
				}
			}

			/** @var Connection $connection */
			$connection =  $managerRegistry->getConnection();

			try{

				$connection->executeStatement(
					'DELETE FROM `system_labels_map` WHERE ' . implode(' OR ', $placeholders),
					$values,
					$types
				);

			}catch (\Exception $e){

				return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
					400);

			}

			return new JsonResponse(['code' => "200", 'message' => 'System Labels was successfully removed from the devices.', 'errors' => ['errors' => ['']]],
				200);

		}

	}

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DeviceRepository $deviceRepository
     * @return Response
     */
    #[Route(path: '/remove/config-profile', name: 'remove_config_profile', methods: ['POST'])]
    public function removeConfigProfileDeviceAction(Request $request, ManagerRegistry $managerRegistry,
                                                    DeviceRepository $deviceRepository): Response
    {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $selectedDevices = $request->get('selectedDevices');

            if (!isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $errors = [];
            $success = [];

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$deviceIds], [ArrayParameterType::INTEGER]);

                $connection->executeStatement(
                    'UPDATE `devices` SET `config_profile` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [null, $globalVersion + 1,  $deviceIds],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Config Profile was successfully removed for selected devices.',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }
    }

    #[Route(path: '/reboot/{deviceId}', name: 'reboot', methods: ['GET'])]
    public function rebootAction(int $deviceId, Request $request, DeviceRepository $deviceRepository,
                                 DeviceApi $deviceApi): Response
    {
        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        $device = $deviceRepository->findBy(['id' => $deviceId]);

        if(!$device){
            return new JsonResponse(['status' => 'Error'], 400);
        }

        try{

            $deviceApi->reboot([$deviceId]);

        }catch (\Exception $e){

            return new JsonResponse(['status' => 'Error', 'message' => $e->getMessage()],
                400);

        }

        return new JsonResponse([
            'code' => 200,
            'message' => 'Device will be rebooted.',
           /* 'success' => $success,
            'errors' => $errors*/
        ],
            200);

    }

}