<?php

namespace App\EventListener;

use App\Entity\Alert\Alert;
use App\Entity\CustomScript\CustomScript;
use App\Entity\Device\Device;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\Tunnel\Tunnels;
use App\Event\AlertCreateEvent;
use App\Event\AlertEditEvent;
use App\Event\CustomScriptCreateEvent;
use App\Event\CustomScriptEditEvent;
use App\Event\DeviceEditEvent;
use App\Event\MonitoringGroupCreateEvent;
use App\Event\WebTunnelEditEvent;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class DoctrineListener
{

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $eventDispatcher;

    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @var bool
     */
    private bool $updatedName = false;

    /**
     * @var bool
     */
    private bool $updateAlert = false;

    /**
     * @var bool
     */
    private bool $updateTunnelWeb = false;
    /**
     * @var true
     */
    private bool $updateCustomScript = false;

    /**
     * @param EventDispatcherInterface $eventDispatcher
     * @param LoggerInterface $logger
     */
    public function __construct(EventDispatcherInterface $eventDispatcher, LoggerInterface $logger)
    {
        $this->eventDispatcher = $eventDispatcher;
        $this->logger = $logger;
    }

    /**
     * @param PreUpdateEventArgs $args
     */
    public function preUpdate(PreUpdateEventArgs $args): void
    {
        $entity = $args->getObject();

        if ($entity instanceof Device && ($args->hasChangedField('hostnameDesired')
                || $args->hasChangedField('osVerDesired') || $args->hasChangedField('configProfile')
                || $args->hasChangedField('deleted')
            )) {
            $this->logger->info('Called event preUpdate Device.');
            $this->updatedName = true;
        }

        if ($entity instanceof Alert && !$args->hasChangedField('version')) {
            $this->logger->info('Called event preUpdate Alert version.');
            $this->updateAlert = true;
        }

        if ($entity instanceof Tunnels && $args->hasChangedField('requestedOperation')) {
            $this->logger->info('Called event preUpdate TunnelWeb requestedOperation.');
            $this->updateTunnelWeb = true;
        }

        if ($entity instanceof CustomScript && !$args->hasChangedField('version')) {
            $this->logger->info('Called event preUpdate CustomScript version.');
            $this->updateCustomScript = true;
        }

    }

    /**
     * @param LifecycleEventArgs $args
     */
    // the listener methods receive an argument which gives you access to
    // both the entity object of the event and the entity manager itself
    public function postUpdate(LifecycleEventArgs $args): void {

        $entity = $args->getObject();

        $this->logger->info('postupdate');

        // if this listener only applies to certain entity types,
        // add some code to check the entity type as early as possible
        if ($entity instanceof Device) {
            $this->logger->info('Called device update event');

            if ($this->updatedName) {
                $this->updatedName = false;

                $event = new DeviceEditEvent($entity->getId());
                $this->eventDispatcher->dispatch($event, DeviceEditEvent::NAME);
            }
        }

        if ($entity instanceof Alert) {
            $this->logger->info('Called alert update event');

            if ($this->updateAlert) {
                $this->updateAlert = false;

                $event = new AlertEditEvent($entity->getId());
                $this->eventDispatcher->dispatch($event, AlertEditEvent::NAME);

            }

        }

        if ($entity instanceof Tunnels) {
            $this->logger->info('Called tunnel update event');

            if ($this->updateTunnelWeb) {
                $this->updateTunnelWeb = false;

                $event = new WebTunnelEditEvent($entity->getId());
                $this->eventDispatcher->dispatch($event, WebTunnelEditEvent::NAME);

            }

        }

        if ($entity instanceof CustomScript) {

            $this->logger->info('Called CustomScript update event');

            if ($this->updateCustomScript) {
                $this->updateCustomScript = false;

                $event = new CustomScriptEditEvent($entity->getId());
                $this->eventDispatcher->dispatch($event, CustomScriptEditEvent::NAME);

            }

        }

    }

    /**
     * @param LifecycleEventArgs $args
     */
    public function postPersist(LifecycleEventArgs $args): void
    {

        $entity = $args->getObject();

        if ($entity instanceof MonitoringGroup) {
            $this->logger->info('Called monitoring group create event');

            $event = new MonitoringGroupCreateEvent($entity->getId());
            $this->eventDispatcher->dispatch($event, MonitoringGroupCreateEvent::NAME);
        }

        if ($entity instanceof Alert) {
            $this->logger->info('Called alert create event');

            $event = new AlertCreateEvent($entity->getId());
            $this->eventDispatcher->dispatch($event, AlertCreateEvent::NAME);
        }

        if ($entity instanceof CustomScript) {
            $this->logger->info('Called Custom script create event');

            $event = new CustomScriptCreateEvent($entity->getId());
            $this->eventDispatcher->dispatch($event, CustomScriptCreateEvent::NAME);
        }

    }

}