#!/bin/bash
# Dwarfg upgrade script
# Copyright (c) Dwarf Technologies, Jan Otte
# NOTE: while the file (itself) is copyrighted, you are invited to use the stuff you learn here anywhere you need

FUNCS="deploy_funcs.sh"

command -v dirname >/dev/null 2>&1 || {
	echo "\"dirname\" command not available, bailing out." >&2
	exit 1
}

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR"/"$FUNCS" || {
	echo "Unable to read deploy functions, exiting..." >&2
	exit 1
}

BASEDEFS="base_defs"

. "$DIR"/"$BASEDEFS" || { echo "Fatal: Definition file ($BASEDEFS) not found." >&2; exit 1; }
. "$DIR"/"$BASEFUNCS" || { echo "Fatal: Basic functions ($BASEFUNCS) not found." >&2; exit 1; }

validate_deploy_commands || fail_fat "Some of required commands cannot be found."

BACKUP_FINISHED="backup_finished"
BACKUP_STARTED="backup_started"
NOBACKUP=""
DBDUMP_FN="${SHORTNAME}_db_dump.sql"

# directories and files (re-evaluated...)
UPG_BACKUP="$DWARFG_BACKUP/$DWARFG_DBN%$VERSION" # NOTE: constructed on 2 places + ROTDIR
UPG_BACKUP_DATA="$UPG_BACKUP/data"
UPG_BACKUP_SRV="$UPG_BACKUP/srv"
UPG_BACKUP_DB="$UPG_BACKUP/dbdump"

print_help() {
	echo
	echo "$NAME upgrade script"
	echo -e "\tExpected mode of operation is that you run this script from the\n\texisting $NAME deployment (e.g. from the OLD package),\n\tproviding path to the new package or to a directory.\n\n\tSupported parameters:\n\n\t\t$PARAM_NOBACKUP ... process without backing up existing $NAME deployment (dangerous)\n"
}

select_version() {
	local ALLVERS
	local ALLVERS_VERSONLY
	local ALLVERS_CNT
	ALLVERS=$(ls "$DWARFG_BACKUP")
	ALLVERS_VERSONLY=$(echo -e "$ALLVERS" | $C_GREP "^[0-9]*.[0-9]*.[0-9]*$")
	ALLVERS_CNT=$(echo -e "$ALLVERS" | $C_GREP -c "^")
	ALLVERS_VERSONLY_CNT=$(echo -e "$ALLVERS_VERSONLY" | $C_GREP -c "^")
	# TODO
}

# tries to locate base_defs file, source it and update backup definitions
# requires one parameter - directory to look the base_defs at
source_and_evaluate() {
	[ $# -eq 1 ] || return 1
	[ -f "$1/$BASEDEFS" ] || return 1
	. "$1/$BASEDEFS"
	UPG_BACKUP="$DWARFG_BACKUP/$DWARFG_DBN%$VERSION" # NOTE: constructed on 2 places + ROTDIR
	UPG_BACKUP_DATA="$UPG_BACKUP/data"
	UPG_BACKUP_SRV="$UPG_BACKUP/srv"
	UPG_BACKUP_DB="$UPG_BACKUP/dbdump"
	return 0
}

# to remove the data, run with nonempty 2nd parameter
backup_cleanup() {
	echo -e "Backup failed ($1).\nAttempting to restart $NAME service (${DWARFG_NAM}.service) ..." >&2
	$C_SYSTEMCTL start "${DWARFG_NAM}.service" || echo "Unable to bring $NAME service back online (${DWARFG_NAM}.service)" >&2
	$C_SYSTEMCTL start "${DWARFG_NAM}_${SNMP_GW}.service" || echo "Unable to bring $NAME SNMP Gateway service back online (${DWARFG_NAM}_${SNMP_GW}.service)" >&2
	[ -n "$2" ] && {
		echo "Cleaning up failed backup..."
		[[ $((${#UPG_BACKUP})) -gt $((1+${#DWARFG_BACKUP})) && -d "$UPG_BACKUP" ]] && rm -r "$UPG_BACKUP"
	}
	return 1
}

# backup up the CURRENT version of Dwarfg
backup() {
	local l_svc_active
	echo "Backing up existing $NAME installation from $BINDIR to $UPG_BACKUP"
	$C_SYSTEMCTL is-active ${DWARFG_NAM}.service >/dev/null && {
		l_svc_active="yes"
		echo "Stopping $NAME service (${DWARFG_NAM}.service) ..."
		$C_SYSTEMCTL stop "${DWARFG_NAM}.service" || { echo "Unable to stop $NAME service (${DWARFG_NAM}.service)" >&2 && return 1; }
		$C_SYSTEMCTL stop "${DWARFG_NAM}_${SNMP_GW}.service" || echo "Unable to stop $NAME SNMP GW service" >&2
	}
	event_log 301 "MSG=$NAME backup to target directory: $UPG_BACKUP"
	echo "Preparing target directory ..."
	[ -d "$UPG_BACKUP" ] && {
		local ROTDIR
		ROTDIR="$DWARFG_BACKUP/rot_$(date +%s)_$DWARFG_DBN%$VERSION" # NOTE: constructed in UPG_BACKUP as well
		echo "Rotating old backup to \"$ROTDIR\"... "
		mv "$UPG_BACKUP" "$ROTDIR" || {
			backup_cleanup "Unable to rotate old backup ($UPG_BACKUP) to \"$ROTDIR\"."
			return 1
		}
	}
	$C_MKDIR -p "$UPG_BACKUP" || {
		backup_cleanup "Unable to create backup directory ($UPG_BACKUP)" "clean"
		return 1
	}
	$C_MKDIR "$UPG_BACKUP_DATA" || {
		backup_cleanup "Unable to create backup data directory ($UPG_BACKUP_DATA)" "clean"
		return 1
	}
	$C_MKDIR "$UPG_BACKUP_DB" || {
		backup_cleanup "Unable to create backup db directory ($UPG_BACKUP_DB)" "clean"
		return 1
	}
	$C_TOUCH "$UPG_BACKUP/$BACKUP_STARTED"
	# perform complete copy and DB dump
	echo "Copying $NAME data ..."
	TEX=
	[ -n "$SECRETSDIR" ] && TEX="--exclude \"$SECRETSDIR/*\""
	$C_RSYNC -a $TEX "$BINDIR/" "$UPG_BACKUP_DATA/" || {
		backup_cleanup "Failed to backup $NAME data" "clean"
		return 1
	}
	$C_RSYNC -a "$SRVDIR/" "$UPG_BACKUP_SRV/" || {
		backup_cleanup "Failed to backup $NAME srv" "clean"
		return 1
	}
	echo "Dumping $NAME DB ..."
	$C_MYSQLDUMP "$DWARFG_DBN" >"$UPG_BACKUP_DB/$DBDUMP_FN" || {
		backup_cleanup "Failed to backup $NAME DB" "clean"
		return 1
	}
	$C_TOUCH "$UPG_BACKUP/$BACKUP_FINISHED"
	$C_RM "$UPG_BACKUP/$BACKUP_STARTED"
	[ -n "$l_svc_active" ] && {
		echo "Restarting $NAME service ..."
		$C_SYSTEMCTL start "${DWARFG_NAM}.service" || echo "Unable to bring $NAME service back online (${DWARFG_NAM}.service)" >&2
		$C_SYSTEMCTL start "${DWARFG_NAM}_${SNMP_GW}.service" || echo "Unable to bring $NAME SNMP GW service back online" >&2
	}
	event_log 352
	return 0 # backup performed correctly regardless if Dwarfg is down or up
}

restore_cleanup() {
	echo "Restore failed ($1). Cleaning up deployment and aborting restore" >&2
	remove_deploy || echo "Unable to remove reserved deployment" >&2
	return 1
}

# restores the CURRENT version of Dwarfg
restore() {
	local STR VSTR BDIR VERS DOMA
	STR=$($C_DIRNAME "$DIR")
	VSTR=$($C_BASENAME "$STR")
	BDIR=$($C_DIRNAME "$STR")
	VERS=${VSTR##*%}
	DOMA=${VSTR%%%*}
	[ "$BDIR" != "$DWARFG_BACKUP" ] && {
		echo -e "To restore a $NAME version, you must use the script from that particular version:\n\tExpected dirname: $DWARFG_BACKUP\n\tGot instead: $BDIR (out of $DIR)" >&2
		return 1
	}
	[[ -z "$VERS" || -z "$DOMA" || "$VSTR" = "$VERS" || "$VSTR" = "$DOMA" ]] && {
		echo -e "Unable to parse domain or version from backup directory:\n\tParsing string: $VSTR\n\tDomain: $DOMA\n\tVersion: $VERS.\nAborting restore." >&2
		return 1
	}
	[ "$VERS" = "$VERSION" ] || {
		echo -e "Parsed version ($VERS) is not equal to the version read ($VERSION).\nAborting restore." >&2
		return 1
	}
	[ "$DOMA" = "$DWARFG_DBN" ] || {
		echo -e "Parsed domain ($DOMA) is not equal to the domain read ($DOMAIN).\nAborting restore." >&2
		return 1
	}
	# check backup has been finished!
	[ -f "$UPG_BACKUP/$BACKUP_FINISHED" ] || {
		echo -e "Backup finish indication file ($UPG_BACKUP/$BACKUP_FINISHED) not found.\nAborting restore." >&2
		return 1
	}
	# grab DB password from .env in gui
	DBPASS=""
	[ -f "$UPG_BACKUP/data/web/gui/.env" ] || {
		echo -e "Unable to locate env file ($UPG_BACKUP/data/web/gui/.env) in the backup. Unable to restore DB password." >&2
		return 1
	}
	DBPASS=$($C_GREP "^DATABASE_URL=" "$UPG_BACKUP/data/web/gui/.env" | $C_TAIL -1 | $C_SED "s/.*:\([^@]*\)@.*/\1/g")
	[ -z "$DBPASS" ] && {
		echo -e "Unable to parse DB password from env file (or password empty).\nAborting restore." >&2
		return 1
	}
	export DBPASS
	# try reserving deployment
	echo "$NAME version/domain: $VERSION/$DOMAIN. Reserving deployment ($DEPLOY_POSTFIX)"
	reserve_deploy || {
		echo "Reserving deployment failed (conflict: $DEPLOY_BLOCK), aborting restore." >&2
		return 1
	}
	# the restore MAY restore Dwarfg to a different deployment number!
	echo "Restoration deployment number: $TARGET_DEPLOY_NUM"
	echo "Please confirm the deployment number above. Port offset and daemon port are calculated from the above."
	read
	PORT_OFFSET=$((OFFSET_SHIFT*(TARGET_DEPLOY_NUM-1)))
	update_offset "$PORT_OFFSET"
	OLD_DAEMON_PORT=$DWARFG_PORT
	DWARFG_PORT=$((DWARFG_PORT+TARGET_DEPLOY_NUM-1))
	update_daemon_port "$DWARFG_PORT"
	[ "$OLD_DAEMON_PORT" -eq "$DWARFG_PORT" ] && OLD_DAEMON_PORT=""
	# check if target directory is not existing
	[[ ( -e "$BINDIR" ) || ( -e "$SRVDIR" ) ]] && {
		restore_cleanup "Conflicting directories (either $BINDIR or $SRVDIR) exists, aborting restore"
		return 1
	}
	add_app_group || {
		restore_cleanup "Unable to add $NAME group ($APPGROUP)"
		return 1
	}
	add_app_user || {
		restore_cleanup "Unable to add user $APPUSER"
		return 1
	}
	add_utg || {
		restore_cleanup "Unable to add user $APPUSER to the group $WMGROUP"
		return 1
	}
	add_app_home || {
		restore_cleanup "Unable to prepare $NAME home directory ($DWARFG_HOME)"
		return 1
	}
	create_app_fwdir || {
		restore_cleanup "Unable to prepare $NAME FW download directory"
		return 1
	}
	# restore data
	echo "Restoring $NAME data... "
	$C_RSYNC -a "$UPG_BACKUP_DATA/" "$BINDIR" || {
		restore_cleanup "Failed to restore $NAME data"
		return 1
	}
	# restore srv
	$C_RSYNC -a "$UPG_BACKUP_SRV/" "$SRVDIR" || {
		restore_cleanup "Failed to restore $NAME srv"
		return 1
	}
	echo "Restoring $NAME database... "
	prep_db || {
		restore_cleanup "Failed to create DB or user"
		return 1
	}
	[ -f "$UPG_BACKUP_DB/$DBDUMP_FN" ] || {
		restore_cleanup "$NAME DB data ($UPG_BACKUP_DB/$DBDUMP_FN) are missing."
		return 1
	}
	# restore DB data
	$C_MYSQL $MYSQOPT "$DWARFG_DBN" < "$UPG_BACKUP_DB/$DBDUMP_FN" || {
		restore_cleanup "Failed to restore $NAME DB data."
		return 1
	}
	add_app_sudoers || {
		restore_cleanup "Unable to add $NAME sudoers rules."
		return 1
	}
	# link apache config
	add_apache_links || {
		restore_cleanup "Unable to link $NAME web into Apache HTML tree."
		return 1
	}
	# update Dwarfg port in apache config if any change there
	[ -n "$OLD_DAEMON_PORT" ] && {
		echo "Daemon port change during restore from $OLD_DAEMON_PORT to $DWARFG_PORT, updating apache config ..."
		$C_SED -i --follow-symlinks "s/\(ProxyPassMatch.*:\)$OLD_DAEMON_PORT\//\1$DWARFG_PORT\//" "$BINDIR/$DWARFG_APACONF" || {
			restore_cleanup "Unable to replace old port ($OLD_DAEMON_PORT) by new one ($DWARFG_PORT) in Apache config"
			return 1
		}
		$C_SED -i --follow-symlinks "s/\(ProxyPassReverse.*:\)$OLD_DAEMON_PORT\//\1$DWARFG_PORT/" "$BINDIR/$DWARFG_APACONF" || {
			restore_cleanup "Unable to replace old port ($OLD_DAEMON_PORT) by new one ($DWARFG_PORT) in Apache config"
			return 1
		}
	}
	apa_servsite || {
		restore_cleanup "Failed to add $NAME web as Apache2 site."
		return 1
	}
	echo "Adding $NAME service and starting it up ..."
	# add Dwarfg service to systemd
	add_app_service || {
		restore_cleanup "Failed to add $NAME service."
		return 1
	}
	run_app_svc || {
		restore_cleanup "Failed to start up restored $NAME service."
		return 1
	}
	$APACHE_RELOAD || {
		restore_cleanup "Unable to reload Apache"
		return 1
	}
	event_log 353
	$C_SYSTEMCTL is-active ${DWARFG_NAM}.service >/dev/null && {
		echo "$NAME restored and on-line"
		return 0
	}
	echo "$NAME restored but NOT on-line"
	return 1
}

upgrade() {
	local TARDIR STR TARVER l_myodir ini_updated
	l_myodir=$(pwd)
	[[ $# -eq 1 && -n "$1" ]] || {
		echo "No archive file with upgrade / no upgrade directory provided." >&2
		return 1
	}
	[[ -f "$1" || -d "$1" ]] || {
		echo "The target provided ($1) is neither a file nor a directory." >&2
		return 1
	}
	# unpack upgrade package
	if [ -f "$1" ] ; then
		TARDIR="/tmp/${SHORTNAME}_upgrade.$$"
		$C_MKDIR "$TARDIR" && cd $TARDIR || return 1
		tar xzf "$1" || {
			echo "Unable to unpack target archive ($1)." >&2
			cd $l_myodir
			rm -rf "$TARDIR" 2>/dev/null
			return 1
		}
	else
		TARDIR="$1"
		TARDIR=${TARDIR%/}
	fi
	[ -d "$TARDIR/${SHORTNAME}_install" ] && TARDIR="$TARDIR/${SHORTNAME}_install"
	[ -d "$TARDIR/appdir" ] && TARDIR="$TARDIR/appdir"
	[ -f "$TARDIR/$BASEDEFS" ] || {
		echo "Target $NAME version package is missing basic definitions file ($TARDIR/$BASEDEFS)" >&2
		cd $l_myodir
		return 1
	}
	STR=$($C_GREP "^VERSION=" "$TARDIR/$BASEDEFS" | $C_TAIL -1)
	STR2="${STR#VERSION=\"}"
	STR2="${STR2%\"}"
	TARVER="${STR2//[^0-9.]/}"
	[[ -n "$STR" && "$TARVER" = "$STR2" ]] || {
		echo "Garbled version in target $NAME package ($STR / $TARVER)" >&2
		cd $l_myodir
		return 1
	}
	# test upgradability (is my version supported by upgrade)
	[ -f "$TARDIR/upgrade_from_${VERSION}.sh" ] || {
		echo "Upgrade from current version ($VERSION) to target version ($TARVER) is not supported." >&2
		cd $l_myodir
		return 1
	}
	# test license - is new version supported by my license?
	if [ -x "$TARDIR/licman" ] ; then
		"$TARDIR/licman" -n -c "$BINDIR/${SHORTNAME}_license.lic"
		RES=$?
		[ "$RES" -eq 0 ] || {
			echo "License check not passed - does your license support target $NAME version? Upgrade aborted."
			cd $l_myodir
			return 1
		}
	else
		echo "License manager not found in the new package, no way to check if your license is ok for target $NAME version" >&2
		cd $l_myodir
		return 1
	fi
	diff "$TARDIR/appdir/${SHORTNAME}.ini" "$BINDIR/${SHORTNAME}.ini"
	ini_updated=$?
	# make backup
	# (stores certificates, dwarflib config, Dwarfg config)
	[ -n "$NOBACKUP" ] || {
		backup || {
			echo "Backup failed, aborting upgrade..." >&2
			$l_myodir
			return 1
		}
	}
	event_log 301 MSG="Attempting to upgrade to $NAME version $TARVER"
	# stop Dwarfg
	echo "Stopping ${DWARFG_NAM}.service..."
	$C_SYSTEMCTL stop "${DWARFG_NAM}.service" || { echo "Unable to stop $NAME service (${DWARFG_NAM}.service)" >&2 && return 1; }
	$C_SYSTEMCTL stop "${DWARFG_NAM}_${SNMP_GW}.service" || echo "Unable to stop $NAME SNMP GW service." >&2
	# kick upgrade (in subshell ???)
	echo "Executing new package upgrade script ($TARDIR/upgrade_from_${VERSION}.sh) ..."
	/bin/bash "$TARDIR/upgrade_from_${VERSION}.sh" "$DIR" || {
		event_log 302 MSG="$NAME upgrade failed"
		if [ -z "$NOBACKUP" ] ; then
			echo "Upgrade not successful, running cleanup and restore..."
			"$BINDIR/cleanup.sh"
			"$UPG_BACKUP_DATA/${SHORTNAME}_upgrade.sh" restore || {
				echo "Restore failed." >&2
				echo "Removing deployment..."
				UPGRADING=0 remove_deploy
			}
		else
			echo -e "Skipping restore because no backup parameter was used.\nYou can still run restore manually:\n\t$UPG_BACKUP/${SHORTNAME}_upgrade.sh restore\n"
			echo "Removing deployment..."
			UPGRADING=0 remove_deploy
		fi
		return 1
	}
	[ "0" != "$ini_updated" ] && echo -e "\nWARNING, your old $SHORTNAME.ini file has been replaced by newer version. If you made custom changes to the file, edit the new version to not loose them. You can find your old version of the file in the backup.\n"
	echo "Upgrade successful."
	return 0
}

interactive() {
	local LAST_ACTION RESP
	LAST_ACTION="no action";
	while true; do
		echo -e "\n\n\n"
		echo "$NAME upgrade script interactive mode."
		echo "Last action was: $LAST_ACTION"
		echo "Select one of the options using number:"
		echo -e "\t1) Exit this script"
		echo -e "\t2) Backup $NAME (version: $VERSION)"
		echo -e "\t3) Restore $NAME (version: $VERSION)"
		echo -e "\t4) Upgrade $NAME (package path: ${PACKAGE_PATH:-\"Not selected yet\"}) and EXIT"
		echo -e "\t5) Set path to upgrade package / directory with unpacked package."
		read -r RESP
		echo
		case "$RESP" in
			"1")
				return 0
				;;
			"2")
				backup
				LAST_ACTION="backup / ERROR: $?"
				;;
			"3")
				restore
				LAST_ACTION="restore / ERROR: $?"
				;;
			"4")
				upgrade "$PACKAGE_PATH"
				RES=$?
				LAST_ACTION="upgrade / ERROR: $RES"
				echo "Now exiting (press ENTER) ..."
				read
				exit $RES
				;;
			"5")
				echo "Enter package path (or directory path with unpacked upgrade package):"
				read -r PACKAGE_PATH_CAND
				if [[ -d "$PACKAGE_PATH_CAND" || -f "$PACKAGE_PATH_CAND" ]] ; then
					PACKAGE_PATH="$PACKAGE_PATH_CAND"
				else
					echo "Provided path is not a directory or a file."
				fi
				;;
			*)
				echo "Garbled selection, iterating menu..."
				LAST_ACTION="skipped"
				$C_SLEEP 1
				;;
		esac
	done
}

# Dwarfg core upgrade script
[ 0 -ne "$(id -u)" ] && {
	echo "$NAME upgrade script must run under root."
	exit 1
}
# need to know: existing version
source_and_evaluate "$(dirname $0)" || {
	echo "Unable to source $BASEDEFS - needs to reside at the same directory this script is situated." >&2
	exit 1
}
# now either exec requested mode or go interactive...
[ $# -eq 0 ] && {
	interactive
	exit
}
while [ $# -gt 0 ] ; do 
	case "$1" in
		"PARAM_HELP")
			print_help
			exit 0
			;;
		"PARAM_NOBACKUP")
			NOBACKUP="nobackup"
			;;
		"backup")
			backup
			exit $?
			;;
		"restore")
			restore
			exit $?
			;;
		"upgrade")
			[ -z "$2" ] && echo "Need archive with new $NAME version..." >&2 && exit 1
			[ ! -d "$2" ] && echo "New $NAME version archive ($2) does not exist." >&2 && exit 1
			upgrade "$2"
			shift
			;;
		*)
			echo "Unknown command ($1) requested." >&2
			print_help
			exit 1
			;;
	esac
	shift
done
exit 0
