<?php

namespace App\Form;

use App\Entity\Tool\SimCard;
use App\Entity\Tool\SmsTemplate;
use App\Validator\Constraints\HasDeviceOneLinkableNetDev;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;

class SmsSendType extends AbstractType
{
    private EntityManagerInterface $em;
    private mixed $devices;
    private mixed $smsTemplates;
    private array $options;

    public function __construct(EntityManagerInterface $em){

        $this->em = $em;

    }

    /**
     * @param FormBuilderInterface $builder
     * @param array $options
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {

        $this->devices = $options['data']['devices'];
        $this->smsTemplates = $options['data']['smsTemplates'];
        $this->options = $options;
        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event): void {

            $sms = $event->getData();
            $form = $event->getForm();

            $this->addElements($form, null, null);

        });

        $builder->addEventListener(FormEvents::PRE_SUBMIT, array($this, 'onPreSubmit'));

    }

    protected function addElements(FormInterface $form, ?SmsTemplate $smsTemplateForm, ?string $destinationType): void
    {

        //remove the send button to make the right input order, not to show msisdn after buttons.
        $form->remove('send');
        $form->remove('destination_type');

        $devicesChoices = [];
        $smsTemplatesChoices = [];
        $deviceChoicesAttr = [];
        foreach ($this->devices as $item){

            $devicesChoices[$item['devid']] = $item['id_device'];

        }

        foreach ($this->smsTemplates as $smsTemplate){

            $smsTemplatesChoices[$smsTemplate->getLabel()] = $smsTemplate->getId();

        }

        $form
            ->add('device', ChoiceType::class, [
                'label' => 'Send from the device',
                'required' => true,
                'choices' => $devicesChoices,
                'choice_attr' => $deviceChoicesAttr,
                'placeholder' => 'Select a Device',
                'data' => $options['data']['device'] ?? null,
                'mapped' => false,
                'auto_initialize' => false,
                'attr' => ['data-row-class' => 'sim_card[device]'], //Important to setup to send data from the form
                'block_prefix' => 'modal_body',
                'constraints' => [
                    new HasDeviceOneLinkableNetDev('device', (array_key_exists('msisdn', $this->options['data'])) ? $this->options['data']['device'] : '', 'Device has more than one net devs set as linkable!'),
                ],
                'help' => '<p style="padding-left: 10px;">You can select all devices with system label <b>Sms</b> and assigned <b>Authentication profile</b>, selected Linkable network device with IP.</b></p>
                            <p style="padding-left: 10px;">Device must have only one <b>Network device</b> set as <b>Linkable</b>, see the device detail to set it up.</p>',
                'help_html' => true,
            ])
            ->add('template', ChoiceType::class, [
                'label' => 'Template',
                'required' => true,
                'choices' => $smsTemplatesChoices,
                //'choice_attr' => $deviceChoicesAttr,
                'placeholder' => 'Select a Template',
                //'data' => (isset($options['data']['device'])) ? $options['data']['device'] : null,
                'mapped' => false,
                'auto_initialize' => false,
                //'attr' => ['data-row-class' => 'sim_card[device]'], //Important to setup to send data from the form
                'block_prefix' => 'modal_body',
                'attr' => ['data-row-class' => 'sms_send_template'/*, 'help'=>'text help'*/],
            ])
        ;

        if($smsTemplateForm && $smsTemplateForm->getLabel() === 'Arbitrary SMS'){

            $form->add('custom_text', TextType::class, [
                'label' => 'Custom text',
                'required' => true,
                'data' => '',
                'mapped' => false,
                'attr' => ['data-row-class' => 'sms_custom_text'/*, 'help'=>'text help'*/],
            ]);

        }

        $form->add('destination_type', ChoiceType::class, [
            'label' => 'Destination Type',
            'required' => true,
            'choices' => ['MSISDN' => 'MSISDN', 'Device' => 'Device'],
            'data' => '',
            'mapped' => false,
            'auto_initialize' => false,
            'placeholder' => 'Select a Destination Type',
            'block_prefix' => 'modal_body',
            'attr' => ['data-row-class' => 'sms_send_destination_type'],
        ]);

        if($destinationType === 'MSISDN'){

            $form->add('msisdn', TextType::class, [
                'label' => 'MSISDN',
                'required' => true,
                'data' => '',
                'mapped' => false,
                'attr' => ['data-row-class' => 'sms_send_msisdn'/*, 'help'=>'text help'*/],
                'constraints' => [
                    new NotBlank(),
                    new Regex('/^[0-9]{3,17}$/')
                ],
                'help' => '<p style="padding-left: 20px;">MSISDN is a phone number in a format 00420_________</p>',
                'help_html' => true,
            ]);

        }

        if($destinationType === 'Device'){

            $activeSimCards = $this->em->getRepository(SimCard::class)->findAllActiveSimCards();

            $activeSimCardsChoices = [];

            foreach ($activeSimCards as $simCard){

                $activeSimCardsChoices[$simCard->getDeviceUser()->getDevId() . ' ' . $simCard->getMsisdn()] = $simCard->getId();

            }

            $form->add('destination', ChoiceType::class, [
                'label' => 'Destination Device',
                'required' => true,
                'choices' => $activeSimCardsChoices,
                'data' => '',
                'mapped' => false,
                'auto_initialize' => false,
                'placeholder' => 'Select a Destination Device',
                'block_prefix' => 'modal_body',
                'attr' => ['data-row-class' => 'sms_send_destination_device'],
            ]);

        }

        $form->add('send', SubmitType::class, [
            'label' => 'Send',
            'label_html' => true,
            'block_prefix' => 'form_footer',
            'attr' => ['data-row-class' => 'alert_targetDevice', 'class' => 'btn btn-primary float-right']
        ])
        ;

    }

    public function onPreSubmit(FormEvent $event): void
    {

        $form = $event->getForm();
        $data = $event->getData();

        $templateId = $data['template'];
        $destinationType = $data['destination_type'];
        $smsTemplate = null;

        if(isset($templateId)){
            $smsTemplate = $this->em->getRepository(SmsTemplate::class)->findOneBy(['id' => $templateId]);
        }

        $this->addElements($form, $smsTemplate, $destinationType);

    }

}