<?php

namespace App\Twig;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Twig\Extension\AbstractExtension;
use Twig\Extension\GlobalsInterface;
use Twig\TwigFilter;

class AppExtension extends AbstractExtension implements GlobalsInterface
{

    protected ContainerInterface $container;

    protected string $productName;
    protected string $productWeb;
    protected string $productEmail;
    protected string $manufacturerName;
    protected string $grafanaRedirectUrl;
    protected string $grafanaLoginUsername;
    protected string $grafanaLoginPassword;

    /**
     * @param ContainerInterface $container
     * @param string $productName
     */
    public function __construct(ContainerInterface $container, string $productName, string $productWeb,
                                string $productEmail, string $manufacturerName, string $grafanaRedirectUrl,
                                string $grafanaLoginUsername, string $grafanaLoginPassword){

        $this->container = $container;
        $this->productName = $productName;
        $this->productWeb = $productWeb;
        $this->productEmail = $productEmail;
        $this->manufacturerName = $manufacturerName;
        $this->grafanaRedirectUrl = $grafanaRedirectUrl;
        $this->grafanaLoginUsername = $grafanaLoginUsername;
        $this->grafanaLoginPassword = $grafanaLoginPassword;

    }

    public function getGlobals(): array {

        return [
            'productName' => $this->productName,
            'productWeb' => $this->productWeb,
            'productEmail' => $this->productEmail,
            'manufacturerName' => $this->manufacturerName,
            'grafanaRedirectUrl' => $this->grafanaRedirectUrl,
            'grafanaLoginUsername' => $this->grafanaLoginUsername,
            'grafanaLoginPassword' => $this->grafanaLoginPassword,
        ];

    }

    /**
     * @return TwigFilter[]
     */
    public function getFilters(): array {

        return [
            new TwigFilter('formatBytes', array($this, 'formatBytes')),
            new TwigFilter('seconds2human', array($this, 'seconds2human')),
        ];

    }

    /**
     * @param int $bytes
     * @param int $precision
     * @return string
     */
    public function formatBytes(int $bytes, int $precision = 2): string{

        $size = ['B', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
        $factor = floor((strlen($bytes) - 1) / 3);

        return sprintf("%.{$precision}f", $bytes / pow(1024, $factor)) . ' ' . @$size[$factor];

    }

    /**
     * @param int $inputSeconds
     * @return string
     */
    public function seconds2human(int $inputSeconds): string {
        
        $secondsInAMinute = 60;
        $secondsInAnHour  = 60 * $secondsInAMinute;
        $secondsInADay    = 24 * $secondsInAnHour;

        // extract days
        $days = abs($inputSeconds / $secondsInADay);
        $days = floor($days);

        // extract hours
        $hourSeconds = $inputSeconds % $secondsInADay;
        $hours = abs($hourSeconds / $secondsInAnHour);
        $hours = floor($hours);

        // extract minutes
        $minuteSeconds = $hourSeconds % $secondsInAnHour;
        $minutes = abs($minuteSeconds / $secondsInAMinute);
        $minutes = floor($minutes);

        // extract the remaining seconds
        $remainingSeconds = $minuteSeconds % $secondsInAMinute;
        $seconds = abs($remainingSeconds);
        $seconds = ceil($remainingSeconds);

        return "$days days, " . sprintf("%02d:%02d:%02d", $hours, $minutes, $seconds);

    }

}