<?php

namespace Unique\UserBundle\Controller;

use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Routing\Annotation\Route;
use Unique\UserBundle\Configuration\ConfigInterface;
use Unique\UserBundle\Event\UserEvent;
use Unique\UserBundle\Model\GroupInterface;
use Unique\UserBundle\Model\UserInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Exception\InvalidArgumentException;
use Symfony\Component\Form\FormError;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class SecurityController extends AbstractController
{
    /**
     * @var TranslatorInterface
     */
    private TranslatorInterface $translator;

    /**
     * @var MailerInterface
     */
    private MailerInterface $mailer;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $dispatcher;

    /**
     * @param TranslatorInterface $translator
     * @param MailerInterface $mailer
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(
        TranslatorInterface $translator,
        MailerInterface $mailer,
        EventDispatcherInterface $dispatcher
    ) {

        $this->translator = $translator;
        $this->mailer = $mailer;
        $this->dispatcher = $dispatcher;
    }

    /**
     * security_login
     * Login.
     */
    #[Route('/login', name: 'security_login')]
    public function login(AuthenticationUtils $authenticationUtils): Response
    {
        // Check Auth
        if ($this->checkAuth()) {
            return $this->redirectToRoute($this->getParameter('login_redirect'));
        }

        // Render
        return $this->render($this->getParameter('template_path') . '/security/login.html.twig', [
            'last_username' => $authenticationUtils->getLastUsername(),
            'error' => $authenticationUtils->getLastAuthenticationError(),
            'user_registration' => $this->getParameter('user_registration'),
        ]);
    }

    #[Route('/logout', name: 'security_logout')]
    public function logout(): Response
    {
        // controller can be blank: it will never be called!
        throw new \Exception('Don\'t forget to activate logout in security.yaml');
    }

    /**
     * Registration.
     */
    #[Route('/register', name: 'security_register')]
    public function register(Request $request, ManagerRegistry $managerRegistry, UserPasswordHasherInterface $hasher): Response
    {

        // Check Auth
        if ($this->checkAuth()) {
            return $this->redirectToRoute($this->getParameter('login_redirect'));
        }

        // Check Disable Register
        if (!$this->getParameter('user_registration')) {
            $this->addFlash('error', $this->translator->trans('security.registration_disable'));

            return $this->redirectToRoute('security_login');
        }

        // Create User
        $user = $this->getParameter('user_class');
        $user = new $user();
        if (!$user instanceof UserInterface) {
            throw new InvalidArgumentException();
        }

        // Dispatch Register Event
        if ($response = $this->dispatcher->dispatch(new UserEvent($user), UserEvent::REGISTER_BEFORE)->getResponse()) {
            return $response;
        }

        // Create Form
        $form = $this->createForm($this->getParameter('register_type'), $user);

        // Handle Form Submit
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Encode Password
            $password = $hasher->hashPassword($user, $form->get('plainPassword')->getData());
            $user->setPassword($password);

            // User Confirmation
            if ($this->getParameter('email_confirmation')) {
                // Disable User
                $user->setActive(false);

                // Create Confirmation Token
                if (empty($user->getConfirmationToken()) || null === $user->getConfirmationToken()) {
                    $user->createConfirmationToken();
                }

                // Send Confirmation Email
                $emailBody = [
                    'confirmationUrl' => $this->generateUrl(
                        'security_register_confirm',
                        ['token' => $user->getConfirmationToken()],
                        UrlGeneratorInterface::ABSOLUTE_URL
                    ),
                ];
                $this->sendEmail($user, 'email.account_confirmation', 'register', $emailBody);
            } elseif ($this->getParameter('welcome_email')) {
                // Send Welcome
                $this->sendEmail($user, 'email.registration_complete', 'welcome');
            }

            // User Add Default Group
            if ($group = $this->getParameter('default_group')) {
                $getGroup = $managerRegistry->getRepository($this->getParameter('group_class'))->find($group);
                if ($getGroup instanceof GroupInterface) {
                    $user->addGroup($getGroup);
                }
            }

            // Save User
            $managerRegistry->getManager()->persist($user);
            $managerRegistry->getManager()->flush();

            // Dispatch Register Event
            if ($response = $this->dispatcher->dispatch(new UserEvent($user), UserEvent::REGISTER)->getResponse()) {
                return $response;
            }

            // Register Success
            return $this->render($this->getParameter('template_path') . '/registration/registerSuccess.html.twig', [
                'user' => $user,
            ]);
        }

        // Render
        return $this->render($this->getParameter('template_path') . '/registration/register.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    /**
     * Registration Confirm Token.
     */
    #[Route('/register/{token}', name: 'security_register_confirm')]
    public function registerConfirm(MailerInterface $mailer, ManagerRegistry $managerRegistry, string $token): Response
    {

        // Find User
        $user = $managerRegistry->getRepository($this->getParameter('user_class'))->findOneBy(['confirmationToken' => $token]);
        if (null === $user) {
            throw $this->createNotFoundException(sprintf($this->translator->trans('security.token_notfound'), $token));
        }

        // Enabled User
        $user->setConfirmationToken(null);
        $user->setActive(true);

        // Send Welcome
        if ($this->getParameter('welcome_email')) {
            $this->sendEmail($user, 'email.registration_complete', 'welcome');
        }

        // Update User
        $managerRegistry->getManager()->persist($user);
        $managerRegistry->getManager()->flush();

        // Dispatch Register Event
        if ($response = $this->dispatcher->dispatch(new UserEvent($user), UserEvent::REGISTER_CONFIRM)->getResponse()) {
            return $response;
        }

        // Register Success
        return $this->render($this->getParameter('template_path') . '/registration/registerSuccess.html.twig', [
            'user' => $user,
        ]);
    }

    /**
     * Lost password Request.
     */
    #[Route('/password/reset', name: 'security_lost_password')]
    public function lostPassword(Request $request, ManagerRegistry $managerRegistry): Response
    {

        // Check Auth
        if ($this->checkAuth()) {
            return $this->redirectToRoute($this->getParameter('login_redirect'));
        }

        // Build Form
        $form = $this->createForm($this->getParameter('resetting_type'));

        // Handle Form Submit
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            // Find User
            $user = $managerRegistry->getRepository($this->getParameter('user_class'))->findOneBy(['email' => $form->get('username')->getData()]);
            if (null === $user) {
                $form->get('username')->addError(new FormError($this->translator->trans('security.user_not_found')));
            } else {
                // Create TTL
                if ($user->isPasswordRequestNonExpired($this->getParameter('resetting_request_time'))) {
                    $form->get('username')->addError(
                        new FormError($this->translator->trans(
                            'security.resetpw_wait_resendig',
                            ['%s' => ($this->getParameter('resetting_request_time') / 3600)]
                        ))
                    );
                } else {
                    // Create Confirmation Token
                    if (empty($user->getConfirmationToken()) || null === $user->getConfirmationToken()) {
                        $user->createConfirmationToken();
                        $user->setPasswordRequestedAt(new \DateTime());
                    }

                    // Send Resetting Email
                    $emailBody = [
                        'confirmationUrl' => $this->generateUrl(
                            'security_reset_password',
                            ['token' => $user->getConfirmationToken()],
                            UrlGeneratorInterface::ABSOLUTE_URL
                        ),
                    ];
                    $this->sendEmail($user, 'email.account_password_resetting', 'resetting', $emailBody);

                    // Update User
                    $managerRegistry->getManager()->persist($user);
                    $managerRegistry->getManager()->flush();

                    // Dispatch Register Event
                    if ($response = $this->dispatcher->dispatch(new UserEvent($user), UserEvent::RESETTING)->getResponse()) {
                        return $response;
                    }

                    // Render
                    return $this->render($this->getParameter('template_path') . '/reset/resetPasswordSuccess.html.twig', [
                        'sendEmail' => true,
                        'form' => $form->createView(),
                    ]);
                }
            }
        }

        // Render
        return $this->render($this->getParameter('template_path') . '/reset/lostPassword.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    /**
     * Reset Password Form.
     */
    #[Route('/password/reset/{token}', name: 'security_reset_password')]
    public function resetPassword(Request $request, ManagerRegistry $managerRegistry, UserPasswordHasherInterface $encoder, string $token): Response
    {

        // Find User
        $user = $managerRegistry->getRepository($this->getParameter('user_class'))->findOneBy(['confirmationToken' => $token]);
        if (null === $user) {
            throw $this->createNotFoundException(sprintf($this->translator->trans('security.token_notfound'), $token));
        }

        // Build Form
        $form = $this->createForm($this->getParameter('resetting_password_type'), $user);

        // Handle Form Submit
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Encode Password & Set Token
            $password = $encoder->hashPassword($user, $form->get('plainPassword')->getData());
            $user->setPassword($password)
                ->setConfirmationToken(null)
                ->setPasswordRequestedAt(null);

            // Save User
            $managerRegistry->getManager()->persist($user);
            $managerRegistry->getManager()->flush();

            // Dispatch Register Event
            if ($response = $this->dispatcher->dispatch(new UserEvent($user), UserEvent::RESETTING_COMPLETE)->getResponse()) {
                return $response;
            }

            // Send Resetting Complete
            $this->sendEmail($user, 'email.password_resetting_completed', 'resetting-complete');

            // Render Success
            return $this->render($this->getParameter('template_path') . '/reset/resetPasswordSuccess.html.twig', [
                'sendEmail' => false,
                'form' => $form->createView(),
            ]);
        }

        // Render
        return $this->render($this->getParameter('template_path') . '/reset/resetPassword.html.twig', [
            'token' => $token,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Check User Authorized.
     */
    private function checkAuth(): bool
    {

        return $this->isGranted('IS_AUTHENTICATED_FULLY') || $this->isGranted('IS_AUTHENTICATED_REMEMBERED');
    }

    /**
     * Send Mail.
     */
    private function sendEmail(UserInterface $user, string $subject, string $templateId, array $data = []): void
    {
        // Create Email
        $email = (new Email())
            ->from(new Address($this->getParameter('mail_sender_address'), $this->getParameter('mail_sender_name')))
            ->to($user->getEmail())
            ->subject($this->translator->trans($subject))
            ->html($this->renderView($this->getParameter('template_path') . "/email/{$templateId}.html.twig", array_merge(['user' => $user], $data)));

        // Send
        $this->mailer->send($email);
    }

    /**
     * Override Parameters
     */
    protected function getParameter(string $name): \UnitEnum|float|array|bool|int|string|null
    {
        return $this->container->has('app.params') ? $this->container->get('app.params')->get($name) : parent::getParameter($name);
    }

    /**
     * Add Custom Services
     */
    public static function getSubscribedServices(): array
    {
        return array_merge([
            'app.params' => '?' . ConfigInterface::class,
        ], parent::getSubscribedServices());
    }
}
