<?php

namespace App\Controller\Admin\Tools;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Controller\Admin\BaseController;
use App\DataTable\AuthManagementDatatable;
use App\Entity\Device\Device;
use App\Entity\Tool\AuthProfile;
use App\Exception\AuthProfileNotFoundException;
use App\Form\AuthProfileType;
use App\Form\DeleteType;
use App\Service\FfiService;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route('/admin/tools', name: 'admin_tool_')]
class AuthProfileManagementController extends BaseController
{

    /**
     * @param Request $request
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '/auth-management', name: 'auth_management_index')]
    public function simCardManagement(
        Request $request,
        DatatableFactory $factory,
        DatatableResponse $responseService,
    ): Response {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');
        $this->managerRegistry->getRepository(Device::class)->getDevices();

        $datatable = $factory->create(AuthManagementDatatable::class);
        $datatable->buildDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        $form = $this->createForm(AuthProfileType::class, []);

        return $this->render('tool/authManagement/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @throws AuthProfileNotFoundException
     */
    #[Route(path: '/auth-management/edit/{authProfileId}', name: 'auth_profile_edit')]
    public function authProfileEdit(
        Request $request,
        int $authProfileId,
        FfiService $ffiService,
    ): Response {
        $authProfile = $this->managerRegistry->getRepository(AuthProfile::class)->findOneBy(['id' => $authProfileId]);

        if (!$authProfile) {
            throw new AuthProfileNotFoundException();
        }

        $authProfileData = [
            'authProfile' => $authProfile,
            'profileName' => $authProfile->getProfileName(),
            'username' => $authProfile->getUsername(),
            'sshPsd' => $authProfile->getSshPwd(),
            'sshKey' => $authProfile->getSshKey(),
        ];

        $form = $this->createForm(AuthProfileType::class, $authProfileData);
        $deleteForm = $this->createForm(DeleteType::class, ['text' => 'Delete Auth Profile']);

        $form->handleRequest($request);
        $deleteForm->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $formData = $form->getData();
            $em = $this->managerRegistry->getManager();

            try {

                $authProfile->setProfileName($formData['profileName']);
                $authProfile->setUsername($formData['username']);
                //Do not update password when is not changed
                if (array_key_exists('sshPwd', $formData) && !empty($formData['sshPwd'])) {

                    $encodedPassword = $ffiService->encode($formData['sshPwd']);
                    $authProfile->setSshPwd($encodedPassword[0]);
                    $authProfile->setPwdSalt($encodedPassword[1]);
                    $authProfile->setPwdPepper($encodedPassword[2]);
                    $authProfile->setPwdChilli($encodedPassword[3]);
                    $authProfile->setPwdVersion($encodedPassword[4]);

                }

                if (array_key_exists('sshKey', $formData) && !empty($formData['sshKey'])) {

                    $encodedSshKey = $ffiService->encode($formData['sshKey']);
                    $authProfile->setSshKey($encodedSshKey[0]);
                    $authProfile->setKeySalt($encodedSshKey[1]);
                    $authProfile->setKeyPepper($encodedSshKey[2]);
                    $authProfile->setKeyChilli($encodedSshKey[3]);
                    $authProfile->setKeyVersion($encodedSshKey[4]);

                }

                $em->persist($authProfile);
                $em->flush();

                $this->addFlash(
                    'success',
                    'Auth Profile was updated.',
                );

            } catch (\Exception $e) {

                $this->addFlash(
                    'danger',
                    'Auth Profile connet be updated.',
                );

            }

            return $this->redirectToRoute('admin_tool_auth_management_index');
        }

        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {

            $em = $this->managerRegistry->getManager();

            try {

                $em->remove($authProfile);
                $em->flush();

                $this->addFlash(
                    'success',
                    'Auth Profile was successfully deleted.',
                );

            } catch (\Exception) {
                $this->addFlash(
                    'danger',
                    'Auth Profile was not deleted.',
                );
            }

            return $this->redirectToRoute('admin_tool_auth_management_index');
        }

        return $this->render('tool/authManagement/edit.html.twig', [
            'authProfileForm' => $form->createView(),
            'deleteForm' => $deleteForm->createView(),
        ]);
    }

}
