<?php

namespace App\EventSubscriber;

use App\Event\DeviceEditEvent;
use App\Repository\DeviceRepository;
use App\Repository\ProductFlagsRepository;
use App\Repository\DeviceSyncStatusRepository;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;


class DeviceSubscriber implements EventSubscriberInterface
{
    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @var EntityManagerInterface
     */
    private EntityManagerInterface $entityManager;

    /**
     * @var DeviceRepository
     */
    private DeviceRepository $deviceRepository;

    /**
     * @var ProductFlagsRepository
     */
    private ProductFlagsRepository $productFlagsRepository;

    /**
     * @var DeviceSyncStatusRepository
     */
    private DeviceSyncStatusRepository $syncStatusRepository;

    /**
     * @param LoggerInterface $logger
     * @param EntityManagerInterface $entityManager
     * @param DeviceRepository $deviceRepository
     * @param ProductFlagsRepository $productFlagsRepository
     * @param DeviceSyncStatusRepository $syncStatusRepository
     */
    public function __construct(LoggerInterface            $logger, EntityManagerInterface $entityManager,
                                DeviceRepository           $deviceRepository, ProductFlagsRepository $productFlagsRepository,
                                DeviceSyncStatusRepository $syncStatusRepository)
    {
        $this->logger = $logger;
        $this->entityManager = $entityManager;
        $this->deviceRepository = $deviceRepository;
        $this->productFlagsRepository = $productFlagsRepository;
        $this->syncStatusRepository = $syncStatusRepository;
    }

    /**
     * @return array[]
     */
    public static function getSubscribedEvents(): array {

        return [
            DeviceEditEvent::NAME => ['onDeviceEditEvent', 20],
        ];

    }

    /**
     * @param DeviceEditEvent $event
     * @return void
     */
    public function onDeviceEditEvent(DeviceEditEvent $event): void {

        $this->logger->info('Device Subscriber says: Event posted this data {id}', [
            'id' => $event->getId(),
        ]);

        $this->entityManager->beginTransaction();

        $userDevchgVersion = $this->productFlagsRepository->getUserDevchgVersion();
        $version = $userDevchgVersion->getVersion();
        $userDevchgVersion->setVersion($version + 1);

        $device = $this->deviceRepository->findOneBy(['id' => $event->getId()]);
        $device->setGuiVersion($version + 1);
        $device->getStats()->setSyncStatus($this->syncStatusRepository->getBeingProcessedStatus());

        $this->entityManager->persist($device);
        $this->entityManager->persist($userDevchgVersion);
        $this->entityManager->flush();

        $this->entityManager->commit();

    }

}