<?php

namespace App\Controller\Admin;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Datatables\CustomScriptDatatable;
use App\Entity\AgentProfile\AgentProfile;
use App\Entity\CustomScript\CustomScript;
use App\Entity\CustomScript\CustomScriptAgentProfileMap;
use App\Entity\CustomScript\CustomScriptDevTypeMap;
use App\Entity\DeviceType;
use App\Exception\CustomScriptNotFoundException;
use App\Form\CustomScriptType;
use App\Form\DeleteType;
use App\Form\LabelType;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use PharIo\Version\Exception;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Datatable\DatatableInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route('/admin/custom-script', name: 'admin_custom_script_')]
class CustomScriptController extends BaseController
{
    /**
     * @param Request $request
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '', name: 'index')]
    public function customScriptIndex(
        Request $request,
        DatatableFactory $factory,
        DatatableResponse $responseService
    ): Response {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $deviceTypes = $this->managerRegistry->getRepository(DeviceType::class)->findAll();
        $agentProfiles = $this->managerRegistry->getRepository(AgentProfile::class)->findAll();

        $datatable = $factory->create(CustomScriptDatatable::class);
        $datatable->buildDatatable(['deviceTypes' => $deviceTypes, 'agentProfiles' => $agentProfiles]);

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        $form = $this->createForm(LabelType::class);

        return $this->render('customScript/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'form' => $form->createView()
            ]);
    }

    #[Route(path: '/add', name: 'add', methods: ['GET', 'POST'])]
    public function addCustomScript(Request $request, ManagerRegistry $managerRegistry,
                             EntityManagerInterface $em): RedirectResponse|Response
    {

        $customScript = new CustomScript();
        $deviceTypes = $managerRegistry->getRepository(DeviceType::class)->findAll();
        $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findAll();

        $form = $this->createForm(CustomScriptType::class, [
            'customScript' => $customScript, 'deviceTypes' => $deviceTypes, 'agentProfiles' => $agentProfiles
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            // $form->getData() holds the submitted values
            // but, the original `$task` variable has also been updated
            /** @var CustomScript $formData */
            $formData = $form->getData();

            //dd($formData);

            $devTypeCollection = new ArrayCollection();
            $agentProfileCollection = new ArrayCollection();

            foreach($formData['deviceType'] as $deviceTypeId) {
                $deviceType =  $managerRegistry->getRepository(DeviceType::class)->findOneBy(['id' => $deviceTypeId]);
                $devTypeMap = new CustomScriptDevTypeMap();
                $devTypeMap->setScript($customScript);
                $devTypeMap->setDevType($deviceType);
                $devTypeCollection->add($devTypeMap);
            }

            foreach($formData['agentProfile'] as $agentProfileId) {
                $agentProfile =  $managerRegistry->getRepository(AgentProfile::class)->findOneBy(['id' => $agentProfileId]);
                $agentProfileMap = new CustomScriptAgentProfileMap();
                $agentProfileMap->setScript($customScript);
                $agentProfileMap->setAgentProfile($agentProfile);
                $agentProfileCollection->add($agentProfileMap);
            }

            try {

                $customScript->setDescription($formData['description']);
                $customScript->setCode($formData['code']);
                $customScript->setDevTypeMaps($devTypeCollection);
                $customScript->setAgentProfileMap($agentProfileCollection);
                $em->persist($customScript);
                $em->flush();

            }catch(Exception $e){

            }

            $this->addFlash(
                'success',
                'Alert was created.'
            );

            return $this->redirectToRoute('admin_custom_script_index', []);

        }

        return $this->render("customScript/createCustomScript.html.twig", [
            'form' => $form->createView(),
        ]);

    }

    #[Route(path: '/edit/{customScriptId}', name: 'edit', methods: ['GET', 'POST'])]
    public function editCustomScript(int $customScriptId, Request $request, ManagerRegistry $managerRegistry,
                              EntityManagerInterface $em): RedirectResponse|Response
    {

        $customScript = $managerRegistry->getRepository(CustomScript::class)->findOneBy(
            ['id' => $customScriptId, 'deleted' => 0]
        );

        if(!$customScript){

            throw new CustomScriptNotFoundException();

        }

        $deviceTypes = $managerRegistry->getRepository(DeviceType::class)->findAll();
        $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findAll();

        $form = $this->createForm(CustomScriptType::class, [
            'customScript' => $customScript, 'deviceTypes' => $deviceTypes, 'agentProfiles' => $agentProfiles
        ]);
        $deleteForm = $this->createForm(DeleteType::class, ['text' => 'Delete Custom script']);

        $form->handleRequest($request);
        $deleteForm->handleRequest($request);

        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {

            try{

                $customScript->setDeleted(true);
                $em->flush();

                $this->addFlash(
                    'success',
                    'Custom script was successfully deleted.'
                );

            }catch (\Exception){

                $this->addFlash(
                    'danger',
                    'Custom script was not deleted.'
                );

            }

            return $this->redirectToRoute('admin_custom_script_index');

        }

        if ($form->isSubmitted() && $form->isValid()) {

            /** @var CustomScript $formData */
            $formData = $form->getData();

            $devTypeCollection = new ArrayCollection();
            $agentProfileCollection = new ArrayCollection();

            $currentDeviceTypes = [];
            $currentAgentProfiles = [];
            $deviceTypesMap = $customScript->getDevTypeMaps();
            $agentProfilesMap = $customScript->getAgentProfileMap();

            foreach ($deviceTypesMap as $deviceTypeMap) {
                $currentDeviceTypes[$deviceTypeMap->getDevType()->getId()] = $deviceTypeMap->getDevType();
            }
            foreach ($agentProfilesMap as $agentProfile) {
                $currentAgentProfiles[$agentProfile->getAgentProfile()->getId()] = $agentProfile->getAgentProfile();
            }

            //dump($currentDeviceTypes);

            foreach($formData['deviceType'] as $deviceTypeId) {

                //dump('if', $deviceTypeId, $currentDeviceTypes);
                //if not exists
                if(!array_key_exists($deviceTypeId, $currentDeviceTypes)){
                    $deviceType =  $this->managerRegistry->getRepository(DeviceType::class)->findOneBy(['id' => $deviceTypeId]);
                    $devTypeMap = new CustomScriptDevTypeMap();
                    $devTypeMap->setScript($customScript);
                    $devTypeMap->setDevType($deviceType);
                    $devTypeCollection->add($devTypeMap);
                }

                //Remove processed from array
                if (isset($currentDeviceTypes[$deviceTypeId])) {
                    unset($currentDeviceTypes[$deviceTypeId]);
                }

            }

            foreach($formData['agentProfile'] as $agentProfileId) {

                //if not exists
                if(!array_key_exists($agentProfileId, $currentAgentProfiles)){
                    $agentProfile =  $this->managerRegistry->getRepository(AgentProfile::class)->findOneBy(['id' => $agentProfileId]);
                    $agentProfileMap = new CustomScriptAgentProfileMap();
                    $agentProfileMap->setScript($customScript);
                    $agentProfileMap->setAgentProfile($agentProfile);
                    $agentProfileCollection->add($agentProfileMap);
                }

                //Remove processed from array
                if (isset($currentAgentProfiles[$agentProfileId])) {
                    unset($currentAgentProfiles[$agentProfileId]);
                }

            }

            try {

                $em->getConnection()->beginTransaction();

                $customScript->setDescription($formData['description']);
                $customScript->setCode($formData['code']);
                $customScript->setDevTypeMaps($devTypeCollection);
                $customScript->setAgentProfileMap($agentProfileCollection);
                $em->persist($customScript);
                $em->flush();

                foreach ($currentDeviceTypes as $currentDeviceTypeToDelete) {

                    $customScriptDevTypeMapping = $managerRegistry->getRepository(CustomScriptDevTypeMap::class)->findOneBy(
                        ['script' => $customScript, 'devType' => $currentDeviceTypeToDelete]);

                    if ($customScriptDevTypeMapping) {

                        $em->remove($customScriptDevTypeMapping);
                        $em->flush();

                    }

                }

                foreach ($currentAgentProfiles as $currentAgentProfileToDelete) {

                    $customScriptAgentProfileMapping = $managerRegistry->getRepository(CustomScriptAgentProfileMap::class)->findOneBy(
                        ['script' => $customScript, 'agentProfile' => $currentAgentProfileToDelete]);

                    if ($customScriptAgentProfileMapping) {

                        $em->remove($customScriptAgentProfileMapping);
                        $em->flush();

                    }

                }

                $em->getConnection()->commit();

                $this->addFlash(
                    'success',
                    'Custom script was updated.'
                );

            }catch (UniqueConstraintViolationException $e){

                $em->getConnection()->rollBack();
                $this->addFlash(
                    'danger',
                    'Custom script was not updated.'
                );

            }catch (\Exception $e){

                $em->getConnection()->rollBack();
                $this->addFlash(
                    'danger',
                    'Custom script was not updated.'
                );

            }

            return $this->redirectToRoute('admin_custom_script_index', []);

        }

        return $this->render("customScript/editCustomScript.html.twig", [
            'form' => $form->createView(),
            'deleteForm' => $deleteForm->createView(),
        ]);

    }

}
