<?php

declare(strict_types=1);

namespace App\Controller\Admin\Manager;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Controller\Admin\BaseController;
use App\Datatables\AgentProfileDatatable;
use App\Entity\AgentProfile\AgentProfile;
use App\Exception\AgentProfileNotFoundException;
use App\Form\AgentProfileType;
use App\Repository\TypesValueTypeRepository;
use App\Service\Api\AgentProfiles;
use Doctrine\Persistence\ManagerRegistry;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/agent-profile', name: 'admin_agent_profile')]
class AgentProfileController extends BaseController
{

    public function __construct(private readonly AgentProfiles $agentProfilesApi,
                                private readonly TypesValueTypeRepository $typesValueTypeRepository,
                                ManagerRegistry $managerRegistry)
    {

        parent::__construct($managerRegistry);
    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '', name: '_index')]
    public function default(Request $request, ManagerRegistry $managerRegistry,
                            DatatableFactory $factory, DatatableResponse $responseService): Response {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        //$userModules = $managerRegistry->getRepository(UserModule::class)->findAll();

        $datatable = $factory->create(AgentProfileDatatable::class);
        $datatable->buildDatatable(/*['userModules' => $userModules]*/);

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            //$qb->where('usermodule.hidden = :hidden');
            //$qb->setParameter('hidden', '0');

            return $responseService->getResponse();
        }

        return $this->render('manager/agentProfile/index.twig', ['datatable' => $datatable, 'filter' => $this->filter]);

    }

    /**
     * @param Request $request
     * @return RedirectResponse|Response
     */
    #[Route(path: '/add', name: '_add', methods: ['GET', 'POST'])]
    public function addAgentProfile(Request $request): RedirectResponse|Response
    {

        $profileData = $this->getProfileFormData(AgentProfile::DEFAULT_AGENT_PROFILE_ID);

        if (!$profileData) {
            throw new AgentProfileNotFoundException();
        }

        $profileData['action'] = 'add';
        $profileData['nameEditDisabled'] = false;
        $form = $this->createForm(AgentProfileType::class, $profileData);

        $inputDataFormNames = array_map(fn($ar) => $ar['name'], $profileData['configuration']);
        $inputDataFormNames[] = 'NAME'; //Add name to the array with all items

        $form->handleRequest($request);

        if($form->isSubmitted() && $form->isValid()) {

            $formData = $form->getData();
            $toUpdateApiData = [];
            foreach ($inputDataFormNames as $key => $inputName){

                $toUpdateApiData[$inputName] = $formData[$inputName];

            }

            $result = $this->agentProfilesApi->createAgentProfile($toUpdateApiData);

            if($result){
                if ($result->operation_result->Code < 0) {

                    $this->addFlash(
                        'danger',
                        $result->operation_result->Description ,
                    );

                }else{

                }

            }else{

                $this->addFlash(
                    'danger',
                    'Cannot update: ' . $profileData['NAME'] ,
                );

            }

            $this->addFlash(
                'success',
                'Settings where saved.',
            );

            return $this->redirectToRoute('admin_agent_profile_index');

        }

        return $this->render('manager/agentProfile/add.html.twig', [
            'agentProfileForm' => $form->createView(),
        ]);

    }

    /**
     * @param int $profileId
     * @param Request $request
     * @return RedirectResponse|Response
     * @throws AgentProfileNotFoundException
     */
    #[Route(path: '/edit/{profileId}', name: '_edit', methods: ['GET', 'POST'])]
    public function editAgentProfile(int $profileId, Request $request): RedirectResponse|Response
    {

        $profileData = $this->getProfileFormData($profileId);

        if (!$profileData) {
            throw new AgentProfileNotFoundException();
        }

        $profileData['action'] = 'edit';
        $profileData['nameEditDisabled'] = $profileId === 1 || $profileId === 2 || $profileId === 3;
        //dump($profileData);
        $form = $this->createForm(AgentProfileType::class, $profileData);

        $inputDataFormNames = array_map(fn($ar) => $ar['name'], $profileData['configuration']);
        $inputDataFormNames[] = 'NAME'; //Add name to the array with all items

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $formData = $form->getData();

            $toUpdateApiData = [];
            foreach ($inputDataFormNames as $key => $inputName){

                $toUpdateApiData[$inputName] = $formData[$inputName];

            }

            $result = $this->agentProfilesApi->setAgentProfile($profileId, $toUpdateApiData);
            if ($result) {
                if ($result->operation_result->Code < 0) {

                    $this->addFlash(
                        'danger',
                        $result->operation_result->Description ,
                    );

                }else{

                }

            }else{

                $this->addFlash(
                    'danger',
                    'Cannot update: ' . $profileData['NAME'] ,
                );

            }

            $this->addFlash(
                'success',
                'Settings where saved.',
            );

            return $this->redirectToRoute('admin_agent_profile_index');

        }

        return $this->render('manager/agentProfile/edit.html.twig', [
            'agentProfileForm' => $form->createView(),
        ]);

    }

    /**
     * @param int $profileId
     * @return bool|array
     */
    private function getProfileFormData(int $profileId): bool|array
    {

        $profile = $this->agentProfilesApi->getAgentProfile($profileId);
        $agentProfile = $this->managerRegistry->getRepository(AgentProfile::class)->findOneBy(['id' => $profileId]);

        if($profile && $profile->operation_result->Code === -15){

            //Profile not found
            return false;

        }

        $types = $this->typesValueTypeRepository->getAllTypes();

        // Cannot get data from API
        if (!$profile) {
            return [];
        }

        $configFormData = [];
        $configFormData['name'] = $agentProfile->getProfileName();

        foreach ($profile->configuration as $configItem) {
            // dump($configItem);
            $configFormData['configuration'][] = [
                'name' => $configItem->Option,
                'label' => $configItem->Caption,
                'description' => $configItem->Description,
                'type' => $types[$configItem->Type]['name'],
                //'value' => $configItem->{'User Value'},
                'actualValue' => $configItem->{'Actual Value'},
                //'unit' => $configItem->Unit,
                //'writable' => (bool)$configItem->Writable,
                'default' => $configItem->Default,
                'min' => $configItem->Minimum,
                'max' => $configItem->Maximum,
            ];
        }

        return $configFormData;

    }

}