<?php

namespace App\Repository;

use App\Entity\DeviceStats;
use App\Entity\DeviceSyncStatus;
use App\Entity\MonitoringGroupMapping;
use App\Entity\VDevice;
use App\Entity\TunnelWeb;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\DBAL\ParameterType;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @method VDevice|null find($id, $lockMode = null, $lockVersion = null)
 * @method VDevice|null findOneBy(array $criteria, array $orderBy = null)
 * @method VDevice[]    findAll()
 * @method VDevice[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class VDeviceRepository extends ServiceEntityRepository
{

    /**
     * @param ManagerRegistry $registry
     */
    public function __construct(ManagerRegistry $registry){

        parent::__construct($registry, VDevice::class);

    }

    /**
     * @param Device $device
     * @return int|mixed|string
     */
    public function getDevicesExceptItself(VDevice $device){

        return $this->createQueryBuilder('p')
            ->where('p.id != :val')
            ->setParameter('val', $device)
            ->getQuery()
            ->getResult()
            ;

    }

    /**
     * @param int $deviceId
     * @param int $devTypeId
     * @return array
     * @throws \Doctrine\DBAL\Exception
     */
    public function getDevices(): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid FROM devices
            WHERE deleted = 0
            ORDER BY devid
            ';

        return $conn->executeQuery($sql)->fetchAllAssociative();

    }

    /**
     * @param int $deviceId
     * @param int $devTypeId
     * @return array
     * @throws \Doctrine\DBAL\Exception
     */
    public function getDevicesExcept(array $deviceIds, int $devTypeId): array {

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid, name FROM devices
            WHERE id_device not IN(:deviceIds) AND dev_type = :devTypeId AND deleted = 0
            ORDER BY devid
            ';

        return $conn->executeQuery($sql, array('deviceIds' => $deviceIds, 'devTypeId' => $devTypeId), ['deviceIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY])->fetchAllAssociative();

    }

    /**
     * @param $id
     * @param $name
     * @return Device|null
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateName(int $id, string $name){

        $entityManager = $this->getEntityManager();

        $device = $this->findOneBy(array('id'=> $id));
        $device->name = $name;

        $entityManager->persist($device);
        $entityManager->flush();

        return $device;

    }

    /**
     * @param array $order
     * @param string|null $filter
     * @return \Doctrine\ORM\Query
     */
    public function findByOrder(array $order, string $filter = null){

        $query =  $this->createQueryBuilder('p')
            // ->from('device', 'd')
            ->join('p.devType', 'devt')
            ->join('p.osVer', 'devosv')
            ->join('devosv.osTypes', 'ost')
            //->orderBy('p.id', 'ASC')
            ->orderBy($order[0], $order[1]);

        if(isset($filter)){

            $query->where('p.name LIKE :filter')
            ->setParameter('filter', '%'. $filter . '%');

        }

      return $query->getQuery();

    }

    /**
     * @param $mntGroupId
     * @return int|mixed|string
     */
    public function findAllNotInMonitoringGroup(int $mntGroupId){

        // get an ExpressionBuilder instance, so that you
        $expr = $this->_em->getExpressionBuilder();

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->where(
                $expr->notIn(
                    'd.id',
                    $this->_em->createQueryBuilder()
                        ->select('IDENTITY(mgm.device)')
                        ->from(MonitoringGroupMapping::class, 'mgm')
                        ->where('mgm.monitoringGroup = :mntGroupId')
                        ->getDQL()
                )
            )
            ->andWhere('d.deleted = :deleted')

            ->setParameter('deleted', '0')
            ->setParameter('mntGroupId', $mntGroupId)
        ;

        /*
        // create a subquery in order to take all address records for a specified user id
        $sub = $this->_em->createQueryBuilder()
            ->select('mgm.device')
            ->from(MonitoringGroupMapping::class, 'mgm')
            ->where('mgm.monitoringGroup = :mntGroupId');

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->where($expr->not($expr->exists($sub->getDQL())))
            ->setParameter('mntGroupId', $mntGroupId);
        */
        return $qb->getQuery()->getResult();

    }

    public function findAllNotInMonitoringGroupPlainSql(int $mntGroupId){

        $conn = $this->getEntityManager()->getConnection();

        $sql = "SELECT id_device, devid, name
                    FROM ". Device::ENTITY_TABLE_NAME ."
                    WHERE id_device NOT IN (
                    SELECT id_device FROM monitoring_group_device_map WHERE mongrp_id = :mntGroupId
                    ) AND deleted = 0";

        return $conn->executeQuery($sql, array('mntGroupId' => $mntGroupId), ['deviceIds' => \PDO::PARAM_INT])->fetchAllAssociative();


    }

    /**
     * @param $webTunnelId
     * @return int|mixed|string
     */
    public function findAllNotAssignedToWebTunnel(){

        $expr = $this->_em->getExpressionBuilder();

        $qb = $this->_em->createQueryBuilder()
            ->select('d')
            ->from(Device::class, 'd')
            ->join('d.stats', 'stats')
            ->join('stats.syncStatus', 'status')
            ->where(
                $expr->notIn(
                    'd.id',
                    $this->_em->createQueryBuilder()
                        ->select('IDENTITY(tw.device)')
                        ->from(TunnelWeb::class, 'tw')
                        ->where('tw.device is not null')
                        ->getDQL()
                )
            )
            ->andWhere('status.id != :statusId')
            ->andWhere('d.deleted = :deleted')

            ->setParameter('statusId', '1')
            ->setParameter('deleted', '0')
        ;

        return $qb->getQuery()->getResult();

    }

    /**
     * @return \mixed[][]
     * @throws \Doctrine\DBAL\Exception
     */
    public function findAllNotAssignedToWebTunnelPlainSql(){

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT d.id_device, d.devid, d.name
                FROM '. Device::ENTITY_TABLE_NAME .' d 
                INNER JOIN device_stats ds ON d.id_device = ds.id_device 
                INNER JOIN device_sync_statuses dss ON ds.sync = dss.id 
                WHERE d.id_device NOT IN (
                    SELECT id_device FROM tunnel_web WHERE id_device IS NOT NULL
                ) AND dss.id <> 1 AND d.deleted = 0
            ';

        return $conn->executeQuery($sql)->fetchAllAssociative();

    }

    /**
     * @param $deviceId
     * @return Device|null
     * @throws \Doctrine\DBAL\ConnectionException
     * @throws \Doctrine\DBAL\Driver\Exception
     * @throws \Doctrine\DBAL\Exception
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteDevice(int $deviceId){

        $entityManager = $this->getEntityManager();
        $conn = $this->getEntityManager()->getConnection();

        $entityManager->getConnection()->beginTransaction(); // suspend auto-commit

        try{

            $device = $this->findOneBy(array('id'=> $deviceId));

            $sql = '
            UPDATE '. Device::ENTITY_TABLE_NAME .' SET config_profile = NULL 
            WHERE config_profile = (
                SELECT automatic_profile FROM (SELECT * FROM '. Device::ENTITY_TABLE_NAME .') a WHERE id_device = :deviceId)
            ';
            $stmt = $conn->prepare($sql);
            $stmt->executeQuery(array('deviceId' => $deviceId));

            $sql = 'DELETE FROM '. MonitoringGroupMapping::ENTITY_TABLE_NAME .' WHERE id_device = :deviceId';
            $stmt = $conn->prepare($sql);
            $stmt->executeQuery(array('deviceId' => $deviceId));

            $device->setDeleted(true);

            $entityManager->persist($device);
            $entityManager->flush();

            $entityManager->getConnection()->commit();

        } catch (\Exception $e) {

            $entityManager->getConnection()->rollBack();
            throw $e;

        }

        return $device;

    }

    /**
     * @param array $devicesId
     * @return \mixed[][]
     * @throws \Doctrine\DBAL\Exception
     */
    public function getByDevIds(array $devicesId){

        $conn = $this->getEntityManager()->getConnection();

        $sql = '
            SELECT id_device, devid FROM '. Device::ENTITY_TABLE_NAME .'
            WHERE id_device IN (?) 
            ';

        return $conn->executeQuery($sql, array($devicesId), array(\Doctrine\DBAL\Connection::PARAM_INT_ARRAY))->fetchAllAssociative();

    }

    /**
     * @param int $configProfileId
     * @param array $devicesId
     * @return \mixed[][]
     * @throws \Doctrine\DBAL\Exception
     */
    public function updateDevicesConfigProfile(?int $configProfileId, array $devicesId){

        $conn = $this->getEntityManager()->getConnection();

        //Increment Gui version for device and set status to beeing processed
        $sql = '
            UPDATE '. Device::ENTITY_TABLE_NAME .' d 
            INNER JOIN '. DeviceStats::ENTITY_TABLE_NAME .' ds ON ds.id_device = d.id_device
            SET d.config_profile = ?, d.gui_version = d.gui_version + 1, ds.sync = ?
            WHERE d.id_device IN (?) 
            ';

        return $conn->executeQuery($sql, array($configProfileId, DeviceSyncStatus::BEING_PROCESSED, $devicesId),
            array((is_int($configProfileId)) ? ParameterType::INTEGER : ParameterType::NULL,
                ParameterType::INTEGER, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY))->fetchAllAssociative();


    }

    /**
     * @return array
     * @throws \Doctrine\DBAL\Exception
     */
    public function getDevicesToClonedFrom(): array {

        $conn = $this->getEntityManager()->getConnection();

        //WHERE d.automatic is not null AND d.dev_type = :devTypeId AND d.deleted = 0
        $sql = '
            SELECT d.id_device, d.devid, d.name FROM devices as d
            WHERE d.automatic_profile is not null AND d.deleted = 0
            ';

        return $conn->executeQuery($sql, [], ['deviceIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY])->fetchAllAssociative();

    }

}