#!/bin/bash

# upgrade script from 1.0.6 to 2.0.0
SNAME="U_106_200_PARTIAL"
STXT="   ---> $SNAME <---   "
echo "Upgrade script $SNAME starting up..."

[ $# -eq 1 ] || {
	echo "Need a path to existing Dwarfguard deployment." >&2
	exit 1
}

BASEDEFS="base_defs"
FUNCS="deploy_funcs.sh"
MYSQOPT=${MYSQOPT:-}
[ -n "$MYSQL_PWD" ] && MYSQOPT="$MYSQOPT -p$MYSQL_PWD"
export MYSQOPT
# get old directory from 1st parameter
DWARFG_OLD_DIR="$1"

# get new directory from $0
DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

g_tardef="$DIR/$BASEDEFS"
[[ -f "$DWARFG_OLD_DIR/base_defs" && -f "$g_tardef" ]] || {
	echo "Either $DWARFG_OLD_DIR/base_defs or $DIR/$BASEDEFS are missing." >&2
	exit 1
}

g_tmpfile="$(mktemp)"
g_usertable_cached=""

# pick the old definitions first...
. "$DWARFG_OLD_DIR/base_defs" || {
	echo "Unable to source old definitions file ($DWARFG_OLD_DIR/base_defs)" >&2
	exit 1
}

[ -n "$DWARFG_HOME" ] || {
	echo "Dwarfguard home variable (DWARFG_HOME) is empty. Possibly corrupted base definitions file. Restore from backup or perform manual recovery steps." >&2
	exit 1
}

g_upg_log="$DWARFG_HOME/upgrade_log.txt"
echo "$STXT Creating upgrade log $g_upg_log"

echo "WARNING: This is PARTIAL upgrade meaning that:"
echo "   * All actual and historic data for your devices will be LOST"
echo "   * The agents on devices SHOULD be able to upgrade"
echo "   * As a result, all devices SHOULD appear under their old Device IDs"
echo "   * All the users will be removed and factory ones created."
echo "   * Use install options to specify initial admin name+password."
echo "   * While the process has been tested, major upgrade is always risky"
echo " Interrupt this script (CTRL-C) if you do not want to loose the data."
read
dlmsg="Data loss during upgrade accepted by admin."
echo -e "\n$STXT $dlmsg" >>"$g_upg_log"

# create upgrade log in dwarfg home
echo -e "\n$STXT $(date -Iseconds) Executing PARTIAL upgrade using $0" >>"$g_upg_log"


UPGRADING=1
export UPGRADING

FORCE_DB_CREATION="yes"
export FORCE_DB_CREATION

mysql $MYSQOPT <<EOF
exit
EOF
[ $? -eq 0 ] || {
	echo "Unable to run mysql under root. You can specify your password in the MYSQL_PWD enviroment variable" | tee -a "$g_upg_log" >&2
	exit 1
}

# update new definitions
echo "$STXT updating definitions in $g_tardef" | tee -a "$g_upg_log"
sed -i --follow-symlinks "s#^\(DOMAIN\)=.*#\1=\"$DOMAIN\"#" $g_tardef
sed -i --follow-symlinks "s#^\(EXTERNURL\)=.*#\1=\"$EXTERNURL\"#" $g_tardef
sed -i --follow-symlinks "s#^\(SERVID\)=.*#\1=\"$SERVID\"#" $g_tardef
sed -i --follow-symlinks "s#^\(PORT_OFFSET\)=.*#\1=\"$PORT_OFFSET\"#" $g_tardef
sed -i --follow-symlinks "s#^\(SERV_TUNSSH_PORT\)=.*#\1=\"$SERV_TUNSSH_PORT\"#" $g_tardef
sed -i --follow-symlinks "s#^\(USE_SSL\)=.*#\1=\"$USE_SSL\"#" $g_tardef
sed -i --follow-symlinks "s#^\(DWARFG_PORT\)=.*#\1=\"$DWARFG_PORT\"#" $g_tardef
sed -i --follow-symlinks "s#^\(BASEDIR\)=.*#\1=\"$BASEDIR\"#" $g_tardef
sed -i --follow-symlinks "s#^\(SYSCTLPATH\)=.*#\1=\"$SYSCTLPATH\"#" $g_tardef
sed -i --follow-symlinks "s#^\(AGENT_FORCED_SECURITY\)=.*#\1=\"$AGENT_FORCED_SECURITY\"#" $g_tardef
cp -r $DWARFG_OLD_DIR/dwarfg_license.lic $DWARFG_OLD_DIR/myCA* $DWARFG_OLD_DIR/server.* $DWARFG_OLD_DIR/ssl $DIR/ 2>/dev/null

# run cleanup
echo "$STXT calling cleanup (upgrade mode) $DWARFG_OLD_DIR/cleanup.sh" | tee -a "$g_upg_log"
"$DWARFG_OLD_DIR/cleanup.sh" --upgrade --yes || {
	echo "Error when running $DWARFG_OLD_DIR/cleanup.sh, trying to continue with upgrade anyway." | tee -a "$g_upg_log" >&2
}

if mysqldump "$DWARFG_DBN" user >"$g_tmpfile"; then
	g_usertable_cached="yes"
else
	echo "Failed to dump existing user data from the DB. Factory users and factory pre-set passwords i(and deploy-used cmdline options) will be used."
fi

echo "Removing DB data..."
echo "DROP DATABASE IF EXISTS \`$DWARFG_DBN\`;" | mysql $MYSQOPT

# switch to new env
echo "$STXT switching to new env ($g_tardef)" | tee -a "$g_upg_log"
. "$g_tardef" || {
	echo "Unable to source new definitions file ($g_tardef)" | tee -a "$g_upg_log" >&2
	exit 1
}
. "$DIR/$FUNCS" || {
	echo "Unable to source new function file ($DIR/$FUNCS)" | tee -a "$g_upg_log" >&2
	exit 1
}
read_defs || {
	echo "Unable to perform read_defs()." | tee -a "$g_upg_log" >&2
	exit 1
}
validate_deploy_commands || {
	echo "Failed to validate deploy commands" | tee -a "$g_upg_log" >&2
	exit 1
}

# recreate directories
echo "$STXT recreating $BINDIR, $SRVDIR..." | tee -a "$g_upg_log"
mkdir -p "$BINDIR" "$SRVDIR"
chown "$APPUSER" "$BINDIR"
touch "$BINDIR/upgrading"


# deploy (using upgrading mode)
echo "$STXT Executing $DIR/deploy.sh $PARAM_UPGRADE $PARAM_FORCE ... (DOM/EURL is $DOMAIN/$EXTERNURL)" | tee -a "$g_upg_log"
"$DIR/deploy.sh" "$PARAM_UPGRADE" "$PARAM_FORCE"

# restore the user table if cached
[ -n "$g_usertable_cached" ] && {
	echo "Restoring the users and their passwords..."
	if mysql "$DWARFG_DBN" <"$g_tmpfile"; then
		rm -f "$g_tmpfile"
	else
		echo "Failed to restore the users, file containing SQL and data (users, roles and encrypted passwords) kept as: '$g_tmpfile'"
	fi
}

# check if roles messed up and fix
echo "Attempting to fix possibly mangled user roles..."
echo | "$DIR/upgrade_fix_user_roles.sh"

event_log 309 MSG="NOTE (before upgrade init): $dlmsg"

# start service
echo "$STXT Starting Dwarfguard service: ${DWARFG_NAM}..." | tee -a "$g_upg_log"
systemctl start "${DWARFG_NAM}.service" || {
	echo "Unable to start Dwarfguard service after upgrade" | tee -a "$g_upg_log" >&2
	exit 1
}

echo -e "$SNAME: $(date -Iseconds) Upgrade by $0 finished\n" | tee -a "$g_upg_log"
event_log 309 MSG="Upgrade finished"
