#!/bin/bash

FUNCS="deploy_funcs.sh"

command -v dirname >/dev/null 2>&1 || {
	echo "\"dirname\" command not available, bailing out." >&2
	exit 1
}

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR"/"$FUNCS" || {
	echo "Unable to read deploy functions, exiting..." >&2
	exit 1
}

BASEDEFS="base_defs"
ERR=0

. "$DIR/$BASEDEFS" || . "$BASEDEFS" || { echo "Fatal: Definition file ($BASEDEFS) not found." >&2; exit 1; }
. "$DIR/$BASEFUNCS" || . "$BASEFUNCS" || { echo "Fatal: Basic functions ($BASEFUNCS) not found." >&2; exit 1; }

RMV=0
RMV_TARGET=2
KEEP_CACHE=${KEEP_CACHE:-0}
FORCE=${FORCE:-0}

print_help() {
	echo -e "\nHi there, this is $NAME cleanup script.\nThe purpose of this script is to (partially) cleanup $NAME installation from your server\n\nAllowed parameters:\n"
	echo -e "\t$PARAM_FORCE ... Confirm removal/run non-interactively."
	echo -e "\t$PARAM_KEEPCACHE ... keep $DWARFG_CACHE on the server even if last instance of $NAME installation is being removed."
	echo -e "\t$PARAM_UPGRADE ... We are performing upgrade. Do not remove deployment or cache (implies $PARAM_KEEPCACHE)"
	echo
}

process_params() {
	while [ $# -gt 0 ] ; do
		case $1 in
			"$PARAM_FORCE")
				FORCE=1
				;;
			"$PARAM_KEEPCACHE")
				KEEP_CACHE=1
				;;
			"$PARAM_UPGRADE")
				UPGRADING=1
				KEEP_CACHE=1
				;;
			"$PARAM_HELP")
				print_help
				exit 0
				;;
			*)
				echo "Uncatched parameter \"$1\""
				print_help
				exit 1
				;;
		esac
		shift
	done
}

MYID=$($C_ID -u)
process_params "$@"
if [ 0 -ne "$MYID" ]; then fail_fat "Cleanup process must run as root"; fi
[ "$UPGRADING" -eq 0 ] && echo "This script will remove $NAME from this OS installation INCLUDING ANY DATA *IRREVERSIBLY*"
validate_deploy_commands || fail_nof "Unable to find some required commands, some cleanup actions may be skipped..."
[ -n "$DOMAIN" ] && echo "NOTE: using domain $DOMAIN"
[[ "$FORCE" -eq 0 && "$UPGRADING" -eq 0 ]] && {
	echo "Capital y (Y) followed by ENTER to continue, anything else to cancel this action."
	read -r RESP
	if [[ -z "$RESP" || "Y" != "$RESP" ]] ; then
		fail_fat "Removal action aborted by user."
	fi
}

# 1. stop running service
echo "Stopping $NAME service: ${DWARFG_NAM}.service (may take some time (check: ERR is $ERR)) ..."
if [ -f "$BINDIR/$DWARFG_CTL" ] ; then
	$C_SYSTEMCTL stop "${DWARFG_NAM}_${SNMP_GW}".service
	$C_SYSTEMCTL stop "${DWARFG_NAM}".service
	if [ $? -ne 0 ] ; then
		"$BINDIR/$DWARFG_CTL" --stop || appdir/"$DWARFG_CTL" --stop || fail_nof "Unable to stop $NAME."
	fi
fi
for i in "${DWARFG_NAM}.service" "${DWARFG_NAM}_${SNMP_GW}.service"; do
	if [ -f "/etc/systemd/system/$i" ] ; then
		$C_SYSTEMCTL disable "$i" || fail_nof "Could not disable $i using systemctl"
		$C_RM /etc/systemd/system/"$i" || fail_nof "Unable to remove $i from systemd"
	fi
done
$C_RM -f /etc/apache2/sites-enabled/"${DWARFG_DBN}"_proxy*.conf
$C_RM -f /etc/apache2/sites-enabled/"${DWARFG_DBN}"_sshgw*.conf
$C_RM -f /etc/apache2/sites-enabled/"${DWARFG_DBN}"_webgw*.conf
$C_RM -f /etc/apache2/sites-available/"${DWARFG_DBN}"_proxy*.conf
$C_RM -f /etc/apache2/sites-available/"${DWARFG_DBN}"_sshgw*.conf
$C_RM -f /etc/apache2/sites-available/"${DWARFG_DBN}"_webgw*.conf
[ -e /etc/apache2/sites-enabled/"$DWARFG_APACONF_AREF" ] && {
	if [ -z "$C_A2DISSITE" ] ; then
		fail_nof "Cannot disable site $DWARFG_APACONF_AREF as the a2dissite command is not available."
	else
		$C_A2DISSITE "$DWARFG_APACONF_AREF" || fail_nof "Could not disable apache site $DWARFG_APACONF_AREF"
	fi
}
$C_RM -f "$APASITESDIR/$DWARFG_APACONF_AREF"
if [ -e "$DWARFG_APAWEBROOT" ] ; then
	$C_RM "$DWARFG_APAWEBROOT" || fail_nof "Unable to remove link $DWARFG_APAWEBROOT"
fi
$APACHE_RELOAD || fail_nof "Apache reload command failed."
# 2. Remove app data when NOT UPGRADING
[ "$UPGRADING" -eq 0 ] && echo "Removing DB..." && echo "DROP DATABASE IF EXISTS \`$DWARFG_DBN\`;" | mysql $MYSQOPT
# 3. Remove app itself, counting number of installations first
NUM=$($C_LS -1 "${BASEDIR}"/"${SHORTNAME}"* | wc -l)
echo "Removing $BINDIR ..." && [ -n "$BINDIR" ] && $C_RM -rf "$BINDIR" && RMV=$((RMV+1))
echo "Removing $SRVDIR ..." && [ -n "$SRVDIR" ] && $C_RM -rf "$SRVDIR" && RMV=$((RMV+1))
$C_RM -f "/etc/logrotate.d/${DWARFG_DBN}_logrotate.conf" 2>/dev/null
# do not remove deployment when upgrading...
if [[ -d "$DEPLOYMENTS" && 0 -eq "$UPGRADING" ]] ; then
	echo "DEPLOY_POSTFIX is $DEPLOY_POSTFIX"
	MYDEPLOY=$($C_LS -1 "$DEPLOYMENTS" | $C_GREP "$DEPLOY_POSTFIX$")
	DEPLOYS=$(echo -e "$MYDEPLOY" | $C_GREP -c "^")
	[[ "$DEPLOYS" -gt 0 && -n "$MYDEPLOY" ]] && {
		echo "DEPLOYS: $DEPLOYS"
		if [ 1 -ne "$DEPLOYS" ] ; then
			echo "Detected $DEPLOYS deployments, whereas 1 deployment expected, unable to cleanup deployment..."
		else
			if [[ -n "$DEPLOYMENTS" && -n "$MYDEPLOY" ]] ; then
				$C_RM "$DEPLOYMENTS/$MYDEPLOY" || fail_nof "Unable to remove deployment ($MYDEPLOY)..."
			else
				fail_nof "Either DEPLOYMENTS ($DEPLOYMENTS) or MYDEPLOY ($MYDEPLOY) is empty, skipping deployment removal..."
			fi
		fi
	}
else
	if [ 0 -eq "$UPGRADING" ] ; then
		echo "Warning, deployments directory ($DEPLOYMENTS) not found, unable to cleanup deployment..."
	else
		echo "Upgrading mode, not removing deployment..."
	fi
fi
# remove my keys from authorized_keys file (if any)
AKEYS="$($C_GETENT passwd $APPUSER | cut -d : -f 6)/.ssh/authorized_keys"
if [ -f "$AKEYS" ] ; then
	echo "Removing tunnel ssh keys..."
	sed -i --follow-symlinks "/[^ ]*${DWARFG_DBN}_[^_ ]*$/d" "$AKEYS"
else
	echo "No tunnel ssh keys removed as ssh keys file ($AKEYS) does not exist."
fi
BACKUPS=0
[ -d "$DWARFG_BACKUP" ] && BACKUPS=$($C_LS "$DWARFG_BACKUP" | $C_GREP -c "^")
if [ "$NUM" -eq 1 ] ; then
	if [[ 1 -eq "$KEEP_CACHE" || ( 0 -eq "$FORCE" && "$BACKUPS" -gt 0 ) || 1 -eq "$UPGRADING" ]] ; then
		echo "$NAME user and cache ($APPUSER and $DWARFG_CACHE) are kept as (keep_cache || (no_force && backups || upgrading))"
	else
		echo "Cleaning up last $NAME installation:"
		echo "   1/3: removing $NAME user ($APPUSER)..."
		$C_DELUSER "$APPUSER"
		RETV=$?
		if [[ 0 -ne "$RETV" && 2 -ne "$RETV" ]] ; then
			fail_nof "Unable to remove user $APPUSER"
		fi
		echo "   2/3: dropping $NAME DB account:"
		echo "DROP USER IF EXISTS ${SHORTNAME}@localhost;" | mysql $MYSQOPT
		echo "   3/3: removing global $NAME cache ($DWARFG_CACHE)..."
		$C_RM -rf "$DWARFG_CACHE"
	fi
fi
$C_RM -f /etc/sudoers.d/"${DWARFG_DBN}" 2>/dev/null
$C_RM -f /etc/cron.d/"${DWARFG_DBN}" 2>/dev/null
if [[ $RMV_TARGET -eq $RMV && 0 -eq $ERR ]]; then
	CRES=0
	echo "Removal complete"
else
	CRES=1
	echo "Removal incomplete, look above for error(s) being reported. Summary:"
	if [ 0 -eq $ERR ]; then
	 echo -e "\t Only $RMV of $RMV_TARGET targets were removed."
 else
	 echo "Error counter is: $ERR"
	fi
fi
exit $CRES
