<?php

namespace App\Form;

use App\Entity\Alert\Alert;
use App\Entity\Alert\AlertTargetMap;
use App\Entity\Alert\AlertTargetSpec;
use App\Entity\Alert\AlertTargetType;
use App\Entity\Device\Device;
use App\Entity\Device\DeviceOsVersion;
use App\Entity\Device\DeviceSyncStatus;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\Types\LivenessMisscount;
use App\Entity\Types\LivenessTime;
use App\Entity\Types\TypesOperator;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

class AlertType extends AbstractType
{

    public const TARGET_TYPE_DEVICE = 1;
    public const TARGET_TYPE_MONITORING_GROUP = 2;
    public const TARGET_TYPE_AT_LEAST_DEVICE_GROUP = 4;

    public const INPUT_NAME_NUMFIRST = 'numFirst';
    public const INPUT_NAME_NUMSECOND = 'numSecond';

    private ?string $caller = null;

    private FormInterface $form;
	private ?Alert $alert;

	/**
	 * @param EntityManagerInterface $em
	 * @param ManagerRegistry $managerRegistry
	 */
    public function __construct(private readonly EntityManagerInterface $em,
								private readonly ManagerRegistry        $managerRegistry)
	{

    }

    /**
     * @param FormBuilderInterface $builder
     * @param array $options
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {

        $factory = $builder->getFormFactory();

        $builder->addEventListener(FormEvents::PRE_SET_DATA, array($this, 'onPreSetData'));
        $builder->addEventListener(FormEvents::PRE_SUBMIT, array($this, 'onPreSubmit'));

    }

	/**
	 * @param FormInterface $form
	 * @param Alert|null $alert
	 * @param AlertTargetType|null $targetType
	 * @param AlertTargetSpec|null $targetSpec
	 * @param $targetAmount
	 * @param Device|null $targetDevice
	 * @param MonitoringGroup|null $targetMonitoringGroup
	 * @param TypesOperator|null $firstOperator
	 * @param $numFirst
	 * @param TypesOperator|null $secondOperator
	 * @param $numSecond
	 * @param Bool $enabled
	 * @return void
	 */
    protected function addElements(FormInterface   $form, ?Alert $alert = null, ?AlertTargetType $targetType = null,
                                    ?AlertTargetSpec $targetSpec = null, $targetAmount = null,
                                    ?Device          $targetDevice = null, ?MonitoringGroup $targetMonitoringGroup = null,
                                    ?TypesOperator   $firstOperator = null, $numFirst = null,
                                    ?TypesOperator   $secondOperator = null, $numSecond = null, bool $enabled = false): void
    {

        $this->form = $form;

        if($this->caller === null){

            $this->alert = $alert;
            $this->caller = debug_backtrace()[1]['function'];

        }else{

            //Called twice, reset the form from previous state
            $form->remove('targetType');
            $form->remove('targetDevice');
            $form->remove('targetMonitoringGroup');
            $form->remove('targetSpec');
            $form->remove('targetAmount');
            $form->remove('operFirst');
            $form->remove('numFirst');
            $form->remove('numSecond');
            $form->remove('enabled');
            $form->remove('save');

            if($targetType->getId() === self::TARGET_TYPE_DEVICE){

                $form->remove('targetMonitoringGroup');

            }else{

                $form->remove('targetDevice');

            }

            if($targetType->getId() !== self::TARGET_TYPE_AT_LEAST_DEVICE_GROUP){

                $form->remove('targetAmount');

            }

        }

        $targetTypeResult = $this->em->getRepository(AlertTargetType::class)->findAll();

        $targetTypeChoices = [];
        foreach ($targetTypeResult as $item){

            if($item->getId() === 0) {
				continue;
			}
            $targetTypeChoices[$item->getName()] = $item->getId();

        }

        $form->add('name', TextType::class, [
            'required' => true,
        ])
            ->add('targetType', ChoiceType::class, [
                'label' => 'Target Type',
                'required' => true,
                'choices' => $targetTypeChoices,
                //'class' => 'App\Entity\Alert\AlertTargetType',
                'data' => ($targetType) ? $targetType->getId() : null,
                'mapped' => false,
                'placeholder' => 'Select a Target Type',
                'block_prefix' => 'modal_body',
                'attr' => ['data-row-class' => 'alert_targetType'],
            ]);

        if($targetType){

            if($targetType->getId() === self::TARGET_TYPE_DEVICE){

                $devices = $this->managerRegistry->getRepository(Device::class)->getDevices();

                $devicesChoices = [];
                $deviceChoicesAttr = [];

				foreach ($devices as $item){

					$devicesChoices[$item['devid']] = $item['id_device'];
					$deviceChoicesAttr[$item['devid']] = ['data-description' => $item['name']];

				}

                $form->add('targetDevice', ChoiceType::class, [
                    'label' => 'Target Device',
                    'required' => true,
                    'choices' => $devicesChoices,
                    'choice_attr' => $deviceChoicesAttr,
                    'placeholder' => 'Select a Device',
                    //'class' => 'AppBundle:City'
                    'data' => ($targetDevice) ? $targetDevice->getId() : null,
                    'mapped' => false,
                    'auto_initialize' => false,
                    'attr' => ['data-row-class' => 'alert_targetDevice'],
                ]);

            }elseif($targetType->getId() === self::TARGET_TYPE_MONITORING_GROUP){

                $this->getMonitoringGroupInput($targetMonitoringGroup);

            }elseif ($targetType->getId() === self::TARGET_TYPE_AT_LEAST_DEVICE_GROUP){

                $this->getMonitoringGroupInput($targetMonitoringGroup);

                $this->form->add('targetAmount', TextType::class, [
                    'label' => 'Target Amount',
                    'required' => true,
                    'data' => $targetAmount,
                    //'type' => 'integer',
                    'mapped' => false,
                    'attr' => ['data-row-class' => 'alert_targetAmount'/*, 'help'=>'text help'*/],
                ]);

            }

            $targetSpecResult = $this->em->getRepository(AlertTargetMap::class)->findBy(['targetType' => $targetType->getId()]);

            $targetSpecsChoices = [];
            foreach ($targetSpecResult as $item){

                if($item->getTargetSpec()->getId() === 0) {
					continue;
				}
                $targetSpecsChoices[$item->getTargetSpec()->getName()] = $item->getTargetSpec()->getId();

            }

            $form->add('targetSpec', ChoiceType::class, [
                'label' => 'Target Specification',
                'required' => true,
                'choices' => $targetSpecsChoices,
                'placeholder' => 'Select a Target Spec',
                //'class' => 'AppBundle:City'
                'data' => ($targetSpec) ? $targetSpec->getId() : null,
                'mapped' => false,
                'auto_initialize' => false,
                'attr' => ['data-row-class' => 'alert_targetSpec'],
            ]);

        }

        if ($targetSpec) {

            $operatorTypeResult = $this->em->getRepository(TypesOperator::class)->findAll();
            unset($operatorTypeResult[0]);

            $operatorTypeChoices = [];
            foreach ($operatorTypeResult as $item){

                $operatorTypeChoices[$item->getOperatorText() . ' ' . (($item->getOperatorText2()) ? $item->getOperatorText2() .' ' : '') .
                (($item->getOperatorText3()) ? $item->getOperatorText3() .' ' : '') . $item->getName()] = $item->getId();

            }

            $form->add('operFirst', ChoiceType::class, [
                'label' => 'Operator',
                'required' => true,
                'choices' => $operatorTypeChoices,
                'placeholder' => 'Select a Operator',
                'mapped' => false,
                'data' => ($firstOperator) ? $firstOperator->getId() : null,
                'auto_initialize' => false,
                'attr' => ['data-row-class' => 'alert_operFirst'],
            ]);

            //dump($targetSpec->getValtype()->getName());
            //dump($targetSpec->getSelType()->getTableName());

            $this->getInput($targetSpec, self::INPUT_NAME_NUMFIRST, $numFirst);

            if( $firstOperator && $firstOperator->getOperands() === 2){

                $this->getInput($targetSpec, self::INPUT_NAME_NUMSECOND, $numSecond);

            }

        }

        $form->add('enabled', CheckboxType::class, [
                'required' => false,
                'label' => 'Enabled',
                'empty_data' => 'on',
                'data' => ($enabled) ? true : false,
        ]);

        $form->add('save', SubmitType::class, [
            'label' => (isset($alert) && $alert->getId()) ? '<i class="fas fa-floppy-disk"></i> Update' : '<i class="fas fa-plus"></i> Create Alert',
            'label_html' => true,
            'block_prefix' => 'form_footer',
            'attr' => ['data-row-class' => 'alert_targetDevice', 'class' => 'btn btn-primary float-right'],
        ]);

    }

    /**
     * @param FormEvent $event
     * @return void
     */
    public function onPreSubmit(FormEvent $event): void {

        $form = $event->getForm();
        $data = $event->getData();

        $targetSpec = null;
        $operFirst = null;
        $targetAmount = null;
        $targetMonitoringGroup = null;
        $targetDevice = null;

        $targetType = $this->em->getRepository(AlertTargetType::class)->findOneBy(['id' => $data['targetType']]);
        if(isset($data['targetSpec'])){
            $targetSpec = $this->em->getRepository(AlertTargetSpec::class)->findOneBy(['id' => $data['targetSpec']]);
        }
        if(isset($data['targetMonitoringGroup'])){
            $targetMonitoringGroup = $this->em->getRepository(MonitoringGroup::class)->findOneBy(['id' => $data['targetMonitoringGroup']]);
        }
        if(isset($data['targetDevice'])){
            $targetDevice = $this->em->getRepository(Device::class)->findOneBy(['id' => $data['targetDevice']]);
        }
        if(isset($data['operFirst'])){
            $operFirst = $this->em->getRepository(TypesOperator::class)->findOneBy(['id' => $data['operFirst']]);
        }
        if(isset($data['targetAmount'])){
            $targetAmount = $data['targetAmount'];
        }

        $this->addElements($form, null, $targetType, $targetSpec, $targetAmount,
            $targetDevice, $targetMonitoringGroup, $operFirst );

    }

    /**
     * @param FormEvent $event
     * @return void
     */
    public function onPreSetData(FormEvent $event): void {

        /** @var Alert $alert */
        $alert = $event->getData();
        $form = $event->getForm();

        $targetType = $alert->getTargetType() ?: null;
        $targetSpec = $alert->getTargetSpec() ?: null;
        $targetAmount = $alert->getTargetAmount() ?: null;
        $targetDevice = $alert->getTargetDevice() ?: null;
        $targetMonitoringGroup = $alert->getTargetMonitoringGroup() ?: null;
        $firstOperator = $alert->getOperFirst() ? $alert->operFirst : null;
        $secondOperator = $alert->getOperSecond() ? $alert->operSecond : null;
        $numFirst = (!is_null($alert->getNumFirst())) ? $alert->getNumFirst(): null;
        $numSecond = (!is_null($alert->getNumSecond())) ? $alert->getNumSecond(): null;
        //If create show enabled, otherwise respect settings of entity
        $enabled = ($alert->getId()) ? $alert->getEnabled() : true;

        $this->addElements($form, $alert, $targetType, $targetSpec, $targetAmount, $targetDevice, $targetMonitoringGroup,
        $firstOperator, $numFirst, $secondOperator, $numSecond, $enabled);

    }

    /**
     * @param OptionsResolver $resolver
     */
    public function configureOptions(OptionsResolver $resolver): void
    {

        $resolver->setDefaults([
            'data_class' => Alert::class,
        ]);

    }

    /**
     * @param MonitoringGroup|null $targetMonitoringGroup
     * @return void
     */
    private function getMonitoringGroupInput(?MonitoringGroup $targetMonitoringGroup): void {

        $monitoringGroupResult = $this->em->getRepository(MonitoringGroup::class)->findAll();

        $monitoringGroupChoices = [];
        foreach ($monitoringGroupResult as $item){

            $monitoringGroupChoices[$item->getName()] = $item->getId();

        }

        $this->form->add('targetMonitoringGroup', ChoiceType::class, [
            'required' => true,
            'choices' => $monitoringGroupChoices,
            'placeholder' => 'Select a Monitoring Group',
            'data' => ($targetMonitoringGroup) ? $targetMonitoringGroup->getId() : null,
            'mapped' => false,
            'auto_initialize' => false,
            'attr' => ['data-row-class' => 'alert_targetMonitoringGroup'],
        ]);

    }

	/**
	 * @param AlertTargetSpec $targetSpec
	 * @param string $inputName
	 * @param string|null $dataValue
	 * @return void
	 */
    private function getInput(AlertTargetSpec $targetSpec, string $inputName, ?string $dataValue): void {

        $valueType = $targetSpec->getValtype()->getName();

        $label = '';

        if($inputName === self::INPUT_NAME_NUMFIRST){

            $label = 'First #';

        }elseif($inputName === self::INPUT_NAME_NUMSECOND){

            $label = 'Second #';

        }

        if($valueType === 'int' || $valueType === 'double') {

            $this->form->add($inputName, NumberType::class, [
                'label' => str_replace('#', 'Number', $label),
                'required' => true,
                'data' => $dataValue,
                //'type' => 'integer',
                'mapped' => false,
                'attr' => ['data-row-class' => 'alert_' . $inputName/*, 'help' => 'text help'*/],
            ]);

        }elseif ($valueType === 'string'){

            $this->form->add($inputName, TextType::class, [
                'label' => str_replace('#', 'Number', $label),
                'required' => true,
                'data' => $dataValue,
                'mapped' => false,
                'attr' => ['data-row-class' => 'alert_' . $inputName],
            ]);

        }elseif($valueType === 'Selection'){

            $tableName = $targetSpec->getSelType()->getTableName();

            if($tableName === $this->em->getClassMetadata(DeviceSyncStatus::class)->getTableName()){

                $result = $this->em->getRepository(DeviceSyncStatus::class)->findAll();

            }elseif ($tableName === $this->em->getClassMetadata(LivenessTime::class)->getTableName()){

                $result = $this->em->getRepository(LivenessTime::class)->findAll();

            }elseif ($tableName === $this->em->getClassMetadata(LivenessMisscount::class)->getTableName()){

                $result = $this->em->getRepository(LivenessMisscount::class)->findAll();

            }elseif($tableName === $this->em->getClassMetadata(DeviceOsVersion::class)->getTableName()){

                $result = $this->em->getRepository(DeviceOsVersion::class)->findBy(['available' => 1]);

            }

            $choices = [];
            foreach ($result as $item){

                $choices[$item->getDescription()] = $item->getId();

            }

            $this->form->add($inputName, ChoiceType::class, [
                'label' => str_replace('#', 'Value', $label),
                'required' => true,
                'placeholder' => 'Select a value',
                'choices' => $choices,
                'data' => $dataValue,
                'mapped' => false,
                'auto_initialize' => false,
                'attr' => ['data-row-class' => 'alert_'.$inputName],
            ]);

        }

    }

}