<?php

declare(strict_types=1);

namespace App\Controller\Admin;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Entity\Alert\Alert;
use App\Entity\Alert\AlertTargetSpec;
use App\Entity\Alert\AlertTargetType;
use App\Entity\Device\Device;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\Types\TypesOperator;
use App\Exception\AlertNotFoundException;
use App\Exception\MonitoringGroupNotFoundException;
use App\Form\AlertType;
use App\Form\DeleteType;
use App\Service\Api\ApiService;
use App\Service\Api\DashboardLights;
use App\Service\DatatableService;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

class AlertController extends BaseController
{

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DatatableService $datatableService
     * @param DatatableResponse $responseService
     * @return JsonResponse|Response
     * @throws \Exception
     */
    #[Route(path: '/admin/alerts', name: 'admin_alerts', methods: ['GET'])]
    public function alerts(Request $request, ManagerRegistry $managerRegistry, DatatableService $datatableService,
                           DatatableResponse $responseService): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        $datatable = $datatableService->getAlertDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('alert.deleted = :deleted');
            $qb->setParameter('deleted', '0');

            return $responseService->getResponse();
        }

        $alerts = $managerRegistry->getRepository(Alert::class)->findAll();

        return $this->render("alert/index.htm.twig", [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'alerts' => $alerts,
            'title' => 'List of alerts:',
        ]);

    }

    /**
     * @param int $monitoringGroupId
     * @param int $subsetId
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DashboardLights $dashboardLightsApi
     * @param DatatableService $datatableService
     * @param DatatableResponse $responseService
     * @return JsonResponse|Response
     * @throws MonitoringGroupNotFoundException
     * @throws \Exception
     */
    #[Route(path: 'admin/alerts/dashboard/monitoring-group/{monitoringGroupId}/subset/{subsetId}', name: 'admin_alerts_dashboard_light')]
    public function listAlertsByDashboardLight(int $monitoringGroupId, int $subsetId, Request $request,
                                                ManagerRegistry $managerRegistry, DashboardLights $dashboardLightsApi,
                                                DatatableService $datatableService, DatatableResponse $responseService): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        /** @var MonitoringGroup $monitoringGroup */
        $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroupId]);

        if(!$monitoringGroup){

            throw new MonitoringGroupNotFoundException();

        }

        $triggeredAlerts = $dashboardLightsApi->getDashboardDetailData($monitoringGroup->getId(), $subsetId);

        $datatable = $datatableService->getAlertDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            /** @var QueryBuilder $qb */
            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('alert.id IN (:alertIds)');
            $qb->setParameter('alertIds', $triggeredAlerts);

            return $responseService->getResponse();
        }

        if($subsetId === 6){
            $dashboardAlertType = 'Active';
        }elseif ($subsetId === 7){
            $dashboardAlertType = 'Past';
        }elseif ($subsetId === 8){
            $dashboardAlertType = 'Raised';
        }else{
            $dashboardAlertType = '';
        }

        return $this->render('alert/index.htm.twig', ['datatable' => $datatable, 'filter' => $this->filter, 'title' => 'List of '. $dashboardAlertType .' alerts:']);

    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param EntityManagerInterface $em
     * @return RedirectResponse|Response
     */
    #[Route(path: '/admin/alert/add', name: 'admin_alert_add', methods: ['GET', 'POST'])]
    public function addAlert(Request $request, ManagerRegistry $managerRegistry,
                             EntityManagerInterface $em): RedirectResponse|Response
    {

        $alerts = $managerRegistry->getRepository(Alert::class)->findAll();

        $alert = new Alert();

        $form = $this->createForm(AlertType::class, $alert /*['targetTypes' => $targetTypes]*/);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            // $form->getData() holds the submitted values
            // but, the original `$task` variable has also been updated
            /** @var Alert $formData */
            $formData = $form->getData();

            $alertData = $request->get('alert');

            $alert = $this->processFormData($formData, $alertData);

            $em->persist($alert);
            $em->flush();

            $this->addFlash(
                'success',
                'Alert was created.'
            );

            return $this->redirectToRoute('admin_alerts', []);

        }

        return $this->render("alert/createAlerts.html.twig", [
            'alerts' => $alerts,
            'form' => $form->createView(),
            ]);

    }

    /**
     * @param int $alertId
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param EntityManagerInterface $em
     * @return RedirectResponse|Response
     * @throws AlertNotFoundException
     */
    #[Route(path: '/admin/alert/edit/{alertId}', name: 'admin_alert_edit', methods: ['GET', 'POST'])]
    public function editAlert(int $alertId, Request $request, ManagerRegistry $managerRegistry,
                              EntityManagerInterface $em): RedirectResponse|Response
    {

        $alert = $managerRegistry->getRepository(Alert::class)->findOneBy(['id' => $alertId, 'deleted' => 0]);

        if(!$alert){

            throw new AlertNotFoundException();

        }

        $form = $this->createForm(AlertType::class, $alert);
        $deleteForm = $this->createForm(DeleteType::class, ['text' => 'Delete Alert']);

        $form->handleRequest($request);
        $deleteForm->handleRequest($request);

        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {

            try{

                $alert->setDeleted(true);
                $em->flush();

                $this->addFlash(
                    'success',
                    'Alert was successfully deleted.'
                );

            }catch (\Exception){

                $this->addFlash(
                    'danger',
                    'Alert was not deleted.'
                );

            }

            return $this->redirectToRoute('admin_alerts');

        }

        if ($form->isSubmitted() && $form->isValid()) {

            /** @var Alert $formData */
            $formData = $form->getData();

            $alertData = $request->get('alert');

            $alert = $this->processFormData($formData, $alertData);

            $em->flush();

            $this->addFlash(
                'success',
                'Alert was updated.'
            );

            return $this->redirectToRoute('admin_alerts', []);

        }

        return $this->render("alert/editAlert.html.twig", [
            'form' => $form->createView(),
            'deleteForm' => $deleteForm->createView(),
        ]);

    }

    /**
     * @return Alert
     */
    private function processFormData(Alert $formData, array $alertData): Alert {

        $targetType = $alertData['targetType'];
        if(isset($alertData['targetDevice'])){

            $targetDevice = $alertData['targetDevice'];
            $targetDevice = $this->managerRegistry->getRepository(
                Device::class)->findOneBy(['id' => $targetDevice]
            );
            $formData->setTargetDevice($targetDevice);

        }else{

            $formData->setTargetDevice(null);

        }

        if(isset($alertData['targetMonitoringGroup'])){

            $targetMonitoringGroup = $alertData['targetMonitoringGroup'];
            $targetMonitoringGroup = $this->managerRegistry->getRepository(
                MonitoringGroup::class)->findOneBy(['id' => $targetMonitoringGroup]
            );
            $formData->setTargetMonitoringGroup($targetMonitoringGroup);

        }else{

            $formData->setTargetMonitoringGroup(null);

        }

        $targetSpec = $alertData['targetSpec'];
        $targetAmount = (isset($alertData['targetAmount']) && !is_null($alertData['targetAmount'])) ? $alertData['targetAmount'] : null;
        $targetOperFirst = $alertData['operFirst'];
        $targetOperSecond = $alertData['operSecond'] ?? null;
        $targetNumFirst = $alertData['numFirst'];
        $targetNumSecond = (isset($alertData['numSecond']) && !is_null($alertData['numSecond'])) ? $alertData['numSecond'] : null;
        $enabled = (isset($alertData['enabled'])) ? true : false;

        $targetType = $this->managerRegistry->getRepository(AlertTargetType::class)->findOneBy(['id' => $targetType]);
        $targetSpec = $this->managerRegistry->getRepository(AlertTargetSpec::class)->findOneBy(['id' => $targetSpec]);
        $targetOperFirst = $this->managerRegistry->getRepository(TypesOperator::class)->findOneBy(['id' => $targetOperFirst]);
        $targetOperSecond = $this->managerRegistry->getRepository(TypesOperator::class)->findOneBy(['id' => $targetOperSecond]);

        $formData->setTargetType($targetType);
        $formData->setTargetSpec($targetSpec);
        if($targetAmount) {
			$formData->setTargetAmount((int) $targetAmount);
		}
        $formData->setOperFirst($targetOperFirst);
        if($targetOperSecond) {
			$formData->setOperSecond($targetOperSecond);
		}
        $formData->setNumFirst((float) $targetNumFirst);
        if(!is_null($targetNumSecond)){

            $formData->setNumSecond((float) $targetNumSecond);

        }

        $formData->setEnabled($enabled);

        return $formData;

    }

    /**
     * @return RedirectResponse
     * @throws AlertNotFoundException
     */
    #[Route(path: '/admin/alert/clear-counter/{alertId}', name: 'admin_alert_clear_counter', methods: ['GET'])]
    public function clearCounterForPastAlert(int $alertId, ManagerRegistry $managerRegistry, ApiService $apiService): RedirectResponse{

        $alert = $managerRegistry->getRepository(Alert::class)->findOneBy(['id' => $alertId]);

        if(!$alert){

            throw new AlertNotFoundException();

        }

        $result = $apiService->clearPastAlertCounter($alertId);

        //if($result === true){

            $this->addFlash(
                'success',
                'Alert counter was cleared.'
            );

        /*}else{

            $this->addFlash(
                'danger',
                'Alert counter cannot be cleared.'
            );

        }*/

        return $this->redirectToRoute('admin_dashboard', []);

    }

}