<?php

declare(strict_types=1);

namespace App\Controller\Admin;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\DataTable\ConfigProfileDatatable;
use App\DataTable\DeviceConfigProfileAssignedDatatable;
use App\Entity\ConfigProfile\ConfigProfile;
use App\Exception\ConfigProfileCannotBeDownloadedException;
use App\Exception\ConfigProfileNotFoundException;
use App\Form\ConfigProfile\ConfigProfileFileUploadConfirmType;
use App\Form\ConfigProfile\ConfigProfileFileUploadType;
use App\Form\ConfigProfile\ConfigProfileFromDeviceType;
use App\Form\ConfigProfile\ConfigProfileFromExistingProfileType;
use App\Service\Api\ConfigProfiles;
use App\Service\ConfigProfile\ConfigProfileService;
use App\Service\DatatableService;
use Doctrine\DBAL\ParameterType;
use Doctrine\Persistence\ManagerRegistry;
use Exception;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

#[Route('/admin/config-profile', name: 'admin_config_profile_')]
class ConfigProfileController extends BaseController
{
	/**
	 * @param Request $request
	 * @param DatatableFactory $factory
	 * @param DatatableResponse $responseService
	 * @return Response
	 * @throws Exception
	 */
    #[Route(path: '/', name: 'list')]
    public function configProfilesListAction(
        Request $request,
        DatatableFactory $factory,
        DatatableResponse $responseService
    ): Response {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $datatable = $factory->create(ConfigProfileDatatable::class);
        $datatable->buildDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

			$qb = $datatableQueryBuilder->getQb();
			$qb->andWhere('configprofile.directive = :directive');
			$qb->andWhere('configprofile.deleted = :deleted');
			$qb->setParameter('directive', true, ParameterType::INTEGER);
			$qb->setParameter('deleted', false, ParameterType::INTEGER);

            return $responseService->getResponse();
        }

        $ConfigProfileFromDeviceForm = $this->createForm(ConfigProfileFromDeviceType::class);
        $configProfileFromExistingProfileForm = $this->createForm(ConfigProfileFromExistingProfileType::class);
		$configProfileFileUploadForm = $this->createForm(ConfigProfileFileUploadType::class, null, ['label' => true] );

        return $this->render('configProfile/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'fromDeviceForm' => $ConfigProfileFromDeviceForm->createView(),
            'fromExistingProfileForm' => $configProfileFromExistingProfileForm->createView(),
            'uploadForm' => $configProfileFileUploadForm->createView()
            ]);
    }

	/**
	 * @param int $configProfileId
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 * @throws ConfigProfileNotFoundException
	 * @throws ClientExceptionInterface
	 * @throws RedirectionExceptionInterface
	 * @throws ServerExceptionInterface
	 * @throws TransportExceptionInterface
	 */
	#[Route(path: '/manage/{configProfileId}', name: 'manage')]
	public function manageConfigProfileAction(int $configProfileId, ManagerRegistry $managerRegistry, ConfigProfiles $configProfilesApi): Response{

		$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

		if(!$configProfile){

			throw new ConfigProfileNotFoundException();

		}

		try{
			$profileData = $configProfilesApi->downloadConfigProfile($configProfile->getId());
		}catch (Exception $exception){
			$profileData = null;
		}

		if(isset($profileData) && $profileData->operation_result->Code !== 0){
			throw new ConfigProfileNotFoundException();
		}

		return $this->render('configProfile/manage.html.twig', [
			'configProfile' => $configProfile,
			'profileData' => ($profileData) ? $profileData->Profile : null,
		]);

	}

	/**
	 * @throws ConfigProfileNotFoundException
	 * @throws Exception
	 */
	#[Route(path: '/assign/{configProfileId}', name: 'assign_device')]
	public function assignDeviceConfigProfileAction(int $configProfileId, Request $request, ManagerRegistry $managerRegistry,
													DatatableResponse $responseService, DatatableService $datatableService): Response{

		$isAjax = $request->isXmlHttpRequest();
		$this->filter = $request->query->get('column');

		$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->find($configProfileId);

		if(!$configProfile){

			throw new ConfigProfileNotFoundException();

		}

		$datatable = $datatableService->getDeviceDatatable(DeviceConfigProfileAssignedDatatable::class, ['configProfileId' => $configProfileId]);

		if ($isAjax) {
			$responseService->setDatatable($datatable);

			$requestParams = $this->getDatatableFilter($request, $datatable);
			$datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
			$responseService->setDatatableQueryBuilder($datatableQueryBuilder);

			$qb = $datatableQueryBuilder->getQb();
			$qb->andWhere('vdevice.configProfile = :configProfile');
			$qb->andWhere('vdevice.deleted = :deleted');
			$qb->setParameter('deleted', '0');
			$qb->setParameter('configProfile', $configProfile);

			return $responseService->getResponse();
		}

		return $this->render('configProfile/assign.html.twig', [
			'datatable' => $datatable,
			'filter' => $this->filter,
			'configProfileId' => $configProfileId,
			'configProfile' => $configProfile
		]);

	}

	/**
	 * @throws ConfigProfileNotFoundException
	 * @throws ConfigProfileCannotBeDownloadedException
	 */
	#[Route(path: '/download/{configProfileId}', name: 'download', methods: ['GET'])]
	public function downloadConfigProfileAction(Request $request, int $configProfileId, ManagerRegistry $managerRegistry,
												ConfigProfiles $configProfilesApi, ConfigProfileService $configProfileService): Response {

		$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

		if(!$configProfile){

			throw new ConfigProfileNotFoundException();

		}

		try {

			$result = $configProfilesApi->downloadConfigProfile($configProfile->getId());

			if($result->operation_result->Code === 0){
				$profileData = $result->Profile;
				$fileData = $configProfileService->getXlsxFile($profileData);
				//dump($fileData);
				$response = new Response();
				$filename = 'ConfigProfile' . $configProfile->getId() . '.xlsx';

				// Set headers
				$response->headers->set('Cache-Control', 'private');
				$response->headers->set('Content-type', 'XLXS' );
				$response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '";');
				$response->headers->set('Content-length', (string)  strlen($fileData));

				// Send headers before outputting anything
				$response->sendHeaders();

				$response->setContent( $fileData );

				return $response;

			}

		}catch (Exception $exception){

			throw new ConfigProfileCannotBeDownloadedException();

		}

	}

	/**
	 * @throws RedirectionExceptionInterface
	 * @throws DecodingExceptionInterface
	 * @throws ClientExceptionInterface
	 * @throws \JsonException
	 * @throws TransportExceptionInterface
	 * @throws ConfigProfileNotFoundException
	 * @throws ServerExceptionInterface
	 */
	#[Route(path: '/upload/{configProfileId}', name: 'upload', methods: ['GET', 'POST'])]
	public function uploadConfigProfileAction(
		Request $request,
		int $configProfileId,
		ManagerRegistry $managerRegistry,
		ConfigProfiles $configProfilesApi,
		ConfigProfileService $configProfileService
	): Response {

		$error = null;
		$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

		if(!$configProfile){

			throw new ConfigProfileNotFoundException();

		}

		$uploadForm = $this->createForm(ConfigProfileFileUploadType::class, null, ['label' => false]);
		$uploadConfirmForm = $this->createForm(ConfigProfileFileUploadConfirmType::class);
		$uploadForm->handleRequest($request);
		$uploadConfirmForm->handleRequest($request);

		if ($uploadForm->isSubmitted() && $uploadForm->isValid()) {

			/** @var UploadedFile $profileFileData */
			$profileFileData = $uploadForm->get('profile')->getData();

			if ($profileFileData) {

				$fileContent = $profileFileData->getContent();
				$data = $configProfileService->readXlsxFile($fileContent);

				if($data){
					$uploadedData = $data->rows();
					//dump($uploadedData);
					//die();

					try{
						$profileData = $configProfilesApi->downloadConfigProfile($configProfile->getId());
					}catch (Exception $exception){
						$this->addFlash(
							'danger',
							'Cannot get profile data. ' . $exception->getMessage(),
						);

						return $this->redirectToRoute(
							'admin_config_profile_manage',
							['configProfileId' => $configProfileId]
						);

					}


					$uploadConfirmForm = $this->createForm(ConfigProfileFileUploadConfirmType::class, ['profileData' => json_encode($uploadedData)]);

					//convert from stdClass to array
					$profileData = array_map(static function($x) { return (array)$x; }, $profileData->Profile);
					$uploadedData = array_map(static function($x) { return ['key' => $x[0], 'value' => $x[1]]; }, $uploadedData);

					$uploadedDataFlatten = [];
					$profileDataFlatten = [];
					foreach ($uploadedData as $settings) {
						$uploadedDataFlatten[$settings['key']] = $settings['value'];
					}

					foreach ($profileData as $settings) {
						$profileDataFlatten[$settings['key']] = $settings['value'];
					}

					$changes = $this->compareArrays($profileDataFlatten, $uploadedDataFlatten);

					//dump($changes);

					return $this->render('configProfile/compare.html.twig', [
						'configProfile' => $configProfile,
						'profileData' => ($profileData) ?: null,
						'uploadedData' => $uploadedData,
						'changes' => $changes,
						'uploadConfirmForm' => $uploadConfirmForm->createView(),
					]);
				}else{

					$error = 'Invalid profile data file given.';

				}

			}else{

				$error = 'No profile data given.';

			}

		}elseif($uploadConfirmForm->isSubmitted() && $uploadConfirmForm->isValid()) {

			try{
				$profileData = json_decode($uploadConfirmForm->getData()['profileData'], true, 512, JSON_THROW_ON_ERROR);
			}catch (Exception $exception){
				$profileData = null;
			}

			$profileDataJson = json_encode($this->makeConfigProfileDataToSend($profileData), JSON_THROW_ON_ERROR);
			//dump($profileDataJson);
			//die();

			try{
				$configProfilesApi->updateConfigProfile($configProfileId, $profileDataJson);

				$this->addFlash(
					'success',
					'Config profile was updated.'
				);

			}catch (Exception $exception){
				$this->addFlash(
					'danger',
					$exception->getMessage(),
				);
			}

			return $this->redirectToRoute(
				'admin_config_profile_manage',
				['configProfileId' => $configProfileId]
			);

		}

		return $this->render('configProfile/upload.html.twig', [
			'configProfile' => $configProfile,
			'uploadForm' => $uploadForm->createView(),
			'error' => $error,
		]);

	}

	private function makeConfigProfileDataToSend(array $profileData): array
	{

		$profileDataArray = [];

		foreach($profileData as $profileDataItem){

			$profileDataArray[] = ['key' => $profileDataItem[0], 'value' => (string) $profileDataItem[1]];

		}

		$profileDataReturn['Profile'] = $profileDataArray;
		return $profileDataReturn;

	}


	/**
	 * Compare two associative arrays and return information about changes
	 * @param array $original
	 * @param array $modified
	 * @return array|array[]
	 */
	private function compareArrays(array $original, array $modified): array
	{
		$diff = [
			'added' => [],
			'removed' => [],
			'changed' => []
		];

		// Find removed keys
		foreach ($original as $key => $value) {
			if (!isset($modified[$key])) {
				$diff['removed'][] = $key;
			}
		}

		// Find added keys
		foreach ($modified as $key => $value) {
			if (!isset($original[$key])) {
				$diff['added'][] = $key ;
			}
		}

		// Find changed keys
		foreach ($original as $key => $value) {
			if (isset($modified[$key]) && $modified[$key] != $value) {
				$diff['changed'][$key] = [
					'old' => $value,
					'new' => $modified[$key]
				];
			}
		}

		return $diff;
	}

}
