<?php

declare(strict_types=1);

namespace App\Controller\Admin;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\DataTable\DeviceInMonitoringGroupDatatable;
use App\DataTable\MonitoringGroupDatatable;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\MonitoringGroup\MonitoringGroupMapping;
use App\Exception\MonitoringGroupNotFoundException;
use App\Form\MonitoringGroupType;
use App\Service\DatatableService;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\Persistence\ManagerRegistry;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

class MonitoringGroupController extends BaseController
{

    /**
     * @param Request $request
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @return JsonResponse|Response
     * @throws \Exception
     */
    #[Route(path: 'admin/monitoring-groups', name: 'admin_mongrp')]
    public function monitoringGroups(
        Request $request,
        DatatableFactory $factory,
        DatatableResponse $responseService
    ): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $datatable = $factory->create(MonitoringGroupDatatable::class);
        $datatable->buildDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        $monitoringGroup = new MonitoringGroup();
        $form = $this->createForm(MonitoringGroupType::class, $monitoringGroup);

        return $this->render('monitoringGroup/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'form' => $form->createView()
        ]);

    }

    /**
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @param DatatableService $datatableService
     * @param int $monitoringGroupId
     * @return JsonResponse|Response
     * @throws MonitoringGroupNotFoundException
     * @throws \Exception
     */
    #[Route(path: 'admin/monitoring-group/{monitoringGroupId}', name: 'mntgrp_detail')]
    public function monitoringGroupDetail(
        Request $request,
        ManagerRegistry $managerRegistry,
        DatatableFactory $factory,
        DatatableResponse $responseService,
        DatatableService $datatableService,
        int $monitoringGroupId
    ): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->find($monitoringGroupId);

        if(!$monitoringGroup){

            throw new MonitoringGroupNotFoundException();

        }

        $datatable = $datatableService->getDeviceDatatable(DeviceInMonitoringGroupDatatable::class, ['monitoringGroupId' => $monitoringGroupId]);

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            $qb->join(MonitoringGroupMapping::class, 'mgm',  Join::WITH,
                'vdevice.id = mgm.device');
            //$qb->join('App:MonitoringGroup', 'mg',  \Doctrine\ORM\Query\Expr\Join::WITH,
            //    'mgm.monitoringGroup = mg.id');
            $qb->andWhere('mgm.monitoringGroup = :id');
            $qb->andWhere('vdevice.deleted = :deleted');
            $qb->setParameter('deleted', '0');
            $qb->setParameter('id', $monitoringGroupId);

            return $responseService->getResponse();
        }

        return $this->render('monitoringGroup/detail.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'mntGroupId' => $monitoringGroupId,
            'mntGroup' => $monitoringGroup
        ]);

    }

}