<?php

declare(strict_types=1);

namespace App\Entity\Device;

use App\Entity\AgentProfile\AgentProfile;
use App\Entity\Alert\Alert;
use App\Entity\ConfigProfile\ConfigProfile;
use App\Entity\DataChangeLog\DataChangeLog;
use App\Entity\Label\LabelDeviceMap;
use App\Entity\Label\SystemLabelDeviceMap;
use App\Entity\MonitoringGroup\MonitoringGroupMapping;
use App\Entity\Tool\AuthProfile;
use App\Entity\Tool\SimCard;
use App\Entity\Tunnel\Tunnels;
use App\Repository\VDeviceRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table(name: 'vdevices')]
#[ORM\Entity(repositoryClass: VDeviceRepository::class)]
class VDevice
{

    final public const ENTITY_TABLE_NAME = 'vdevices';

    /**
     * @var int
     */
    #[ORM\Column(name: 'id_device', type: Types::INTEGER)]
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: 'AUTO')]
    private readonly int $id;

    /**
     * @var string
     */
    #[ORM\Column(name: 'devid', type: Types::STRING, length: 8, unique: true, nullable: false)]
    public string $devId;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 32, nullable: true)]
    public ?string $name = null;

    /**
     * @var DeviceType
     */
    #[ORM\ManyToOne(targetEntity: 'DeviceType', inversedBy: 'devices')]
    #[ORM\JoinColumn(name: 'dev_type')]
    public DeviceType $devType;

	/**
	 * @var DeviceType
	 */
	#[ORM\ManyToOne(targetEntity: DeviceType::class, inversedBy: 'snmpDevices')]
	#[ORM\JoinColumn(name: 'snmp_subdevtype')]
	public DeviceType $snmpSubDevType;

    /**
     * @var DeviceOsVersion|null
     */
    #[ORM\ManyToOne(targetEntity: 'DeviceOsVersion', inversedBy: 'devices')]
    #[ORM\JoinColumn(name: 'os_ver')]
    public ?DeviceOsVersion $osVer = null;

    /**
     * @var DeviceOsVersion|null
     */
    #[ORM\ManyToOne(targetEntity: 'DeviceOsVersion', inversedBy: 'devicesDesired')]
    #[ORM\JoinColumn(name: 'os_ver_desired')]
    public ?DeviceOsVersion $osVerDesired = null;

    /**
	 * @deprecated to be deleted
     * @var string|null
     */
    //#[ORM\Column(name: 'macaddr', type: Types::STRING, length: 12, nullable: true)]
    //public ?string $macAddr = null;

    /**
     * @var string|null
     */
    #[ORM\Column(name: 'serno', type: Types::STRING, length: 32, nullable: true)]
    public ?string $serial = null;

    /**
     * @var string|null
     */
    #[ORM\Column(name: 'active_sim', type: Types::STRING, length: 32, nullable: true)]
    public ?string $activeSim = null;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 64, nullable: true)]
    public ?string $snmpname = null;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 64, nullable: true)]
    public ?string $hostname = null;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 64, nullable: true)]
    public ?string $hostnameDesired = null;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
    public ?string $netDevs;

    /**
     * @var string|null
     */
    #[ORM\Column(type: Types::STRING, length: 16, nullable: true)]
    public ?string $uuid = null;

    /**
     * @var DeviceStats
     */
    #[ORM\OneToOne(targetEntity: DeviceStats::class)]
    #[ORM\JoinColumn(name: 'id_device', referencedColumnName: 'id_device')]
    public DeviceStats $stats;

    /**
     * @var bool
     */
    #[ORM\Column(type: Types::BOOLEAN, nullable: false, options: ['default' => false])]
    public bool $deleted;

    /**
     * @var int
     */
    #[ORM\Column(type: Types::INTEGER, nullable: false)]
    public int $guiVersion;

    /**
     * @var AgentProfile
     */
    #[ORM\ManyToOne(targetEntity: AgentProfile::class, inversedBy: 'devices')]
    #[ORM\JoinColumn(name: 'agent_profile')]
    public AgentProfile $agentProfile;

    /**
     * @var AuthProfile
     */
    #[ORM\ManyToOne(targetEntity: AuthProfile::class, inversedBy: 'devices')]
    #[ORM\JoinColumn(name: 'auth_profile')]
    public AuthProfile $authProfile;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: MonitoringGroupMapping::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id')]
    public Collection $monitoringGroupMappings;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: DeviceNetDev::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $netDevices;

    /**
     * @var DeviceNote|null
     */
    #[ORM\OneToOne(mappedBy: 'device', targetEntity: DeviceNote::class, cascade: ['persist'])]
    //#[ORM\JoinColumn(name: 'id_device', referencedColumnName: 'id_device', nullable: true)]
    public ?DeviceNote $noteDevice = null;

    /**
     * @var DeviceModel|null
     */
	#[ORM\OneToOne(targetEntity: DeviceModel::class, cascade: ['persist'])]
	#[ORM\JoinColumn(name: 'model', referencedColumnName: 'id')]
	public ?DeviceModel $model = null;

    /**
     * @var DeviceCustomField|null
     */
    #[ORM\OneToOne(mappedBy: 'device', targetEntity: DeviceCustomField::class, cascade: ['persist'])]
    //#[ORM\JoinColumn(name: 'id_device', referencedColumnName: 'id_device', nullable: true)]
    public ?DeviceCustomField $customFieldDevice = null;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'targetDevice', targetEntity: Alert::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $alerts;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: LabelDeviceMap::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $labels;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: SystemLabelDeviceMap::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $systemLabels;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'deviceUser', targetEntity: SimCard::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $deviceSimCards;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'deviceLastSeen', targetEntity: SimCard::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $deviceLastSeen;

    /**
     * @var ConfigProfile|null
     */
    #[ORM\ManyToOne(targetEntity: ConfigProfile::class, cascade: ['persist'], inversedBy: 'devices')]
    #[ORM\JoinColumn(name: 'config_profile', nullable: true)]
    public ?ConfigProfile $configProfile = null;

    /**
     * @var ConfigProfile|null
     */
    #[ORM\OneToOne(mappedBy: 'automatic', targetEntity: ConfigProfile::class, cascade: ['persist'])]
    //#[ORM\JoinColumn(name: 'automatic_profile', nullable: true)]
    public ?ConfigProfile $configProfileAutomatic = null;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: Tunnels::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $webTunnels;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: DataChangeLog::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $dataChangeLogs;

    /**
     * @var Collection
     */
    #[ORM\OneToMany(mappedBy: 'device', targetEntity: DeviceUserModule::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'id_device')]
    public Collection $userModules;

    /**
     * Device constructor
     */
    public function __construct()
    {
        $this->monitoringGroupMappings = new ArrayCollection();
        $this->netDevices = new ArrayCollection();
        $this->alerts = new ArrayCollection();
        $this->labels = new ArrayCollection();
        $this->systemLabels = new ArrayCollection();
        $this->webTunnels = new ArrayCollection();
        $this->dataChangeLogs = new ArrayCollection();
        $this->userModules = new ArrayCollection();
    }

    /**
     * @return int|null
     */
    public function getId(): ?int {

        return $this->id;

    }

    /**
     * @return string|null
     */
    public function getDevId(): ?string {

        return $this->devId;

    }

    /**
     * @return string
     */
    public function getName(): ?string {

        return $this->name;

    }

    /**
     * @return DeviceType
     */
    public function getDevType(): DeviceType {

        return $this->devType;

    }

    /**
     * @return DeviceOsVersion|null
     */
    public function getOsVer(): ?DeviceOsVersion {

        return $this->osVer;

    }

    /**
     * @return DeviceOsVersion|null
     */
    public function getOsVerDesired(): ?DeviceOsVersion {

        return $this->osVerDesired;

    }

    /**
     * @param DeviceOsVersion|null $osVersion
     */
    public function setOsVerDesired(?DeviceOsVersion $osVersion): void {

        $this->osVerDesired = $osVersion;

    }

    /**
     * @return string|null
     */
    public function getSerial(): ?string {

        return $this->serial;

    }

    public function getSnmpname(): ?string {

        return $this->snmpname;

    }

    /**
     * @return string|null
     */
    public function getHostname(): ?string {

        return $this->hostname;

    }

    /**
     * @return string|null
     */
    public function getHostnameDesired(): ?string {

        return $this->hostnameDesired;

    }

    /**
     * @param string|null $hostname
     */
    public function setHostnameDesired(?string $hostname): void {

        $this->hostnameDesired = $hostname;

    }

    /**
     * @return string|null
     */
    public function getUuid(): ?string {

        return $this->uuid;

    }

    /**
     * @return DeviceStats|null
     */
    public function getStats(): ?DeviceStats {

        return $this->stats;

    }

    /**
     * @return mixed
     */
    public function getDeleted(): bool{

        return $this->deleted;

    }

    /**
     * @return int
     */
    public function getGuiVersion(): int {

        return $this->guiVersion;

    }

    /**
     * @param mixed $deleted
     */
    public function setDeleted(bool $deleted): void {

        $this->deleted = $deleted;

    }

    public function setGuiVersion(int $version): void {

        $this->guiVersion = $version;

    }

    /**
     * @return ConfigProfile|null
     */
    public function getConfigProfile(): ?ConfigProfile {

        return $this->configProfile;

    }

    /**
     * @return ConfigProfile|null
     */
    public function getAgentProfile(): ?AgentProfile {

        return $this->agentProfile;

    }

    /**
     * @return void
     */
    public function setAgentProfile(AgentProfile $agentProfile): void {

        $this->agentProfile = $agentProfile;

    }

    /**
     * @return AuthProfile
     */
    public function getAuthProfile(): AuthProfile
    {
        return $this->authProfile;
    }

    /**
     * @param AuthProfile $authProfile
     */
    public function setAuthProfile(AuthProfile $authProfile): void
    {
        $this->authProfile = $authProfile;
    }

    /**
     * @param ConfigProfile|null $configProfile
     */
    public function setConfigProfile(?ConfigProfile $configProfile): void {

        $this->configProfile = $configProfile;

    }

    /**
     * @return ConfigProfile|null
     */
    public function getConfigProfileAutomatic(): ?ConfigProfile {

        return $this->configProfileAutomatic;

    }

    /**
     * @return ArrayCollection
     */
    public function getNetDevices(): ArrayCollection {

        return $this->netDevices;

    }

    /**
     * @return DeviceNote|null
     */
    public function getNoteDevice(): ?DeviceNote
    {
        return $this->noteDevice;
    }

    /**
     * @return DeviceModel|null
     */
    public function getModel(): ?DeviceModel
    {
        return $this->model;
    }



    /**
     * @return Collection
     */
    public function getUserModules(): Collection {

        return $this->userModules;

    }

    /**
     * @return bool
     */
    public function isDeleted(): bool {

        return $this->deleted;

    }


}