<?php

namespace App\EventSubscriber;

use App\Entity\Alert\Alert;
use App\Event\AlertCreateEvent;
use App\Event\AlertEditEvent;
use App\Repository\AlertRepository;
use App\Repository\EventLogRepository;
use App\Service\Api\ApiService;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class AlertSubscriber implements EventSubscriberInterface
{

    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @var EntityManagerInterface
     */
    private EntityManagerInterface $entityManager;

    /**
     * @var AlertRepository
     */
    private AlertRepository $alertRepository;

    /**
     * @var EventLogRepository
     */
    private EventLogRepository $eventLogRepository;

    /**
     * @var ApiService
     */
    private ApiService $apiService;
	private Security $security;

	/**
     * @param LoggerInterface $logger
     * @param EntityManagerInterface $entityManager
     * @param AlertRepository $alertRepository
     * @param EventLogRepository $eventLogRepository
     * @param ApiService $apiService
     */
    public function __construct(LoggerInterface $logger, EntityManagerInterface $entityManager, Security $security,
                                AlertRepository $alertRepository, EventLogRepository $eventLogRepository,
                                ApiService $apiService)
    {
        $this->logger = $logger;
        $this->entityManager = $entityManager;
        $this->security = $security;
        $this->alertRepository = $alertRepository;
        $this->eventLogRepository = $eventLogRepository;
        $this->apiService = $apiService;
    }

    /**
     * @return array[]
     */
    public static function getSubscribedEvents(): array {
        return [
            AlertCreateEvent::NAME => ['onAlertCreateEvent', 10],
            AlertEditEvent::NAME => ['onAlertEditEvent', 20],
        ];
    }

    /**
     * @param AlertCreateEvent $event
     * @return void
     * @throws \Exception
     */
    public function onAlertCreateEvent(AlertCreateEvent $event): void {

        $this->logger->info('Alert Subscriber: Event posted id:{id}', [
            'id' => $event->getId(),
        ]);

        $this->entityManager->beginTransaction();

        $alert = $this->alertRepository->findOneBy(['id' => $event->getId()]);
        $alert->setVersion(1);

        $eventLog = $this->eventLogRepository->addEventLog(
            Alert::class, $this->security->getUser(), $alert->getId(), null,EventLogRepository::ACTION_CREATE
        );

        $this->entityManager->persist($alert);
        $this->entityManager->persist($eventLog);
        $this->entityManager->flush();

        $this->entityManager->commit();
        $this->apiService->syncCall();

    }

    /**
     * @param AlertEditEvent $event
     * @return void
     * @throws \Exception
     */
    public function onAlertEditEvent(AlertEditEvent $event): void {

        $this->logger->info('Alert Subscriber: Event posted id:{id}', [
            'id' => $event->getId(),
        ]);

        $this->entityManager->beginTransaction();

        $alert = $this->alertRepository->findOneBy(['id' => $event->getId()]);
        $version = $alert->getVersion();
        $alert->setVersion($version + 1);

        $eventLog = $this->eventLogRepository->addEventLog(
            Alert::class, $this->security->getUser(), $alert->getId(), null,EventLogRepository::ACTION_UPDATE
        );

        $this->entityManager->persist($alert);
        $this->entityManager->persist($eventLog);
        $this->entityManager->flush();

        $this->entityManager->commit();
        $this->apiService->syncCall();

    }

}