<?php

declare(strict_types=1);

namespace App\Service\Api;

use App\Service\ConfigurationService;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpClient\Exception\ClientException;
use Symfony\Component\HttpClient\Exception\RedirectionException;
use Symfony\Component\HttpClient\Exception\ServerException;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class ApiService
{
    public const URL_SCHEMA = 'http://';
    public const URL_WS_IP_ADDRESS = '127.0.0.1';
    private const URL_API_DASHBOARD = '/g/dasb';
    private const URL_API_LIST_TRIGGERED_ALERT_DEVICES = '/g/altrd';
    //get list of past devices for which the alert was triggered.
    private const URL_API_LIST_PAST_ALERT_DEVICES = '/g/alpad';
    private const URL_API_CLEAR_COUNTER_PAST_ALERTS = '/g/clral';
    private const URL_API_SYNC = '/g/synch';
    private const URL_API_DEVICE_DELETE = '/g/devde';
    private const URL_API_SEND_SMS = '/g/rasms';

    public const API_PARAM_NVAL = 'nval=';
    public const API_PARAM_TEXT = 'text=';

    public const HTTP_CLIENT_REQUEST_DEFAULT_TIMEOUT = 1; //seconds
    public const HTTP_CLIENT_REQUEST_SMS_SEND_TIMEOUT = 30; //seconds

    protected HttpClientInterface $httpClient;

    protected ConfigurationService $configurationService;

    protected LoggerInterface $logger;

    protected string $baseUrl;

    /**
     * @param HttpClientInterface $httpClient
     * @param ConfigurationService $configurationService
     * @param LoggerInterface $logger
     * @throws \Exception
     */
    public function __construct(
        HttpClientInterface $httpClient,
        ConfigurationService $configurationService,
        LoggerInterface $logger
    ) {
        $this->configurationService = $configurationService;
        $this->httpClient = $httpClient;
        $this->logger = $logger;

        $localPort = $this->configurationService->getServerLocalPort();

        $this->baseUrl = self::URL_SCHEMA . self::URL_WS_IP_ADDRESS . ':' . $localPort;
    }

    /**
     * @return object|bool
     */
    public function getDashboardData(): object|bool
    {
		//Explicitly raised the timeout
        return $this->getApiCall($this->baseUrl . self::URL_API_DASHBOARD, false, 3);
    }

    /**
     * @param int $alertId
     * @return object|false
     */
    public function getTriggeredAlertDevices(int $alertId): mixed
    {

        return $this->getApiCall($this->baseUrl . self::URL_API_LIST_TRIGGERED_ALERT_DEVICES . '?alid=' . $alertId);
    }

    /**
     * @param int $alertId
     * @return false|mixed
     */
    public function getPastAlertDevices(int $alertId): mixed
    {

        return $this->getApiCall($this->baseUrl . self::URL_API_LIST_PAST_ALERT_DEVICES . '?alid=' . $alertId);

    }

    /**
     * @param int $alertId
     * @return mixed|null
     */
    public function clearPastAlertCounter(int $alertId): mixed
    {
        //Return always false because of invalid response
        return $this->getApiCall($this->baseUrl . self::URL_API_CLEAR_COUNTER_PAST_ALERTS . '?alid=' . $alertId);
    }

    /**
     * @return false|mixed
     */
    public function syncCall(): mixed
    {

        return $this->getApiCall($this->baseUrl . self::URL_API_SYNC);
    }

    /**
     * @param int $deviceId
     * @return mixed|null
     */
    public function deleteDevice(int $deviceId): mixed
    {

        return $this->getApiCall($this->baseUrl . self::URL_API_DEVICE_DELETE . '?devid=' . $deviceId);
    }

    /**
     * @param int $deviceId
     * @param string $msisdn
     * @param int $interfaceId
     * @param string $message
     * @return mixed
     * @throws \Exception
     */
    public function sendSms(int $deviceId, string $msisdn, int $interfaceId, string $message): mixed
    {

        return $this->getApiCall($this->baseUrl . self::URL_API_SEND_SMS .
            '?devid=' . $deviceId . '&daddr=' . $msisdn . '&nval=' . $interfaceId . '&text=' . $message,
        true, self::HTTP_CLIENT_REQUEST_SMS_SEND_TIMEOUT);
    }

    /**
     * @param string $url
     * @param bool $returnException
     * @param int $timeout
     * @return mixed
     * @throws TransportExceptionInterface
     * @throws \JsonException
     * @throws \Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface
     */
    protected function getApiCall(string $url, bool $returnException = false, int $timeout = self::HTTP_CLIENT_REQUEST_DEFAULT_TIMEOUT): mixed
    {

        $this->logger->info('Called GET:' . $url, ['Api']);

        try {

            $response = $this->httpClient->request('GET', $url, [
                'timeout' => $timeout,
            ]);

            $data = json_decode($response->getContent(), null, 512, JSON_THROW_ON_ERROR);

        } catch (TransportExceptionInterface $exception) {

            $this->logger->error($exception->getMessage(), ['Api', 'ServerCall']);

            if($returnException){
                throw $exception;
            }

            return false;

        } catch (\JsonException $exception){

            $this->logger->error($exception->getMessage(), ['Api', 'JsonException']);

            if($returnException){
                throw $exception;
            }

            return false;

        } catch (\Exception $exception) {

            $this->logger->error($exception->getMessage(), ['Api', 'Error']);

            if ($returnException) {
                throw $exception;
            }

            return false;

        }

        return $data;

    }

	/**
	 * @param string $url
	 * @param string|null $jsonData
	 * @param bool $returnException
	 * @param bool $throwRequestException
	 * @param int $timeout
	 * @return mixed
	 * @throws TransportExceptionInterface
	 * @throws \JsonException
	 * @throws \Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface
	 * @throws \Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface
	 * @throws \Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface
	 */
	protected function postApiCall(string $url, ?string $jsonData = null,
								   bool $returnException = false, bool $throwRequestException = true,
								   int $timeout = self::HTTP_CLIENT_REQUEST_DEFAULT_TIMEOUT): mixed
	{

		$this->logger->info('Called POST:' . $url, ['Api']);

		try {

			$options = ['timeout' => $timeout,
						'headers' => [
							'Content-Type: application/json',
							'Accept' => 'application/json',
						]
			];

			/*dump($jsonData);
			dump($url);
			die();*/

			if($jsonData){
				//'json' => $jsonData, //Use BODY instead, escaping is done by HtttpClientTrait, $options['body'] = self::jsonEncode($options['json']);
				$options['body'] = $jsonData;
			}

			$request = $this->httpClient->request('POST', $url, $options);

			$response = $request->getContent($throwRequestException);
			//$httpLogs = $request->getInfo('debug');
			//dump($httpLogs);
			$code = $request->getStatusCode();

			if (500 <= $code) {
				throw new ServerException($request);
			}

			if (400 <= $code) {
				throw new ClientException($request);
			}

			if (300 <= $code) {
				throw new RedirectionException($request);
			}


			//dump('data');
			$data = json_decode($response, null, 512, JSON_THROW_ON_ERROR);
			//dump($data);
			//die();

		} catch (ClientException $exception) {

			$this->logger->error($exception->getMessage() . ( (isset($response)) ? ', response:' . $response : ''), ['Api', 'ServerCall ClientException']);

			//dump($code);
			//dump($response);

			//$data = json_decode($response);
			//dump($data);
			//die('tas');

			if($returnException){
				throw $exception;
			}

			return false;

		} catch (TransportExceptionInterface $exception) {

			$this->logger->error($exception->getMessage(), ['Api', 'ServerCall']);

			if($returnException){
				throw $exception;
			}

			return false;

		} catch (\JsonException $exception){

			$this->logger->error($exception->getMessage(), ['Api', 'JsonException']);

			if($returnException){
				throw $exception;
			}

			return false;

		} catch (\Exception $exception) {

			$this->logger->error($exception->getMessage(), ['Api', 'Error']);

			if ($returnException) {
				throw $exception;
			}

			return false;

		}

		return $data;


	}

}
