<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Device\DeviceOsType;
use App\Entity\Device\DeviceOsVersion;
use App\Form\Firmware\FwType;
use App\Repository\DeviceOsVersionRepository;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route(path: '/admin/ajax/fw', name: 'ajax_fw_')]
class FwController extends Controller
{
    /**
     * Create new Firmware
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: 'admin/firmware/create', name: 'create', methods: ['POST'])]
    public function createFirmwareAction(Request $request, ManagerRegistry $managerRegistry): Response
    {

        $deviceOsTypes = $managerRegistry->getRepository(DeviceOsType::class)->findAll();

        $osVersion = new DeviceOsVersion();
        $form = $this->createForm(FwType::class, ['osTypes' => $deviceOsTypes]);
        $form->handleRequest($request);
        $errors = [];

        if ($form->isSubmitted()) {
            $errorsValidator = [];
            foreach ($errorsValidator as $error) {
                $errors[] = $error->getMessage();
            }

            if (count($errors) === 0) {
                $formData = $form->getData();
                $osType = $managerRegistry->getRepository(DeviceOsType::class)->findOneBy(['id' => $formData['os_type']]);
                $em = $managerRegistry->getManager();

                try {
                    $osVersion->setDescription($formData['version']);
                    $osVersion->setOsType($osType);
                    $em->persist($osVersion);
                    $em->flush();
                } catch (UniqueConstraintViolationException) {
                    return new JsonResponse(
                        ['code' => 400, 'message' => 'error', 'errors' => ['error' => "This Firmware version already exists."]],
                        400
                    );
                } catch (\Exception) {
                    return new JsonResponse(
                        ['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Firmware version."]],
                        400
                    );
                }

                return new JsonResponse(
                    ['code' => 200, 'message' => 'success', 'errors' => ['errors' => ['']]],
                    200
                );
            }
        }

        return new JsonResponse(
            ['code' => 400, 'message' => 'error', 'errors' => ['errors' => $errors]],
            400
        );
    }

    /**
     * Get All Fw versions
     * @param Request $request
     * @param DeviceOsVersionRepository $deviceOsVersionRepository
     * @return Response
     */
    #[Route(path: '/all', name: 'get_all', methods: ['GET'])]
    public function getAllLabelsAction(Request $request, DeviceOsVersionRepository $deviceOsVersionRepository): Response
    {

        if ($request->isXmlHttpRequest()) {
            $fws = $deviceOsVersionRepository->getAvailableAndApprovedVersions();
            $result = [];

            foreach ($fws as $fw) {
                $result[$fw->getId()] = ['value' => $fw->getVersion()];
            }

            return new JsonResponse($result);
        }

        return new Response('Bad request.', 400);
    }
}
