<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Device\Device;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\MonitoringGroup\MonitoringGroupMapping;
use App\Form\MonitoringGroupType;
use App\Repository\MonitoringGroupRepository;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route(path: '/admin/ajax/monitoring-group', name: 'ajax_monitoring_group_')]
class MonitoringGroupController extends Controller
{

    /**
     * Create new monitoring group
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/create', name: 'create', methods: ['POST'])]
    public function createMonitoringGroupAction(Request $request, ManagerRegistry $managerRegistry): Response {

        $monitoringGroup = new MonitoringGroup();
        $form = $this->createForm(MonitoringGroupType::class, $monitoringGroup);
        $form->handleRequest($request);
        $errors = [];

        if ($form->isSubmitted()) {
            //$validator = $this->get('validator');
            //$errorsValidator = $validator->validate($student);
            $errorsValidator = [];
            foreach ($errorsValidator as $error) {
                array_push($errors, $error->getMessage());
            }

            if (count($errors) == 0) {

                $em = $managerRegistry->getManager();

                try{

                    $em->persist($monitoringGroup);
                    $em->flush();

                }catch (UniqueConstraintViolationException){

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Monitoring Group already exists."]],
                        400);

                }catch (\Exception){

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Monitoring Group."]],
                        400);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Monitoring group successfully created.', 'success' => ''],
                    200);
            }

        }

        return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['errors' => $errors]],
            400);
    }

    /**
     * Get All Monitoring Groups
     * @param Request $request
     * @param MonitoringGroupRepository $monitoringGroupRepository
     * @return Response
     */
    #[Route(path: '/all', name: 'get_all', methods: ['GET'])]
    public function getAllMonitoringGroupsAction(Request $request, MonitoringGroupRepository $monitoringGroupRepository): Response {

        if ($request->isXmlHttpRequest()) {

            $monitoringGroups = $monitoringGroupRepository->getAllMonitoringGroups();
            $result = [];

            foreach ($monitoringGroups as $monitoringGroup) {
                $result[$monitoringGroup->getId()] = ['value' => $monitoringGroup->getName()];
            }

            return new JsonResponse($result);

        }

        return new Response('Bad request.', 400);

    }

    /**
     * Add Devices to the Monitoring Group
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/add-devices', name: 'devices_add', methods: ['POST'])]
    public function addDevicesToMonitoringGroup(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {
            return new Response('Bad Request', 400);
        }

        if(isset($request->request)) {

            // Request has request data ?
            /*if (!isset($request->request))
                return new JsonResponse(array('status' => 'Error'),400);*/

            $deviceIds = (!is_null($request->get('items'))) ? (array) $request->get('items') : null;
            $mntGroupId = (!is_null($request->get('mntGroupId'))) ? (int) $request->get('mntGroupId') : null;

            if (is_null($deviceIds) || is_null($mntGroupId))
                return new JsonResponse(['status' => 'Missing some data'],400);

            $placeholders = [];
            $values = [];
            $types = [];

            foreach ($deviceIds as $deviceId) {

                    $placeholders[] = '(?)';
                    $values[] = [$mntGroupId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;
            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{
                $connection->executeStatement(
                    'INSERT IGNORE `monitoring_group_device_map` (`mongrp_id`, `id_device`)  VALUES ' .
                    implode(', ', $placeholders),
                    $values,
                    $types
                );
            }catch (\Exception $e){
                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse(['code' => 200, 'message' => 'Devices sucessfully inserted to the group.', 'errors' => ['errors' => ['']]],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Error'],400);

    }

    /**
     * Remove Device form Monitoring Group
     * @param Request $request
     * @param int $monitoringGroupId
     * @param int $deviceId
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: 'remove-device/{monitoringGroupId}/{deviceId}', name: 'device_remove', methods: ['GET'])]
    public function deviceMonitoringGroupDelete(Request $request, int $monitoringGroupId, int $deviceId, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {
            return new Response('Bad Request', 400);
        }

        if(isset($request->request)) {

            if (!isset($monitoringGroupId) && !isset($deviceId))
                return new JsonResponse(['status' => 'Error'],400);

            $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroupId]);
            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);
            $monitoringGroupMapping = $managerRegistry->getRepository(MonitoringGroupMapping::class)->findOneBy(
                ['device' => $device, 'monitoringGroup' => $monitoringGroup]);

            if ($monitoringGroupMapping) {

                $em = $managerRegistry->getManager();
                $em->remove($monitoringGroupMapping);
                $em->flush();

            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Device from the Monitoring Group was removed.',
                'errors' => [],
                200]);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Error'],
            400);

    }

    /**
     * Remove devices from Monitoring Group
     * @param Request $request
     * @param int $monitoringGroupId
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/remove-devices/{monitoringGroupId}', name: 'devices_remove', methods: ['POST'])]
    public function removeDevicesFromMonitoringGroup(Request $request, int $monitoringGroupId, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $selectedDevices = $request->get('selectedDevices');
            //$token = $request->request->get('token');

            /*if (!$this->isCsrfTokenValid('multiselect', $token)) {
                throw new AccessDeniedException('The CSRF token is invalid.');
            }*/

            $deviceIds = [];
            $placeholders = [];
            $values = [];
            $types = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            foreach ($deviceIds as $deviceId) {

                    $placeholders[] = '(?)';
                    $values[] = [$monitoringGroupId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{
                $connection->executeStatement(
                    'DELETE FROM `monitoring_group_device_map` WHERE (`mongrp_id`, `id_device`) IN (' .
                    implode(', ', $placeholders) . ')',
                    $values,
                    $types
                );
            }catch (\Exception $e){
                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Devices from the monitoringGroup was removed.',
                'errors' => [],
                200]);

        }

        return new Response('Bad Request', 400);

    }

    /**
     * Delete Monitoring Group
     * @return Response
     */
    #[Route(path: '/delete/{monitoringGroupId}', name: 'delete', methods: ['GET'])]
    public function monitoringGroupDelete(Request $request, int $monitoringGroupId, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroupId]);

            if($monitoringGroup){

                try{

                    $em = $managerRegistry->getManager();
                    $em->remove($monitoringGroup);
                    $em->flush();

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Monitoring group cannot be deleted: ',
                        'errors' => ['Something bad happened.']],
                        200);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Monitoring group "'. $monitoringGroup->getName() .'" was deleted.', 'success' => ''],
                    200);

            }

        }

        return new Response('Bad Request', 400);

    }

}