<?php

declare(strict_types=1);

namespace App\Controller\Admin;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\DataTable\TunnelDatatable;
use App\Entity\Tunnel\TunnelOperation;
use App\Entity\Tunnel\TunnelStatus;
use App\Exception\TunnelException;
use App\Exception\TunnelNotFoundException;
use App\Repository\TunnelWebRepository;
use App\Service\EnvConfigurationService;
use Doctrine\Persistence\ManagerRegistry;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

class TunnelController extends BaseController
{
	/**
	 * @param EnvConfigurationService $envConfigurationService
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @param DatatableFactory $factory
	 * @param DatatableResponse $responseService
	 * @return JsonResponse|Response
	 * @throws \Exception
	 */
    #[Route(path: 'admin/tunnel', name: 'admin_tunnel')]
    public function webTunnel(
        EnvConfigurationService $envConfigurationService,
        Request $request,
        ManagerRegistry $managerRegistry,
        DatatableFactory $factory,
        DatatableResponse $responseService
    ): JsonResponse|Response {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $tunnelStatuses = $managerRegistry->getRepository(TunnelStatus::class)->findAll();
        $tunnelOperations = $managerRegistry->getRepository(TunnelOperation::class)->findAll();
        $isServiceEnabled = $envConfigurationService->isPythonComponentsInstalled();

        $datatable = $factory->create(TunnelDatatable::class);
        $datatable->buildDatatable(['tunnelStatuses' => $tunnelStatuses, 'tunnelOperations' => $tunnelOperations,
            'isPythonComponentsInstalled' => $isServiceEnabled]);

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        return $this->render('tunnel/index.html.twig', [
            'isServiceEnabled' => $isServiceEnabled,
            'datatable' => $datatable,
            'filter' => $this->filter
        ]);
    }

    /**
     * @param Request $request
     * @param int $tunnelId
     * @param TunnelWebRepository $tunnelWebRepository
     * @return RedirectResponse
     * @throws \Exception
     */
    #[Route(path: 'admin/tunnel/connect/{tunnelId}', name: 'admin_tunnel_connect')]
    public function webTunnelDeviceUrl(
        Request $request,
        int $tunnelId,
        TunnelWebRepository $tunnelWebRepository
    ): RedirectResponse {

        $tunnel = $tunnelWebRepository->findOneBy(['id' => $tunnelId]);

        if (!$tunnel) {
            throw new TunnelNotFoundException();
        }

        if (is_null($tunnel->getUrl())) {
            throw new TunnelException('No valid URL defined for tunnelId: ' . $tunnelId);
        }

        $connectUrl = $request->getScheme() . '://' . $tunnel->getUrl() . '/';

        return $this->redirect($connectUrl);
    }
}
