<?php

declare(strict_types=1);

namespace App\Controller\Admin\System;

use App\Entity\EventLog\EventType;
use App\Event\ButtonTypeEventLogEvent;
use App\Form\System\SystemStatusType;
use App\Repository\TunnelWebRepository;
use App\Service\Command;
use App\Service\EnvConfigurationService;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Routing\Attribute\Route;

class StatusController extends Controller
{
    public function __construct(
		private readonly Command $commandService,
		private readonly LoggerInterface $logger,
		private readonly EventDispatcherInterface $eventDispatcher
	)
    {
    }

	/**
	 * @param EnvConfigurationService $envConfigurationService
	 * @param Request $request
	 * @param TunnelWebRepository $tunnelRepository
	 * @return Response
	 * @throws \Doctrine\DBAL\Exception
	 */
    #[Route(path: '/admin/system/status', name: 'admin_system_status')]
    public function default(EnvConfigurationService $envConfigurationService,
							Request $request, TunnelWebRepository $tunnelRepository): Response
    {

		$dgStatus = $this->commandService->getServiceStatus();

		$snmpGatewayStatus = '';
		$isSnmpServiceEnabled = $envConfigurationService->isPythonComponentsInstalled();
		if($isSnmpServiceEnabled){
			$snmpGatewayStatus = $this->commandService->getSnmpGatewayServiceStatus();
		}

        $form = $this->createForm(SystemStatusType::class, [
			'isSnmpServiceEnabled' => $isSnmpServiceEnabled,
            'dgStatus' => $dgStatus,
            'snmpGatewayStatus' => $snmpGatewayStatus
        ]);

        $form->handleRequest($request);

        if ($form->has('dgRestart') && $form->get('dgRestart')->isClicked()) {
            return $this->restartService();
        }

        if ($form->has('dgStart') && $form->get('dgStart')->isClicked()) {
            return $this->startService();
        }

        if ($form->has('snmpGatewayRestart') && $form->get('snmpGatewayRestart')->isClicked()) {
            return $this->startSnmpGatewayService();
        }

        if ($form->has('snmpGatewayStart') && $form->get('snmpGatewayStart')->isClicked()) {
            return $this->startSnmpGatewayService();
        }

        if ($form->has('snmpGatewayStop') && $form->get('snmpGatewayStop')->isClicked()) {
            return $this->stopSnmpGatewayService();
        }

        if ($form->has('tunnelCleanup') && $form->get('tunnelCleanup')->isClicked()) {

            try{
                $tunnelRepository->cleanup();
                return $this->restartService();
            }catch (\Exception $exception){
                $this->addFlash(
                    'danger',
                    'Tunnels cannot be cleaned. Reason: ' . $exception->getMessage()
                );
            }

        }

        return $this->render("system/status.html.twig", [
            'form' => $form->createView(),
			'isSnmpServiceEnabled' => $isSnmpServiceEnabled,
            'dgStatus' => trim($dgStatus),
            'snmpGatewayStatus' => trim($snmpGatewayStatus),
        ]);
    }

    /**
     * @return RedirectResponse
     */
    private function restartService(): RedirectResponse
    {
        try {
            $isRestarted = $this->commandService->restartService();
			$this->eventDispatcher->dispatch(new ButtonTypeEventLogEvent($this->getUser(), EventType::SYSTEM_SERVICE_RESTART_BUTTON_ID, 'Service Restart'), ButtonTypeEventLogEvent::NAME);
        } catch (ProcessFailedException $exception) {
            $this->addFlash(
                'danger',
                'Service restart failed. Reason: ' . $exception->getMessage()
            );

            return $this->redirectToRoute('admin_system_status');
		}

        if (!$isRestarted) {
            $this->addFlash(
                'danger',
                'Service restart failed.'
            );

            return $this->redirectToRoute('admin_system_status');
        }

        $this->addFlash(
            'success',
            'Service was restarted.'
        );

        return $this->redirectToRoute('admin_system_status');
    }

    /**
     * @return RedirectResponse
     */
    public function startService(): RedirectResponse
    {

        try {
            $isStarted = $this->commandService->startService();
			$this->eventDispatcher->dispatch(new ButtonTypeEventLogEvent($this->getUser(), EventType::SYSTEM_SERVICE_START_BUTTON_ID, 'Service Start'), ButtonTypeEventLogEvent::NAME);
        } catch (ProcessFailedException $exception) {
			$this->logger->error($exception->getMessage(), ['StartDgService']);
            $this->addFlash(
                'danger',
                'Service start failed. Reason: ' . $exception->getMessage()
            );

            return $this->redirectToRoute('admin_system_status');
        } /*catch (ExceptionInterface $e) {
			$this->logger->error('Dispatch of Event:'. ButtonTypeEventLogMessageMessage::MESSAGE_NAME . ' with buttonId: ' . EventType::SYSTEM_SERVICE_START_BUTTON_ID . ' failed.');
		}*/

		if (!$isStarted) {
            //throw new ProcessFailedException($process);
            $this->addFlash(
                'danger',
                'Service start failed.'
            );
            return $this->redirectToRoute('admin_system_status');
        }

        $this->addFlash(
            'success',
            'Service was started.'
        );

        return $this->redirectToRoute('admin_system_status');
    }

    private function restartSnmpGatewayService(): RedirectResponse
    {
        try {
            $isRestarted = $this->commandService->restartSnmpGatewayService();
			$this->eventDispatcher->dispatch(new ButtonTypeEventLogEvent($this->getUser(), EventType::SNMP_SERVICE_RESTART_BUTTON_ID, 'SNMP Gateway restart'), ButtonTypeEventLogEvent::NAME);
        } catch (ProcessFailedException $exception) {
            $this->addFlash(
                'danger',
                'SNMP Gateway Service restart failed. Reason: ' . $exception->getMessage()
            );

            return $this->redirectToRoute('admin_system_status');
		} /*catch (ExceptionInterface $e) {
			$this->logger->error('Dispatch of Event:'. ButtonTypeEventLogMessageMessage::MESSAGE_NAME . ' with buttonId: ' . EventType::SNMP_SERVICE_RESTART_BUTTON_ID . ' failed.');
		}*/

        if (!$isRestarted) {
            $this->addFlash(
                'danger',
                'SNMP Gateway Service restart failed.'
            );

            return $this->redirectToRoute('admin_system_status');
        }

        $this->addFlash(
            'success',
            'SNMP Gateway Service was restarted.'
        );

        return $this->redirectToRoute('admin_system_status');
    }
    public function startSnmpGatewayService(): RedirectResponse
    {
        try {
            $isStarted = $this->commandService->startSnmpGatewayService();
			$this->eventDispatcher->dispatch(new ButtonTypeEventLogEvent($this->getUser(), EventType::SNMP_SERVICE_START_BUTTON_ID, 'SNMP Gateway start'), ButtonTypeEventLogEvent::NAME);
        } catch (ProcessFailedException $exception) {
            $this->addFlash(
                'danger',
                'SNMP Gateway Service start failed. Reason: ' . $exception->getMessage()
            );

            return $this->redirectToRoute('admin_system_status');
		} /*catch (ExceptionInterface $e) {
			$this->logger->error('Dispatch of Event:'. ButtonTypeEventLogMessageMessage::MESSAGE_NAME . ' with buttonId: ' . EventType::SNMP_SERVICE_START_BUTTON_ID . ' failed.');
		}*/

        if (!$isStarted) {
            //throw new ProcessFailedException($process);
            $this->addFlash(
                'danger',
                'SNMP Gateway Service start failed.'
            );

            return $this->redirectToRoute('admin_system_status');
        }

        $this->addFlash(
            'success',
            'SNMP Gateway Service was started.'
        );

        return $this->redirectToRoute('admin_system_status');
    }

    public function stopSnmpGatewayService(): RedirectResponse
    {
        try {
            $isStarted = $this->commandService->stopSnmpGatewayService();
			$this->eventDispatcher->dispatch(new ButtonTypeEventLogEvent($this->getUser(), EventType::SNMP_SERVICE_STOP_BUTTON_ID, 'SNMP Gateway stop'), ButtonTypeEventLogEvent::NAME);
        } catch (ProcessFailedException $exception) {
            $this->addFlash(
                'danger',
                'SNMP Gateway Service stop failed. Reason: ' . $exception->getMessage()
            );

            return $this->redirectToRoute('admin_system_status');
		} /*catch (ExceptionInterface $e) {
			$this->logger->error('Dispatch of Event:'. ButtonTypeEventLogMessageMessage::MESSAGE_NAME . ' with buttonId: ' . EventType::SNMP_SERVICE_STOP_BUTTON_ID . ' failed.');
		}*/

        if (!$isStarted) {
            //throw new ProcessFailedException($process);
            $this->addFlash(
                'danger',
                'SNMP Gateway Service stop failed.'
            );

            return $this->redirectToRoute('admin_system_status');
        }

        $this->addFlash(
            'success',
            'SNMP Gateway Service was stopped.'
        );

        return $this->redirectToRoute('admin_system_status');
    }

}
