<?php

namespace App\DataTable\Column;

use Closure;
use Sg\DatatablesBundle\Datatable\Column\AbstractColumn;
use Sg\DatatablesBundle\Datatable\Column\FilterableTrait;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;
use Sg\DatatablesBundle\Datatable\Helper;
use Symfony\Component\OptionsResolver\OptionsResolver;

class CustomLinkColumn extends AbstractColumn
{
    // The LinkColumn is filterable.
    use FilterableTrait;

    /**
     * The route name
     * A required option.
     *
     * @var string
     */
    protected string $route;

    /**
     * The route params.
     *
     * @var array|Closure
     */
    protected array|Closure $routeParams;

    /**
     * The text rendered if data is null.
     *
     * @var string
     */
    protected string $empty_value;

    /**
     * The text displayed for each item in the link.
     *
     * @var Closure|null
     */
    protected ?Closure $text;

    /**
     * The separator for to-many fields.
     *
     * @var string
     */
    protected string $separator;

    /**
     * Function to filter the toMany results.
     *
     * @var Closure|null
     */
    protected ?Closure $filterFunction;

	/**
	 * @var string
	 */
    protected string $link;

    //-------------------------------------------------
    // ColumnInterface
    //-------------------------------------------------

    /**
     * {@inheritdoc}
     */
    public function renderSingleField(array &$row): static
    {
        $path = Helper::getDataPropertyPath($this->data);
        $content = '';

        if ($this->accessor->isReadable($row, $path)) {

			if (\is_callable($this->text)) {
				$text = \call_user_func($this->text, $row);
			} else {
				$text = $this->accessor->getValue($row, $path);
			}

				$content .= '<a href="' . $this->link . '/' . $text .'">';
				$content .= $text;
				$content .= '</a>';


            $this->accessor->setValue($row, $path, $content);
        }

        return $this;
    }

	public function renderToMany(array &$row): static
	{


	}

    /**
     * {@inheritdoc}
     */
    public function getCellContentTemplate(): string
    {
        return '@SgDatatables/render/link.html.twig';
    }

    /**
     * {@inheritdoc}
     */
    public function renderPostCreateDatatableJsContent(): null
	{
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getColumnType(): string
    {
        return parent::ACTION_COLUMN;
    }

    //-------------------------------------------------
    // Options
    //-------------------------------------------------

    /**
     * @return $this
     */
    public function configureOptions(OptionsResolver $resolver): static
    {
        parent::configureOptions($resolver);

        $resolver->setDefaults([
            'filter' => [TextFilter::class, []],
            'link' => '',
            'empty_value' => '',
            'text' => null,
            'separator' => '',
            'filterFunction' => null,
            'email' => false,
        ]);

        $resolver->setAllowedTypes('filter', 'array');
        $resolver->setAllowedTypes('link', 'string');
        $resolver->setAllowedTypes('empty_value', ['string']);
        $resolver->setAllowedTypes('text', ['Closure', 'null']);
        $resolver->setAllowedTypes('separator', ['string']);
        $resolver->setAllowedTypes('filterFunction', ['null', 'Closure']);
        $resolver->setAllowedTypes('email', ['bool']);

        return $this;
    }

    /**
     * @return string
     */
    public function getLink(): string
    {
        return $this->link;
    }

	/**
	 * @param string $link
	 * @return $this
	 */
    public function setLink(string $link): static
    {
        $this->link = $link;

        return $this;
    }

    /**
     * Get empty value.
     *
     * @return array|Closure
     */
    public function getEmptyValue(): array|Closure
    {
        return $this->empty_value;
    }

    /**
     * Set empty value.
     *
     * @param array|Closure $emptyValue
     *
     * @return $this
     */
    public function setEmptyValue(string $emptyValue): static
    {
        $this->empty_value = $emptyValue;

        return $this;
    }

    /**
     * @return Closure|null
     */
    public function getText(): ?Closure
    {
        return $this->text;
    }

    /**
     * @param Closure|null $text
     *
     * @return $this
     */
    public function setText($text): static
    {
        $this->text = $text;

        return $this;
    }

    /**
     * @return string
     */
    public function getSeparator(): string
    {
        return $this->separator;
    }

    /**
     * @param string $separator
     *
     * @return $this
     */
    public function setSeparator(string $separator): static
    {
        $this->separator = $separator;

        return $this;
    }

    /**
     * Get filter function.
     *
     * @return string
     */
    public function getFilterFunction(): string|Closure|null
    {
        return $this->filterFunction;
    }

    /**
     * Set filter function.
     *
     * @param Closure|null $filterFunction
     *
     * @return $this
     */
    public function setFilterFunction(?Closure $filterFunction): static
    {
        $this->filterFunction = $filterFunction;

        return $this;
    }

    /**
     * Get email boolean.
     *
     * @return bool
     */
    public function getEmail(): bool
    {
        return $this->email;
    }

    /**
     * Set email boolean.
     *
     * @param bool $email
     *
     * @return $this
     */
    public function setEmail(bool $email): static
    {
        $this->email = $email;

        return $this;
    }

    //-------------------------------------------------
    // Helper
    //-------------------------------------------------

    /**
     * Render template.
     *
     * @param string|null $data
     *
     * @return string
     */
    private function renderTemplate(?string $data): string
	{
        return $this->twig->render(
            $this->getCellContentTemplate(),
            [
                'data' => $data,
            ]
        );
    }
}
