<?php

namespace App\Controller\Admin\Tools;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Controller\Admin\BaseController;
use App\Datatables\SimCardDatatable;
use App\Entity\Device;
use App\Entity\Tool\SimCard;
use App\Exception\SimCardNotFoundException;
use App\Form\DeleteType;
use App\Form\SimCardType;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Sg\DatatablesBundle\Datatable\DatatableFactory;
use Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route('/admin/tools', name: 'admin_tool_')]
class SimCardManagementController extends BaseController
{

    /**
     * @param Request $request
     * @param DatatableFactory $factory
     * @param DatatableResponse $responseService
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '/sim-cards', name: 'sim_cards_index')]
    public function simCardManagement(
        Request $request,
        DatatableFactory $factory,
        DatatableResponse $responseService,
    ): Response {
        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->query->get('column');

        $devices = $this->managerRegistry->getRepository(Device::class)->getDevices();

        $datatable = $factory->create(SimCardDatatable::class);
        $datatable->buildDatatable(['devices' => $devices]);

        if ($isAjax) {
            $responseService->setDatatable($datatable);

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            return $responseService->getResponse();
        }

        $form = $this->createForm(SimCardType::class, ['devices' => $devices]);

        return $this->render('tool/simCard/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @throws SimCardNotFoundException
     */
    #[Route(path: '/sim-cards/edit/{simCardId}', name: 'sim_cards_edit')]
    public function simCardEdit(
        Request $request,
        int $simCardId,
        DatatableFactory $factory,
        DatatableResponse $responseService,
    ): Response {

        $simCard = $this->managerRegistry->getRepository(SimCard::class)->findOneBy(['id' => $simCardId]);

        if (!$simCard) {
            throw new SimCardNotFoundException();
        }

        $devices = $this->managerRegistry->getRepository(Device::class)->getDevices();
        $simCardData = [
            'simCard' => $simCard,
            'iccid' => $simCard->getIccid(),
            'msisdn' => $simCard->getMsisdn(),
            'label' => $simCard->getLabel(),
            'device' => ($simCard->getDeviceUser() !== null) ? $simCard->getDeviceUser()->getId() : null,
            'devices' => $devices,
        ];

        $form = $this->createForm(SimCardType::class, $simCardData);
        $deleteForm = $this->createForm(DeleteType::class, ['text' => 'Delete Sim Card']);

        $form->handleRequest($request);
        $deleteForm->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $requestData = $request->request->all()['sim_card'];
            $deviceId = (array_key_exists('device', $requestData)) ? $requestData['device'] : null;
            $formData = $form->getData();
            $em = $this->managerRegistry->getManager();

            try {

                $device = $this->managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);

                $simCard->setIccid($formData['iccid']);
                $simCard->setMsisdn($formData['msisdn']);
                $simCard->setLabel($formData['label']);
                $simCard->setDeviceUser($device);

                $em->persist($simCard);
                $em->flush();
            } catch (UniqueConstraintViolationException $e) {
                dump($e->getMessage());
                die();

                $this->addFlash(
                    'danger',
                    'Cannot update: ',
                );
            } catch (\Exception $e) {
                dump($e->getMessage());
                die();
            }

            $this->addFlash(
                'success',
                'Sim Card was updated.',
            );

            return $this->redirectToRoute('admin_tool_sim_cards_index');
        }

        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {

            $em = $this->managerRegistry->getManager();

            try{

                $em->remove($simCard);
                $em->flush();


                $this->addFlash(
                    'success',
                    'Sim Card was successfully deleted.'
                );

            }catch (\Exception){

                $this->addFlash(
                    'danger',
                    'Sim Card was not deleted.'
                );

            }

            return $this->redirectToRoute('admin_tool_sim_cards_index');

        }

        return $this->render('tool/simCard/edit.html.twig', [
            'simCardForm' => $form->createView(),
            'deleteForm' => $deleteForm->createView(),
        ]);
    }

}
