<?php

namespace App\Service;

use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mailer\Transport\Smtp\EsmtpTransport;
use Symfony\Component\Mime\Email;
use Symfony\Component\Notifier\Chatter;
use Symfony\Component\Notifier\Message\ChatMessage;
use Symfony\Component\Notifier\Transport;
use \Symfony\Component\Mailer\Transport as EmailTransport;

class MessageService
{

    /**
     * @var DbService \
     */
    private DbService $dbService;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $dispatcher;

    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @var Chatter
     */
    private Chatter $chatter;

    /**
     * @var Mailer
     */
    private Mailer $mailer;

    /**
     * @var string
     */
    private string $adminEmail;

    /**
     * @var string
     */
    private ?string $senderEmail;

    private ParameterBagInterface $params;

    /**
     * @param DbService $dbService
     * @param LoggerInterface $logger
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(ParameterBagInterface $params, DbService $dbService, LoggerInterface $logger,
                                EventDispatcherInterface $dispatcher){

        $this->params = $params;
        $this->dbService = $dbService;
        $this->dispatcher = $dispatcher;
        $this->logger = $logger;

    }

    /**
     * @param $message
     * @return array
     * @throws \Symfony\Component\Notifier\Exception\TransportExceptionInterface
     */
    public function sendMessage($message): array
    {

        $resultEmail = null;

        try{

            $resultSlack = $this->sendSlackMessage($message);

        }catch (\Exception $exception){

            $resultSlack = false;

        }

        if(!$resultSlack){

            try{

                $resultEmail = $this->sendEmail($message);

            }catch (\Exception $exception){

                $resultEmail = false;

            }

        }

        return [$resultSlack, $resultEmail];

    }

    /**
     * init Slack
     * @throws \Exception
     */
    private function initSlack(): void
    {

        $slackToken = $this->dbService->getSlackToken();
        $slackChannel = $this->dbService->getSlackChannel();

        if(empty($slackToken) || empty($slackChannel)){

            throw new \Exception('No valid slack settings given.');

        }

        $slackDsn = 'slack://' . urlencode($slackToken) . '@default?channel=' . urlencode($slackChannel);

        $transport = Transport::fromDsn($slackDsn);
        $this->chatter = new Chatter($transport);

    }

    /**
     * init Email
     * @throws \Exception
     */
    private function initEmail(): void
    {

        //MAILER_DSN=smtp://user:pass@smtp.example.com:port

        $smtpServer = $this->dbService->getSmtpServer();
        $smtpPort = $this->dbService->getSmtpPort();
        $smtpSecurity = $this->dbService->getSmtpSecurity();
        $smtpLogin = $this->dbService->getSmtpLogin();
        $smtpPassword = $this->dbService->getSmtpPassword();
        $this->adminEmail = $this->dbService->getAdminEmail();
        $this->senderEmail = $this->dbService->getSenderEmail();

        if(empty($smtpServer) || empty($smtpPort) || empty($smtpSecurity) || empty($smtpLogin)
            || empty($smtpPassword) || empty($this->adminEmail) || empty($this->senderEmail)){

            throw new \Exception('No valid email settings given.');

        }

        $tls = null;

        if($smtpSecurity === 0){
            $tls = null;
        }elseif ($smtpSecurity === 1){
            $tls = true;
        }elseif ($smtpSecurity === 2){
            $tls = false;
        }

        /*$transport = EmailTransport::fromDsn(
            'smtp://' . $smtpLogin . ':' . $smtpPassword . '@' . $smtpServer . ':' . $smtpPort
        );*/

        $transport = new EsmtpTransport( ((is_null($smtpServer)) ? '' : $smtpServer), $smtpPort, $tls, $this->dispatcher, $this->logger);

        $transport->setUsername((is_null($smtpLogin)? '' : $smtpLogin) );
        $transport->setPassword((is_null($smtpPassword) ? '' : $smtpPassword) );

        $stream = $transport->getStream();
        $streamOptions = $stream->getStreamOptions();

        $streamOptions['ssl']['verify_peer'] = false;
        //$streamOptions['ssl']['verify_peer_name'] = false;
        //$streamOptions['ssl']['allow_self_signed'] = true;
        //$streamOptions['ssl']['crypto_method'] = \STREAM_CRYPTO_PROTO_TLSv1_3;

        $stream->setStreamOptions($streamOptions);

        $this->mailer = new Mailer($transport);

    }

    /**
     * @param $message
     * @throws \Symfony\Component\Notifier\Exception\TransportExceptionInterface
     */
    public function sendSlackMessage($message): bool
    {

        $this->initSlack();

        $chatMessage = (new ChatMessage($message))
            // if not set explicitly, the message is send to the
            // default transport (the first one configured)
            ->transport('slack');

        try{

            $this->chatter->send($chatMessage);
            $this->logger->info('Slack message sent:' . $message);
            return true;

        }catch (\Exception $e){

            $this->logger->error('Cannot send slack message:' . $message . ', ' . $e->getMessage());
            throw $e;

        }

    }

    /**
     * @param $message
     * @return bool
     * @throws \Exception
     */
    public function sendEmail($message, $testingMessage = false): bool
    {

        //$notifier = new EmailMessage('email');
        /*
        $notification = (new Notification('New Invoice'))
            ->content('You got a new invoice for 15 EUR.')
            ->importance(Notification::IMPORTANCE_HIGH);

        $recipient = new Recipient('wouter@wouterj.nl');

        $email = EmailMessage::fromNotification($notification, $recipient);

        */

        if($testingMessage || $this->dbService->isEnabledEmailNotifications()){

            $this->initEmail();

            $sendTo = $this->getSendToEmailAddress();

            $email = (new Email())
                ->from($this->senderEmail /*'no-reply@mamas.cz'*/)
                ->to($sendTo)
                //->cc('cc@example.com')
                //->bcc('bcc@example.com')
                //->replyTo('fabien@example.com')
                //->priority(Email::PRIORITY_HIGH)
                ->subject($this->params->get('env_dwarf_ui_brand_long') . ' notification')
                ->text($message);
            //->html('<p>See Twig integration for better HTML integration!</p>');

            try{

                $this->mailer->send($email);
                $this->logger->info('Email message sent:' . $message);
                return true;

            }catch (\Exception $e){

                $this->logger->error('Cannot send email message:' . $message . ', ' . $e->getMessage());
                throw $e;

            }

        }else{

            throw new \Exception('Email notifications is not enabled.');

        }

    }

    /**
     * @return string|null
     * @throws \Exception
     */
    private function getSendToEmailAddress(): ?string
    {

        if($this->dbService->isEnabledEmailNotificationToAddress()){

            $toEmail = $this->dbService->getNotificationEmail();

            if(!$toEmail){

                throw new \Exception('No valid email to send notification given.');

            }

            return $toEmail;

        }else{

            return $this->dbService->getAdminEmail();

        }

    }

}