<?php

declare(strict_types=1);

namespace App\Controller\Admin\Datatable;

use App\Entity\Device\Device;
use App\Entity\Device\VDevice;
use App\Entity\Label\Label;
use App\Entity\Label\SystemLabel;
use App\Entity\MonitoringGroup\MonitoringGroup;
use DateTime;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\MappingException;
use Doctrine\Persistence\ManagerRegistry;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Component\PropertyAccess\PropertyAccessor;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

class DatatableController extends AbstractController
{

    public function __construct(private readonly ManagerRegistry $managerRegistry,
								private readonly EntityManagerInterface $entityManager
	)
    {
    }

	/**
	 * Edit field.
	 *
	 *
	 * @param Request $request
	 * @return Response
	 * @throws Exception
	 */
    #[Route(path: '/datatables/edit/field', name: 'app_datatables_edit', methods: ['POST'])]
    public function editAction(Request $request): Response
    {

        if ($request->isXmlHttpRequest()) {
            // x-editable sends some default parameters
            $pk = (int) $request->request->get('pk');       // entity primary key
            $field = $request->request->get('name');  // e.g. comments.createdBy.username
            $value = $request->request->get('value'); // the new value

            // additional params
            $entityClassName = $request->request->get('entityClassName'); // e.g. AppBundle\Entity\Post
            $token = $request->request->get('token');
            $originalTypeOfField = $request->request->get('originalTypeOfField');
            $path = $request->request->get('path'); // for toMany - the current element

            // check token
            if (! $this->isCsrfTokenValid('sg-datatables-editable', $token)) {
                throw new AccessDeniedException('DatatableController::editAction(): The CSRF token is invalid.');
            }

            //Ugly hack to use devices table instead of view which cannot be updated, use RealDevice entity
            if($entityClassName === VDevice::class){
                $entityClassName = Device::class;
            }

			//Check if it is editable through entity annotations options: ['editable' => true]
			$isAllowed = $this->checkEntityMetadataRights($entityClassName, $field);

			if(!$isAllowed){
				return new Response('Not allowed.', 405);
			}

			// get an object by its primary key
			$entity = $this->getEntityByPk($entityClassName, $pk);

			/** @var PropertyAccessor $accessor */
            $accessor = PropertyAccess::createPropertyAccessorBuilder()
                ->enableMagicCall()
                ->getPropertyAccessor()
            ;

            // normalize the new value
            $value = $this->normalizeValue($originalTypeOfField, $value);

            // set new value
            null !== $path ? $accessor->setValue($entity, $path, $value) : $accessor->setValue($entity, $field, $value);

            // save all
            try{

                $em = $this->managerRegistry->getManager();
                $em->persist($entity);
                $em->flush();

            }catch (UniqueConstraintViolationException){

                $entityName = 'Entity';
                
                if($entityClassName === Label::class) {
                    $entityName = 'Label';
                }elseif($entityClassName === SystemLabel::class){
                    $entityName = 'System Label';
                }elseif($entityClassName === MonitoringGroup::class){
                    $entityName = 'Monitoring Group';
                }

                return new Response($entityName . ' already exists.', 400);

            }catch (\Exception){

                return new Response('Bad request', 400);

            }

            return new Response('Success', 200);
        }

        return new Response('Bad request', 400);
    }

    //-------------------------------------------------
    // Helper
    //-------------------------------------------------

    /**
     * Finds an object by its primary key / identifier.
     *
     * @param string $entityClassName
     * @param int $pk
     * @return object
     */
    private function getEntityByPk(string $entityClassName, int $pk): object
    {
        $em = $this->managerRegistry->getManager();

        $entity = $em->getRepository($entityClassName)->find($pk);
        if (! $entity) {
            throw $this->createNotFoundException('DatatableController::getEntityByPk(): The entity does not exist.');
        }

        return $entity;
    }

	/**
	 * @param string $originalTypeOfField
	 * @param mixed $value
	 * @return mixed
	 * @throws Exception
	 */
    private function normalizeValue(string $originalTypeOfField, mixed $value): mixed
    {
        switch ($originalTypeOfField) {
            case Types::DATETIME_MUTABLE:
                $value = new DateTime($value);

                break;
            case Types::BOOLEAN:
                $value = $this->strToBool($value);

                break;
            case Types::TEXT:
            case Types::STRING:
                break;
            case Types::SMALLINT:
            case Types::INTEGER:
                $value = (int) $value;

                break;
            case Types::BIGINT:
                $value = (string) $value;

                break;
            case Types::FLOAT:
            case Types::DECIMAL:
                $value = (float) $value;

                break;
            default:
                throw new \RuntimeException("DatatableController::prepareValue(): The field type {$originalTypeOfField} is not editable.");
        }

        return $value;
    }

	/**
	 * @param string $str
	 * @return bool|null
	 */
    private function strToBool(string $str): ?bool
    {
        $str = strtolower($str);

        if ('null' === $str) {
            return null;
        }

        if ('true' === $str || '1' === $str) {
            return true;
        }

        if ('false' === $str || '0' === $str) {
            return false;
        }

        throw new \RuntimeException('DatatableController::strToBool(): Cannot convert string to boolean, expected string "true" or "false".');
    }

	/**
	 * @param string $entityClassName
	 * @param string $field
	 * @return bool
	 * @throws MappingException
	 */
	private function checkEntityMetadataRights(string $entityClassName, string $field): bool
	{

		// Get all metadata for the entity
		$metadata = $this->entityManager->getClassMetadata($entityClassName);

		// Doctrine stores the ReflectionProperty for every mapped field
		if ($metadata->getPropertyAccessor($field)) {

			$mapping = $metadata->getFieldMapping($field);
			// Access the options array
			$isEditable = $mapping['options']['datatable_editable'] ?? false;

			if($isEditable){
				return true;
			}
		}

		return false;

	}
}
